/* ==================================================================
 *	DAEMON.C
 *
 *	This file provides the 'daemon' implementation of
 *	xtell.
 *
 *
 *	Daemon-
 *		Regular daemon which waits for connections
 *		forks, handles connections, cleans up,
 *		dies, etc.
 *
 * =============================================================== */

#include "xtelld.h"

/* the entry point is daemon_service (last function, at the bottom)	*/

unsigned long	open_connections;


/* ------------------------------------
 *	child_reaper:
 *		free resources of child service
 *		as well as close the descriptor
 *		on the parent end
 * ------------------------------------
 */
static void child_reaper (int s)
{
	int	status;
	
	wait (&status);
	if (WIFEXITED(status))	/* exit value is a socket descriptor */
		killsock (WEXITSTATUS(status));
	
	signal (SIGCHLD, child_reaper);
	
	if (open_connections > 0)
		open_connections--;
	
}




/* ------------------------------------------------------------------
 * daemon_core:
 *	this maintains a standalone -forking- daemon
 *	it basically waits for a connection, forks off
 *	a handler for it, and cleans up when it dies.
 *	results in much smaller code, but also a wee bit more
 *	overhead.
 * ------------------------------------------------------------------
 */
static void daemon_core (int sd)
{
	struct	sockaddr_in	sin;
	int	sinsize, sd_peal;
	
	sinsize = sizeof (struct sockaddr);
	
	signal (SIGCHLD, child_reaper);
	
	while (1) {

		sd_peal = accept (sd, (struct sockaddr *)&sin, &sinsize);
		if (sd_peal == -1) {
			if (errno == EINTR)
				continue;
			else
				break;
		}
		else if (open_connections < max_connections) {
			switch (fork ()) {
				case 0:
					alarm (client_timeout);
					signal (SIGALRM, xtell_killtic);
					
						/* we trick the inetd service handler into
						 * thinking it was spawned by inetd, rather
						 * than this process. harmless prank. */	
					
					inetd_service (sd_peal, sd_peal);

					killsock (sd_peal);
					exit (sd_peal);
				
				case -1:
                                        fprintf(stderr, "daemon_service: could not fork!!\n");
                                        syslog(LOG_ERR, "daemon_service: could not fork!!");
					break;
				
				default:
					open_connections++;
					break;
			}
		}
		else
			killsock (sd_peal);
	
	}
	fprintf(stderr, "accept() returned error: %s\n", strerror (errno));
	syslog(LOG_ERR, "accept() returned error: %s", strerror (errno));
}





/* ------------------------------------------------------------------
 * daemon_service	:
 * ------------------------------------------------------------------
 */
void daemon_service (void)
{
	int	sd, err;
	struct sockaddr_in	sin;
	
	open_connections = 0;
	
	openlog ("xtell", LOG_PID, LOG_AUTH);
	
	setsid ();
        
	if (fork ())		/* daemonize */
		exit (0);
	chdir ("/"); 

	sd = socket (PF_INET, SOCK_STREAM, 0);
	if (sd == -1) {
		syslog(LOG_ERR, "Cannot allocate socket: %s", strerror (errno));
		fprintf(stderr, "Cannot allocate socket: %s\n", strerror (errno));
		exit (-1);
	}
	
	sin.sin_family = AF_INET;
	sin.sin_port = htons (xtell_port);
	sin.sin_addr.s_addr = INADDR_ANY;
	err = bind (sd, (struct sockaddr *)&sin, sizeof (sin));
	if (err == -1) {
		syslog(LOG_ERR, "Cannot bind to port %d: %s", 
			xtell_port, strerror (errno));
		fprintf(stderr, "Cannot bind to port %d: %s\n", 
			xtell_port, strerror (errno));
		exit (-1);
	}
	
	if (-1 == listen (sd, 5)) {
		syslog(LOG_ERR, "Cannot listen on port %d: %s", 
			xtell_port, strerror (errno));
		fprintf(stderr, "Cannot listen on port %d: %s\n", 
			xtell_port, strerror (errno));
		exit (-1);
	}

	syslog (LOG_NOTICE, "xtell services started [ %s ]", 
		VERSION);
	daemon_core (sd);
	syslog(LOG_ERR, "xtell services terminated (exiting on internal error)");
	fprintf(stderr, "xtell services terminated (exiting on internal error)\n");
	closelog ();
	killsock (sd);

}
