(async () => {

  // --- Define listeners for background notifications from legacy code

  /**
   * NotifyTools currently is not 100% compatible with the behavior of
   * runtime.sendMessage. While runtime messaging is ignoring non-Promise return
   * values, NotifyTools only ignores <null>.
   * 
   * Why does this matter? Consider the following three listeners:
   * 
   * async function dominant_listener(data) {
   *  if (data.type == "A") {
   *    return { msg: "I should answer only type A" };
   *  }
   * }
   * 
   * function silent_listener(data) {
   *  if (data.type == "B") {
   *    return { msg: "I should answer only type B" };
   *  }
   * }
   * 
   * function selective_listener(data) {
   *  if (data.type == "C") {
   *    return Promise.resolve({ msg: "I should answer only type C" });
   *  }
   * }
   * 
   * When all 3 listeners are registered for the runtime.onMessage event,
   * the dominant listener will always respond, even for data.type != "A" requests,
   * because it is always returning a Promise (async function). The return value of 
   * the silent listener is ignored, and the selective listener returns a value just
   * for data.type == "C". But since the dominant listener also returns <null> for
   * these requests, the actual return value depends on which listener is faster
   * and/or was registered first.
   * 
   * All notifyTools listener however ignore <null> return values (so null can actually
   * never be returned). The above dominant listener will only respond to type == "A"
   * requests, the silent listener will only respond to type == "B" requests and the
   * selective listener will respond only to type == "C" requests.
   * 
   */

  // This listener returns a selective Promise and is therefore compatible with
  // runtime messaging (messenger.storage.local.set/get return Promises).
  function listener(data) {
    switch (data.command) {
      case "getPref":
        return messenger.storage.local.get({ time: "0" });
      case "setPref":
        return messenger.storage.local.set({ time: data.time });
    }
  }

  messenger.NotifyTools.onNotifyBackground.addListener(listener);

  // --- Wait until onInstalled has run (just for fun)

  await new Promise(resolve => {
    const listener = async (details) => {
      // Do stuff on installed.
      console.log("Start", details.reason);
      messenger.storage.local.set({ "index": 0 });
      await new Promise(res => setTimeout(res, 3000));

      console.log("Ready", details.reason);
      // Remove since we do not need it anymore.
      messenger.runtime.onInstalled.removeListener(listener);
      resolve();
    };
    messenger.runtime.onInstalled.addListener(listener);
  });


  // --- Test if NotifyTools can be used in WebExt events

  async function sendAndReceive(msg) {
    let rv = await messenger.NotifyTools.notifyExperiment({ msg });
    console.log(msg, rv.msg == `${msg}: ${msg}` ? "OK" : "ERR", rv);
  }

  messenger.messageDisplay.onMessageDisplayed.addListener(async (tab, message) => {
    await sendAndReceive("onMessageDisplayed");
  });

  messenger.compose.onBeforeSend.addListener(async (tab, details) => {
    await new Promise(resolve => window.setTimeout(resolve, 3000));
    await sendAndReceive("onBeforeSend");
    return { cancel: true };
  });

  // --- Start up legacy part

  messenger.WindowListener.registerWindow(
    "chrome://messenger/content/messenger.xhtml",
    "messenger.js");

  messenger.WindowListener.startListening();
})();
