/******************************************************************************
*   This file is part of TinTin++                                             *
*                                                                             *
*   Copyright 2004-2020 Igor van den Hoven                                    *
*                                                                             *
*   TinTin++ is free software; you can redistribute it and/or modify          *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 3 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with TinTin++.  If not, see https://www.gnu.org/licenses.           *
******************************************************************************/

/******************************************************************************
*                               T I N T I N + +                               *
*                                                                             *
*                      coded by Igor van den Hoven 2006                       *
******************************************************************************/


#include "tintin.h"

#ifdef HAVE_SYS_IOCTL_H
  #include <sys/ioctl.h>
#endif
#include <termios.h>

void init_terminal(struct session *ses)
{
	struct termios io;

	if (HAS_BIT(gtd->flags, TINTIN_FLAG_NOHUP))
	{
		return;
	}

	if (tcgetattr(0, &gtd->old_terminal))
	{
		syserr_fatal(-1, "init_terminal: tcgetattr 1");
	}

	io = gtd->old_terminal;

	/*
		Canonical mode off
	*/

	DEL_BIT(io.c_lflag, ICANON);

	io.c_cc[VMIN]   = 1;
	io.c_cc[VTIME]  = 0;
	io.c_cc[VSTART] = 255;
	io.c_cc[VSTOP]  = 255;
	io.c_cc[VINTR]  = 4; // ctrl-d

	/*
		Make the terminalal as raw as possible
	*/

/*
	DEL_BIT(io.c_iflag, IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
	DEL_BIT(io.c_oflag, OPOST);
	DEL_BIT(io.c_cflag, CSIZE|PARENB);
*/


	DEL_BIT(io.c_lflag, ECHO|ECHONL|IEXTEN|ISIG);
//	DEL_BIT(io.c_lflag, ECHO|ECHONL|IEXTEN|ISIG);

	SET_BIT(io.c_cflag, CS8);

	if (tcsetattr(0, TCSANOW, &io))
	{
		syserr_printf(ses, "init_terminal: tcsetattr");
	}

	if (tcgetattr(0, &gts->cur_terminal))
	{
		syserr_fatal(-1, "init_terminal: tcgetattr 2");
	}

	print_stdout(0, 0, "\e[?1004h\e=\e[>4;1m");
}

void reset_terminal(struct session *ses)
{
	if (HAS_BIT(gtd->flags, TINTIN_FLAG_NOHUP))
	{
		return;
	}
	
	if (gtd->detach_port == 0)
	{
		if (tcsetattr(0, TCSANOW, &gtd->old_terminal))
		{
			syserr_printf(ses, "reset_terminal: tcsetattr");
		}
	}

	if (HAS_BIT(gtd->flags, TINTIN_FLAG_MOUSETRACKING))
	{
		print_stdout(0, 0, "\e[?1000l\e[?1002l\e[?1006l");
	}
	print_stdout(0, 0, "\e[?25h\e[23t\e[?1004l\e[>4n\e[>4;0m\e[?47l\e[r\e[0#t");
}


void save_session_terminal(struct session *ses)
{
	tcgetattr(0, &ses->cur_terminal);
}

void refresh_session_terminal(struct session *ses)
{
//	tcsetattr(0, TCSANOW, &ses->cur_terminal);
}

void echo_off(struct session *ses)
{
	struct termios io;

	tcgetattr(STDIN_FILENO, &io);

	DEL_BIT(io.c_lflag, ECHO|ECHONL);

	tcsetattr(STDIN_FILENO, TCSADRAIN, &io);
}

void echo_on(struct session *ses)
{
	struct termios io;

	tcgetattr(STDIN_FILENO, &io);

	SET_BIT(io.c_lflag, ECHO|ECHONL);

	tcsetattr(STDIN_FILENO, TCSADRAIN, &io);
}

void init_terminal_size(struct session *ses)
{
	struct winsize screen;
	static int old_rows, old_cols;

	push_call("init_terminal_size(%p)",ses);

	if (ses == gts)
	{
		old_rows = gtd->screen->rows;
		old_cols = gtd->screen->cols;

		if (ioctl(1, TIOCGWINSZ, &screen) >= 0)
		{
			init_screen(screen.ws_row, screen.ws_col, screen.ws_ypixel, screen.ws_xpixel);

			if (gtd->attach_sock)
			{
				char buf[100];
				sprintf(buf, "\e[8;%d;%dt\e[4;%d;%dt\e[7t", screen.ws_row, screen.ws_col, screen.ws_ypixel, screen.ws_xpixel);

				if (write(gtd->attach_sock, buf, strlen(buf)) == -1)
				{
					printf("error: init_terminal_size: write:\n");
				}
			}
		}
		else
		{
			init_screen(SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT * 16, SCREEN_WIDTH * 10);
		}
	}

	if (ses->scroll)
	{
		SET_BIT(ses->scroll->flags, SCROLL_FLAG_RESIZE);
	}

	if (ses->map)
	{
		SET_BIT(ses->map->flags, MAP_FLAG_RESIZE);
	}

	init_split(ses, ses->split->sav_top_row, ses->split->sav_top_col, ses->split->sav_bot_row, ses->split->sav_bot_col);

	check_all_events(ses, EVENT_FLAG_SCREEN, 0, 4, "SCREEN RESIZE", ntos(gtd->screen->rows), ntos(gtd->screen->cols), ntos(gtd->screen->height), ntos(gtd->screen->width));

	if (old_rows <= old_cols / 2 && gtd->screen->rows > gtd->screen->cols / 2)
	{
		check_all_events(ses, EVENT_FLAG_SCREEN, 0, 4, "SCREEN ROTATE PORTRAIT", ntos(gtd->screen->rows), ntos(gtd->screen->cols), ntos(gtd->screen->height), ntos(gtd->screen->width));
	}
	else if (old_rows >= old_cols / 2 && gtd->screen->rows < gtd->screen->cols / 2)
	{
		check_all_events(ses, EVENT_FLAG_SCREEN, 0, 4, "SCREEN ROTATE LANDSCAPE", ntos(gtd->screen->rows), ntos(gtd->screen->cols), ntos(gtd->screen->height), ntos(gtd->screen->width));
	}

	msdp_update_all("SCREEN_ROWS",   "%d", gtd->screen->rows);
	msdp_update_all("SCREEN_COLS",   "%d", gtd->screen->cols);
	msdp_update_all("SCREEN_HEIGHT", "%d", gtd->screen->height);
	msdp_update_all("SCREEN_WIDTH",  "%d", gtd->screen->width);

	pop_call();
	return;
}

int get_scroll_rows(struct session *ses)
{
	return (ses->split->bot_row - ses->split->top_row);
}

int get_scroll_cols(struct session *ses)
{
	return ses->wrap;
}

char *get_charset(struct session *ses)
{
	int index;

	for (index = 0 ; *charset_table[index].name ; index++)
	{
		if (ses->charset == charset_table[index].flags)
		{
			return charset_table[index].name;
		}
	}
	return "ASCII";
}
