/**
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isString = require( '@stdlib/assert/is-string' ).isPrimitive;
var hasOwnProp = require( '@stdlib/assert/has-own-property' );
var PKG_TO_STANDALONE = require( './../data/data.json' );


// MAIN //

/**
* Returns the standalone package name associated with a provided internal package name.
*
* @param {string} pkg - package name
* @throws {TypeError} must provide a string
* @returns {(string|null)} standalone package name
*
* @example
* var v = pkg2standalone( '@stdlib/math/base/special/sin' );
* // returns '@stdlib/math-base-special-sin'
*/
function pkg2standalone( pkg ) {
	if ( !isString( pkg ) ) {
		throw new TypeError( 'invalid argument. Must provide a string. Value: `' + pkg + '`.' );
	}
	if ( hasOwnProp( PKG_TO_STANDALONE, pkg ) ) {
		return PKG_TO_STANDALONE[ pkg ];
	}
	return null;
}


// EXPORTS //

module.exports = pkg2standalone;
