use strict;
use warnings;

use Test::More tests => 36;

use Crypt::Mac::HMAC 'hmac_hex';

# test vectors from https://csrc.nist.gov/projects/cryptographic-standards-and-guidelines/example-values#aMsgAuth

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA3-224.pdf
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b");
  is( hmac_hex('SHA3_224', $key, $input), '332cfd59347fdb8e576e77260be4aba2d6dc53117b3bfb52c6d18c04', 'SHA3_224');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f");
  is( hmac_hex('SHA3_224', $key, $input), 'd8b733bcf66c644a12323d564e24dcf3fc75f231f3b67968359100c7', 'SHA3_224');
}
{
  my $input = "Sample message for keylen>blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaab");
  is( hmac_hex('SHA3_224', $key, $input), '078695eecc227c636ad31d063a15dd05a7e819a66ec6d8de1e193e59', 'SHA3_224');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b");
  is( substr(hmac_hex('SHA3_224', $key, $input), 0, 28), '8569c54cbb00a9b78ff1b391b0e5', 'SHA3_224');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA3-256.pdf
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f");
  is( hmac_hex('SHA3_256', $key, $input), '4fe8e202c4f058e8dddc23d8c34e467343e23555e24fc2f025d598f558f67205', 'SHA3_256');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f8081828384858687");
  is( hmac_hex('SHA3_256', $key, $input), '68b94e2e538a9be4103bebb5aa016d47961d4d1aa906061313b557f8af2c3faa', 'SHA3_256');
}
{
  my $input = "Sample message for keylen>blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7");
  is( hmac_hex('SHA3_256', $key, $input), '9bcf2c238e235c3ce88404e813bd2f3a97185ac6f238c63d6229a00b07974258', 'SHA3_256');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f");
  is( substr(hmac_hex('SHA3_256', $key, $input), 0, 32), 'c8dc7148d8c1423aa549105dafdf9cad', 'SHA3_256');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA3-384.pdf
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f");
  is( hmac_hex('SHA3_384', $key, $input), 'd588a3c51f3f2d906e8298c1199aa8ff6296218127f6b38a90b6afe2c5617725bc99987f79b22a557b6520db710b7f42', 'SHA3_384');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f6061626364656667");
  is( hmac_hex('SHA3_384', $key, $input), 'a27d24b592e8c8cbf6d4ce6fc5bf62d8fc98bf2d486640d9eb8099e24047837f5f3bffbe92dcce90b4ed5b1e7e44fa90', 'SHA3_384');
}
{
  my $input = "Sample message for keylen>blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f9091929394959697");
  is( hmac_hex('SHA3_384', $key, $input), 'e5ae4c739f455279368ebf36d4f5354c95aa184c899d3870e460ebc288ef1f9470053f73f7c6da2a71bcaec38ce7d6ac', 'SHA3_384');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f");
  is( substr(hmac_hex('SHA3_384', $key, $input), 0, 48), '25f4bf53606e91af79d24a4bb1fd6aecd44414a30c8ebb0a', 'SHA3_384');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA3-512.pdf
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f");
  is( hmac_hex('SHA3_512', $key, $input), '4efd629d6c71bf86162658f29943b1c308ce27cdfa6db0d9c3ce81763f9cbce5f7ebe9868031db1a8f8eb7b6b95e5c5e3f657a8996c86a2f6527e307f0213196', 'SHA3_512');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f4041424344454647");
  is( hmac_hex('SHA3_512', $key, $input), '544e257ea2a3e5ea19a590e6a24b724ce6327757723fe2751b75bf007d80f6b360744bf1b7a88ea585f9765b47911976d3191cf83c039f5ffab0d29cc9d9b6da', 'SHA3_512');
}
{
  my $input = "Sample message for keylen>blocklen";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f8081828384858687");
  is( hmac_hex('SHA3_512', $key, $input), '5f464f5e5b7848e3885e49b2c385f0694985d0e38966242dc4a5fe3fea4b37d46b65ceced5dcf59438dd840bab22269f0ba7febdb9fcf74602a35666b2a32915', 'SHA3_512');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f");
  is( substr(hmac_hex('SHA3_512', $key, $input), 0, 64), '7bb06d859257b25ce73ca700df34c5cbef5c898bac91029e0b27975d4e526a08', 'SHA3_512');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA1.pdf
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F");
  is( hmac_hex('SHA1', $key, $input), '5fd596ee78d5553c8ff4e72d266dfd192366da29', 'SHA1');
}
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F10111213");
  is( hmac_hex('SHA1', $key, $input), '4c99ff0cb1b31bd33f8431dbaf4d17fcd356a807', 'SHA1');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F60616263");
  is( hmac_hex('SHA1', $key, $input), '2d51b2f7750e410584662e38f133435f4c4fd42a', 'SHA1');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F30");
  is( substr(hmac_hex('SHA1', $key, $input), 0, 24), 'fe3529565cd8e28c5fa79eac', 'SHA1');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA224.pdf
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F");
  is( hmac_hex('SHA224', $key, $input), 'c7405e3ae058e8cd30b08b4140248581ed174cb34e1224bcc1efc81b', 'SHA224');
}
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B");
  is( hmac_hex('SHA224', $key, $input), 'e3d249a8cfb67ef8b7a169e9a0a599714a2cecba65999a51beb8fbbe', 'SHA224');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F60616263");
  is( hmac_hex('SHA224', $key, $input), '91c52509e5af8531601ae6230099d90bef88aaefb961f4080abc014d', 'SHA224');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F30");
  is( substr(hmac_hex('SHA224', $key, $input), 0, 32), 'd522f1df596ca4b4b1c23d27bde067d6', 'SHA224');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA256.pdf
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F");
  is( hmac_hex('SHA256', $key, $input), '8bb9a1db9806f20df7f77b82138c7914d174d59e13dc4d0169c9057b133e1d62', 'SHA256');
}
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F");
  is( hmac_hex('SHA256', $key, $input), 'a28cf43130ee696a98f14a37678b56bcfcbdd9e5cf69717fecf5480f0ebdf790', 'SHA256');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F60616263");
  is( hmac_hex('SHA256', $key, $input), 'bdccb6c72ddeadb500ae768386cb38cc41c63dbb0878ddb9c7a38a431b78378d', 'SHA256');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F30");
  is( substr(hmac_hex('SHA256', $key, $input), 0, 32), '27a8b157839efeac98df070b331d5936', 'SHA256');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA384.pdf
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F");
  is( hmac_hex('SHA384', $key, $input), '63c5daa5e651847ca897c95814ab830bededc7d25e83eef9195cd45857a37f448947858f5af50cc2b1b730ddf29671a9', 'SHA384');
}
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F");
  is( hmac_hex('SHA384', $key, $input), '6eb242bdbb582ca17bebfa481b1e23211464d2b7f8c20b9ff2201637b93646af5ae9ac316e98db45d9cae773675eeed0', 'SHA384');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7");
  is( substr(hmac_hex('SHA384', $key, $input), 0, 48), '5b664436df69b0ca22551231a3f0a3d5b4f97991713cfa84', 'SHA384');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F30");
  is( substr(hmac_hex('SHA384', $key, $input), 0, 48), 'c48130d3df703dd7cdaa56800dfbd2ba2458320e6e1f98fe', 'SHA384');
}

# https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/HMAC_SHA512.pdf
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F");
  is( hmac_hex('SHA512', $key, $input), 'fc25e240658ca785b7a811a8d3f7b4ca48cfa26a8a366bf2cd1f836b05fcb024bd36853081811d6cea4216ebad79da1cfcb95ea4586b8a0ce356596a55fb1347', 'SHA512');
}
{
  my $input = "Sample message for keylen<blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F");
  is( hmac_hex('SHA512', $key, $input), 'fd44c18bda0bb0a6ce0e82b031bf2818f6539bd56ec00bdc10a8a2d730b3634de2545d639b0f2cf710d0692c72a1896f1f211c2b922d1a96c392e07e7ea9fedc', 'SHA512');
}
{
  my $input = "Sample message for keylen=blocklen";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7");
  is( hmac_hex('SHA512', $key, $input), 'd93ec8d2de1ad2a9957cb9b83f14e76ad6b5e0cce285079a127d3b14bccb7aa7286d4ac0d4ce64215f2bc9e6870b33d97438be4aaa20cda5c5a912b48b8e27f3', 'SHA512');
}
{
  my $input = "Sample message for keylen<blocklen, with truncated tag";
  my $key = pack("H*", "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F30");
  is( substr(hmac_hex('SHA512', $key, $input), 0, 64), '00f3e9a77bb0f06de15f160603e42b5028758808596664c03e1ab8fb2b076778', 'SHA512');
}
