/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggwidgetkeyslider.h"


#include "ierror.h"

#include "iggwidgetrendermodebutton.h"

#include "ibgwidgetentrysubject.h"

#include "iggsubjectfactory.h"


using namespace iParameter;


//
//  Template slider
//
template<class T>
iggWidgetKeySlider<T>::iggWidgetKeySlider(int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyHandler<T>(WidgetType::Entry,key,rm,parent,index,indkey)
{
	this->mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,true,numdig,label);
	new iggWidgetSlider(numdig,label,mSubject,this);
	this->mSubject->SetStretch(0,10,0);

	if(rm != RenderMode::NoRender)
	{
		iggWidgetRenderModeButton *rm = new iggWidgetRenderModeButton(this); IERROR_ASSERT(rm);
		rm->ClearLaidOutFlag(this->mSubject);
		this->mSubject->AddButton(rm->GetSubject());
	}

	this->SetRenderMode(rm);

	this->mHighResolutionValue = false;
}


template<class T>
void iggWidgetKeySlider<T>::SetRenderMode(int m)
{
	iggWidgetKeyHandler<T>::SetRenderMode(m);
	this->mSubject->SetDelayedEdit(this->GetRenderMode()==RenderMode::Delayed);
}


template<class T>
void iggWidgetKeySlider<T>::QueryValue(T &v) const
{
	double dv;
	bool ok = this->mHighResolutionValue;

	if(ok)
	{
		dv = mSubject->GetText().ToDouble(ok); // if failed, use low res value
	}

	if(ok)
	{
		v = this->GetInternalValue((T)dv);
	}
	else
	{
		this->ConvertFromSliderInt(this->mSubject->GetValue(),v); 
	}
}


template<class T>
void iggWidgetKeySlider<T>::UpdateValue(T v)
{
	int sint;

	this->ConvertToSliderInt(v,sint);
	this->mSubject->SetValue(sint,double(this->GetDisplayedValue(v)));
}


template<class T>
void iggWidgetKeySlider<T>::OnInt1Body(int)
{
	T val;
	this->ConvertFromSliderInt(this->mSubject->GetValue(),val);
	this->mSubject->SetValue(this->mSubject->GetValue(),double(this->GetDisplayedValue(val)));
	this->mHighResolutionValue = false;
	this->ExecuteControl(false);
}


template<class T>
void iggWidgetKeySlider<T>::OnVoid1Body()
{
	this->ExecuteControl(true);
#ifdef I_CHECK2
	//
	//  Check that GetDisplayedValue and GetInternalValue are consistent
	//
	int smin, smax;
	T vmin, vmax, dvmin, dvmax, ivmin, ivmax;

	this->mSubject->GetRange(smin,smax);
	this->ConvertFromSliderInt(smin,vmin); 
	this->ConvertFromSliderInt(smax,vmax); 

	dvmin = this->GetDisplayedValue(vmin);
	ivmin = this->GetInternalValue(dvmin);
	dvmax = this->GetDisplayedValue(vmax);
	ivmax = this->GetInternalValue(dvmax);
	if(vmin!=ivmin || vmax!=ivmax)
	{
		IERROR_HIGH("Bug: GetDisplayedValue and GetInternalValue are NOT consistent.");
		dvmin = this->GetDisplayedValue(vmin);
		ivmin = this->GetInternalValue(dvmin);
		dvmax = this->GetDisplayedValue(vmax);
		ivmax = this->GetInternalValue(dvmax);
		IERROR_FATAL("Bug: GetDisplayedValue and GetInternalValue are NOT consistent.");
	}
#endif
}


template<class T>
void iggWidgetKeySlider<T>::OnString1Body(const iString &s)
{
	int sint;
	bool ok;
	double dv = s.ToDouble(ok);

	if(!ok) return; // in the process of editing

	T val = this->GetInternalValue((T)dv);

	this->ConvertToSliderInt(val,sint);
	this->mSubject->SetValue(sint);
	this->mSubject->SetDelayedEdit(this->GetRenderMode()==RenderMode::Delayed);
	this->mHighResolutionValue = true;
	//
	//  disable self-cheking temporarily
	//
	bool sc = this->mSelfChecking;
	this->mSelfChecking = false;
	this->ExecuteControl(false,val);
	this->mSelfChecking = sc;
}


template<class T>
T iggWidgetKeySlider<T>::GetDisplayedValue(T val) const
{
	return val;
}


template<class T>
T iggWidgetKeySlider<T>::GetInternalValue(T dv) const
{
	return dv;
}


template<class T>
void iggWidgetKeySlider<T>::SetRange(int smin, int smax, int pagestep)
{
	this->mSubject->SetRange(smin,smax,pagestep);
}


template<class T>
void iggWidgetKeySlider<T>::SetStretch(int label, int slider, int number)
{
	this->mSubject->SetStretch(label,slider,number);
}


template<class T>
void iggWidgetKeySlider<T>::SetNumberOfTicks(int number)
{
	this->mSubject->SetNumberOfTicks(number);
}

