/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#ifndef __slider
#define __slider

#include "widget.h"
#include "dingl.h"
#include "viewwin.h"
#include "utils.h"
#include "font.h"

#include <algorithm>

extern int mousex;
extern viewport view;
extern int lmb;

template <typename T> class values {

  T low, high, delta, val;
  float amount;

  public:

    values (T l = 0, T h = 0, float a = 0) {
      set_limits (l, h);
      set_amount (a);
    }

    operator T() {
      return ((T) val);
    }

    void set_limits (T l, T h) {
      low = l;
      high = h;
      delta = high - low;
    }

    void get_limits (T& l, T& h) {
      l = low;
      h = high;
    }

    T get_val () const { return val; }

    void set_val (const T& v) {
      val = v;
      clamp<T> (low, val, high);
      amount = (val - low) * 1. / delta;
    }

    void set_amount (float a) {
      amount = a;
      clamp<float> (0.0, amount, 1.0);
      val = low + delta * amount;
    }

    float get_amount () const { return amount; }

};

#include <list>
using std::list;

template <typename T> class slider : public widget {

  values<T> vx;
  int dx;

  int lmb_clicked;
  int slide;

  list < change_listener<slider> *> lsnrs;
  typedef typename list< change_listener<slider> * >::iterator iter;

  public:

    slider (int w, int h) : widget (0, 0, w, h), vx (0, 0, 0), dx (0) {lmb_clicked = slide = 0;}

    int handle_input () {

      widget::handle_input ();

      const box<int>& e = get_extents ();

      int ret = 0;
      if (lmb) {
        if (lmb_clicked == 0) {
          if (slide) {
            slide = 0;
            widget::focus = 0;
            ret = 1;
          }
          else if (hovering()) {
            slide = 1;
            widget::focus = this;
            ret = 1;
          }
        }
        lmb_clicked = 1;
      } else {
        if (slide) {
          dx = mousex - e.left;
          clamp (0, dx, e.width);
          float amount = dx * e.width_1;
          vx.set_amount (amount);
          for (iter i = lsnrs.begin (), j = lsnrs.end (); i != j; ++i) (*i)->changed (*this);
          ret = 1;
        }
        lmb_clicked = 0;
      }

      return ret;

    }

    void draw () {

      glEnable (GL_BLEND);
      glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

      const color& clr = get_color ();

      glColor4f (clr.r, clr.g, clr.b, 0.25);

      const box<int>& e = get_extents ();
        glRecti (e.left, e.bottom, e.right, e.top);

      glColor4f (clr.r, clr.g, clr.b, 1);
        glRecti (e.left, e.bottom, e.left + dx, e.top);

      glDisable (GL_BLEND);

    }

    void update () {
      float amount = vx.get_amount ();
      const box<int>& e = get_extents ();
      dx = amount * e.width;
      for (iter i = lsnrs.begin (), j = lsnrs.end (); i != j; ++i) (*i)->changed (*this);
    }

    void add_listener (change_listener<slider>* sl) {
      lsnrs.push_back (sl);
    }

    T get_val () const { return vx.get_val (); }

    void set_val (const T& t) {
      vx.set_val (t);
      update ();
    }

    void set_limits (T l, T h) {
      float v = get_val ();
      vx = values<T> (l, h);
      set_val (v);
    }

    void get_limits (T& l, T& h) {
      vx.get_limits (l, h);
    }



};

#endif
