#! /bin/bash

# Copyright (C) 2015 Thomas Lange <lange@informatik.uni-koeln.de>
# Copyright (C) 2022-2023 Pädagogisches Landesinstitut Rheinland-Pfalz
# Copyright (C) 2022-2023 Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

if [ X$FAI_ACTION = Xinstall -o X$FAI_ACTION = Xdirinstall -o X$FAI_ACTION = X ]; then
	:
else
	return 0
fi

[ "$flag_menu" ] || return 0

# this script is localizable
export TEXTDOMAIN="debian-edu-router-fai"
. gettext.sh
. /etc/environment.FAI-dialogs

out=$(tty)
tempfile=`(mktemp) 2>/dev/null`
tempfile2=`(mktemp) 2>/dev/null`
trap "rm -f $tempfile $tempfile2" EXIT INT QUIT

# declare the data structure, use associative arrays
declare -A arshort
declare -A ardesc
declare -A arlong
declare -A arclasses
declare -a list


parse_profile() {

	# read a profile and add all info to the data structure

	local short
	local long
	local desc
	local name
	local classes
	local lflag=0

	# disable word splitting when reading a line, this helps reading a keyword without a value
	local OIF=$IFS
	IFS=

	while read -r line || [[ -n $line ]]; do

		if [[ $line =~ "Name: " ]]; then
			if [ -n "$long" ]; then
				arlong[$name]="$long"
			fi
			short=
			desc=
			long=
			classes=
			lflag=0
			name=${line##Name: }
			[ $debug ] && echo "XX NAME $name found"
			list+=("$name")  # add new item to list
			continue
		fi

		if [[ $line =~ "Description: " ]]; then
			lflag=0
			desc=${line##Description: }
			[ $debug ] && echo "XX $desc found"
			ardesc[$name]="$desc"
			continue
		fi

		if [[ $line =~ "Short: " ]]; then
			lflag=0
			short=${line##Short: }
			[ $debug ] && echo "XX $short found"
			arshort[$name]="$short"
			continue
		fi

		if [[ $line =~ "Classes: " ]]; then
			lflag=0
			classes=${line##Classes: }
			[ $debug ] && echo "XX classes found"
			arclasses[$name]="$classes"
			continue
		fi

		if [[ $line =~ "Long: " ]]; then
			lflag=1
			long=${line##Long: }
			[ $debug ] && echo "XX long found"
		# else it's another long line
		elif [ $lflag -eq 1 ]; then
			long+="\n$line"
		fi

		if [[ $line =~ "Default: " ]]; then
			lflag=0
			default=${line##Default: }
			continue
		fi

	done < $1

	if [ -n "$long" ]; then
		arlong[$name]="$long"
	fi
	IFS=$OIF
}

prtresult() {

	# set newclasses which is used by fai-class(1)
	local res=$(<$tempfile)
	echo "$BASH_SOURCE defined new classes: ${arclasses[$res]}"
	newclasses="${arclasses[$res]}"
}

updateinstallerlocale() {
	# Extract LANG=<language> class from newclasses (string containing those classes
	# applicable to the FAI installation)
	local lang_class=`echo " $newclasses " | sed -Ee "s/.* (LANG_[^ ]+) .*/\1/"`

	if [ -s "${lang_class}.var" ]; then
		# unset LANG env var and try to detect it from LANG_<language>.var class file
		unset LANG
		. "${lang_class}.var"
		if [ -n "${LANG}" ]; then
			echo "export LANG=\"${LANG}\"" > "/etc/environment.FAI-dialogs"
			return 0
		fi
	fi

	echo "export LANG=\"C.UTF-8\"" > "/etc/environment.FAI-dialogs"
}

# read all files with name matching *.profile
_parsed=0
shopt -s nullglob
for _f in *.profile; do
	parse_profile $_f
	_parsed=1
done
unset _f

# do nothing if no profile was read
if [ $_parsed -eq 0 ]; then
	unset _parsed
	return 0
fi

# create the argument list containing the menu entries
# and the help text file
for i in "${list[@]}"; do
	par+=("$i")
	par+=("${ardesc[${i}]}")
	par+=("${arshort[${i}]}")
	echo "$(eval_gettext "Name:") ${i}" >> $tempfile2
	echo -e ${arlong[${i}]} >> $tempfile2
	echo -e "$(eval_gettext "Classes:") " ${arclasses[${i}]} "\n" >> $tempfile2
done
unset i

while true; do

	dialog --clear --item-help --backtitle "$(eval_gettext "FAI - Fully Automatic Installation")" \
	       --title " $(eval_gettext "Debian Edu Router Installation") " \
	       --help-button \
	       --default-item "$default" \
	       --menu "$(eval_gettext "\nPlease, select your Debian Edu Router installation profile.\n\nThe profile will determine installation properties such as system language, network setup, available features, etc.\n\n\n")" \
	       15 78 0 "${par[@]}" 2> $tempfile  1> $out

	_retval=$?
	case $_retval in
		0)
			prtresult
			updateinstallerlocale
			break ;;
		1)
			echo "$(eval_gettext "No profile selected.")"
			break ;;
		2)
			dialog --title "$(eval_gettext "Description of all profiles")" --textbox $tempfile2 0 0 1> $out;;
	esac

done
unset LANG
unset par ardesc arshort arlong arclasses list tempfile tempfile2 _parsed _retval line
