!-------------------------------------------------------------------------------

! This file is part of Code_Saturne, a general-purpose CFD tool.
!
! Copyright (C) 1998-2013 EDF S.A.
!
! This program is free software; you can redistribute it and/or modify it under
! the terms of the GNU General Public License as published by the Free Software
! Foundation; either version 2 of the License, or (at your option) any later
! version.
!
! This program is distributed in the hope that it will be useful, but WITHOUT
! ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
! details.
!
! You should have received a copy of the GNU General Public License along with
! this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
! Street, Fifth Floor, Boston, MA 02110-1301, USA.

!-------------------------------------------------------------------------------

!> \file chem_luscheme3.f90
!> \brief Routines for atmospheric chemical scheme 3
!> \remarks
!>  These routines are automatically generated by SPACK
!>  See CEREA: http://cerea.enpc.fr/polyphemus

!> kinetic_3
!> \brief Computation of kinetic rates for atmospheric chemistry
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name              role
!------------------------------------------------------------------------------
!> \param[in]     nr                total number of chemical reactions
!> \param[in]     option_photolysis flag to activate or not photolysis reactions
!> \param[in]     azi               solar zenith angle
!> \param[in]     att               atmospheric attenuation variable
!> \param[in]     temp              temperature
!> \param[in]     press             pressure
!> \param[in]     xlw               water massic fraction
!> \param[out]    rk(nr)            kinetic rates
!______________________________________________________________________________

subroutine kinetic_3(nr,rk,temp,xlw,press,azi,att,                  &
     option_photolysis)

implicit none

! Arguments

integer nr
double precision rk(nr),temp,xlw,press
double precision azi, att
integer option_photolysis

! Local variables

double precision effko,rapk,factor,summ
double precision ylh2o

!     Compute third body.
!     Conversion = Avogadro*1d-6/Perfect gas constant.
!     PRESS in Pascal, SUMM in molecules/cm3, TEMP in Kelvin

summ = press * 7.243d16 / temp

!     Number of water molecules computed from the massic fraction
!     (absolute humidity)

ylh2o = 29.d0*summ*xlw/(18.d0+11.d0*xlw)

!     For the zenithal angle at tropics

azi=abs(azi)


if(option_photolysis.eq.2) then
 rk(  1)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(  1)=-0.1302720567168795d-07
 rk(  1)=-0.7822279432831311d-06+(azi- 0.00d+00) * rk(  1)
 rk(  1)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(  1)
 rk(  1)= 0.9310260000000001d-02+(azi- 0.00d+00) * rk(  1)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(  1)= 0.3771617015067078d-08
 rk(  1)=-0.1173044113433769d-05+(azi- 0.10d+02) * rk(  1)
 rk(  1)=-0.1955272056716901d-04+(azi- 0.10d+02) * rk(  1)
 rk(  1)= 0.9219010000000000d-02+(azi- 0.10d+02) * rk(  1)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(  1)=-0.5859262388581815d-08
 rk(  1)=-0.1059895602981758d-05+(azi- 0.20d+02) * rk(  1)
 rk(  1)=-0.4188211773132428d-04+(azi- 0.20d+02) * rk(  1)
 rk(  1)= 0.8909950000000000d-02+(azi- 0.20d+02) * rk(  1)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(  1)=-0.7024567460738029d-08
 rk(  1)=-0.1235673474639213d-05+(azi- 0.30d+02) * rk(  1)
 rk(  1)=-0.6483780850753392d-04+(azi- 0.30d+02) * rk(  1)
 rk(  1)= 0.8379279999999999d-02+(azi- 0.30d+02) * rk(  1)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(  1)=-0.9202467768466835d-08
 rk(  1)=-0.1446410498461367d-05+(azi- 0.40d+02) * rk(  1)
 rk(  1)=-0.9165864823853972d-04+(azi- 0.40d+02) * rk(  1)
 rk(  1)= 0.7600310000000000d-02+(azi- 0.40d+02) * rk(  1)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(  1)=-0.1612556146540100d-07
 rk(  1)=-0.1722484531515342d-05+(azi- 0.50d+02) * rk(  1)
 rk(  1)=-0.1233475985383066d-03+(azi- 0.50d+02) * rk(  1)
 rk(  1)= 0.6529880000000000d-02+(azi- 0.50d+02) * rk(  1)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(  1)= 0.3226471363007382d-07
 rk(  1)=-0.2206251375477548d-05+(azi- 0.60d+02) * rk(  1)
 rk(  1)=-0.1626349576082332d-03+(azi- 0.60d+02) * rk(  1)
 rk(  1)= 0.5108030000000000d-02+(azi- 0.60d+02) * rk(  1)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(  1)= 0.2027078243961372d-06
 rk(  1)=-0.1238309966574737d-05+(azi- 0.70d+02) * rk(  1)
 rk(  1)=-0.1970805710287543d-03+(azi- 0.70d+02) * rk(  1)
 rk(  1)= 0.3293320000000000d-02+(azi- 0.70d+02) * rk(  1)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(  1)=-0.7448311471194499d-07
 rk(  1)= 0.3626677818932555d-05+(azi- 0.78d+02) * rk(  1)
 rk(  1)=-0.1779736282099126d-03+(azi- 0.78d+02) * rk(  1)
 rk(  1)= 0.1741210000000000d-02+(azi- 0.78d+02) * rk(  1)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(  1)= 0.2490309929270573d-05
 rk(  1)= 0.1839083065842406d-05+(azi- 0.86d+02) * rk(  1)
 rk(  1)=-0.1342475411316713d-03+(azi- 0.86d+02) * rk(  1)
 rk(  1)= 0.5113930000000000d-03+(azi- 0.86d+02) * rk(  1)
elseif(azi.ge.90.d0)then
 rk(  1)= 0.1632080000000000d-03
endif
if(att.lt.0.99999) rk(  1) = rk(  1) * att
endif

rk(  2) = summ * 6.0d-34 * (temp/3.d2) ** (-2.4d0)
rk(  2) = rk(  2) * summ * 0.2d0
rk(  3) =  dexp(-0.2653240882726044d+02                           &
 - (  0.1500000000000000d+04 )/temp)
rk(  4) =  dexp(-0.2590825451818744d+02                           &
 - ( -0.1800000000000000d+03 )/temp)
effko =  0.2500000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.1800000000000000d+01))
rapk =  0.2200000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.7000000000000000d+00))
rk(  5) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.9000000000000000d-31* (temp / 3.d2)                    &
           **(- ( 0.1500000000000000d+01))
rapk =  0.3000000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk(  6) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk(  7) =  dexp(-0.2975128465212864d+02                           &
 - (  0.2450000000000000d+04 )/temp)

if(option_photolysis.eq.2) then
 rk(  8)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(  8)=-0.5928286648807996d-09
 rk(  8)=-0.3096171335119280d-07+(azi- 0.00d+00) * rk(  8)
 rk(  8)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(  8)
 rk(  8)= 0.4927580000000000d-03+(azi- 0.00d+00) * rk(  8)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(  8)= 0.1444859946426876d-09
 rk(  8)=-0.4874657329761677d-07+(azi- 0.10d+02) * rk(  8)
 rk(  8)=-0.7970828664880956d-06+(azi- 0.10d+02) * rk(  8)
 rk(  8)= 0.4890690000000000d-03+(azi- 0.10d+02) * rk(  8)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(  8)=-0.5511531369012520d-10
 rk(  8)=-0.4441199345833616d-07+(azi- 0.20d+02) * rk(  8)
 rk(  8)=-0.1728668534047625d-05+(azi- 0.20d+02) * rk(  8)
 rk(  8)= 0.4763680000000000d-03+(azi- 0.20d+02) * rk(  8)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(  8)=-0.3000247398821449d-09
 rk(  8)=-0.4606545286904014d-07+(azi- 0.30d+02) * rk(  8)
 rk(  8)=-0.2633442997321385d-05+(azi- 0.30d+02) * rk(  8)
 rk(  8)= 0.4545850000000000d-03+(azi- 0.30d+02) * rk(  8)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(  8)=-0.2397857267812366d-09
 rk(  8)=-0.5506619506550444d-07+(azi- 0.40d+02) * rk(  8)
 rk(  8)=-0.3644759476666826d-05+(azi- 0.40d+02) * rk(  8)
 rk(  8)= 0.4233440000000000d-03+(azi- 0.40d+02) * rk(  8)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(  8)=-0.1844832352993067d-08
 rk(  8)=-0.6225976686893995d-07+(azi- 0.50d+02) * rk(  8)
 rk(  8)=-0.4818019096011278d-05+(azi- 0.50d+02) * rk(  8)
 rk(  8)= 0.3811500000000000d-03+(azi- 0.50d+02) * rk(  8)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(  8)= 0.1101151387530619d-09
 rk(  8)=-0.1176047374587370d-06+(azi- 0.60d+02) * rk(  8)
 rk(  8)=-0.6616664139287950d-05+(azi- 0.60d+02) * rk(  8)
 rk(  8)= 0.3248990000000000d-03+(azi- 0.60d+02) * rk(  8)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(  8)=-0.1557211541866474d-07
 rk(  8)=-0.1143012832961418d-06+(azi- 0.70d+02) * rk(  8)
 rk(  8)=-0.8935724346837023d-05+(azi- 0.70d+02) * rk(  8)
 rk(  8)= 0.2470820000000000d-03+(azi- 0.70d+02) * rk(  8)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(  8)= 0.4048472604232427d-07
 rk(  8)=-0.4880320533439059d-06+(azi- 0.78d+02) * rk(  8)
 rk(  8)=-0.1375439103995816d-04+(azi- 0.78d+02) * rk(  8)
 rk(  8)= 0.1603080000000000d-03+(azi- 0.78d+02) * rk(  8)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(  8)= 0.2066880316654862d-06
 rk(  8)= 0.4836013716715513d-06+(azi- 0.86d+02) * rk(  8)
 rk(  8)=-0.1378983649333310d-04+(azi- 0.86d+02) * rk(  8)
 rk(  8)= 0.3976700000000000d-04+(azi- 0.86d+02) * rk(  8)
elseif(azi.ge.90.d0)then
 rk(  8)= 0.5573310000000000d-05
endif
if(att.lt.0.99999) rk(  8) = rk(  8) * att
endif


if(option_photolysis.eq.2) then
 rk(  9)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(  9)=-0.8776629099833464d-10
 rk(  9)=-0.1165033709001661d-07+(azi- 0.00d+00) * rk(  9)
 rk(  9)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(  9)
 rk(  9)= 0.3523480000000000d-04+(azi- 0.00d+00) * rk(  9)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(  9)= 0.1474988729949909d-09
 rk(  9)=-0.1428332581996665d-07+(azi- 0.10d+02) * rk(  9)
 rk(  9)=-0.2593366290998327d-06+(azi- 0.10d+02) * rk(  9)
 rk(  9)= 0.3398200000000000d-04+(azi- 0.10d+02) * rk(  9)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(  9)= 0.1300707990183827d-09
 rk(  9)=-0.9858359630116941d-08+(azi- 0.20d+02) * rk(  9)
 rk(  9)=-0.5007534836006686d-06+(azi- 0.20d+02) * rk(  9)
 rk(  9)= 0.3010780000000000d-04+(azi- 0.20d+02) * rk(  9)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(  9)= 0.1988179309314732d-09
 rk(  9)=-0.5956235659565481d-08+(azi- 0.30d+02) * rk(  9)
 rk(  9)=-0.6588994364974921d-06+(azi- 0.30d+02) * rk(  9)
 rk(  9)= 0.2424450000000000d-04+(azi- 0.30d+02) * rk(  9)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(  9)= 0.2219574772557277d-09
 rk(  9)= 0.8302268378835231d-11+(azi- 0.40d+02) * rk(  9)
 rk(  9)=-0.7183787704093613d-06+(azi- 0.40d+02) * rk(  9)
 rk(  9)= 0.1725870000000000d-04+(azi- 0.40d+02) * rk(  9)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(  9)= 0.1913521600455895d-09
 rk(  9)= 0.6667026586050136d-08+(azi- 0.50d+02) * rk(  9)
 rk(  9)=-0.6516254818650674d-06+(azi- 0.50d+02) * rk(  9)
 rk(  9)= 0.1029770000000000d-04+(azi- 0.50d+02) * rk(  9)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(  9)= 0.1602388256152816d-10
 rk(  9)= 0.1240759138741867d-07+(azi- 0.60d+02) * rk(  9)
 rk(  9)=-0.4608793021303539d-06+(azi- 0.60d+02) * rk(  9)
 rk(  9)= 0.4639500000000000d-05+(azi- 0.60d+02) * rk(  9)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(  9)=-0.3089359890783960d-09
 rk(  9)= 0.1288830786428400d-07+(azi- 0.70d+02) * rk(  9)
 rk(  9)=-0.2079203096133002d-06+(azi- 0.70d+02) * rk(  9)
 rk(  9)= 0.1287490000000000d-05+(azi- 0.70d+02) * rk(  9)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(  9)=-0.2034952628632162d-09
 rk(  9)= 0.5473844126395715d-08+(azi- 0.78d+02) * rk(  9)
 rk(  9)=-0.6102309368797090d-07+(azi- 0.78d+02) * rk(  9)
 rk(  9)= 0.2908040000000000d-06+(azi- 0.78d+02) * rk(  9)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(  9)= 0.1623544916128788d-09
 rk(  9)= 0.5899578175158973d-09+(azi- 0.86d+02) * rk(  9)
 rk(  9)=-0.1251267813581064d-07+(azi- 0.86d+02) * rk(  9)
 rk(  9)= 0.4875570000000000d-07+(azi- 0.86d+02) * rk(  9)
elseif(azi.ge.90.d0)then
 rk(  9)= 0.1853500000000000d-07
endif
if(att.lt.0.99999) rk(  9) = rk(  9) * att
endif

rk( 10) =  dexp(-0.2458649867820512d+02                           &
 - ( -0.1020000000000000d+03 )/temp)
rk( 10) = rk( 10) * summ
rk( 11) =  0.2200000000000000d-09
rk( 11) = rk( 11) * ylh2o
rk( 12) =  dexp(-0.2710039286486638d+02                           &
 - (  0.9400000000000000d+03 )/temp)
rk( 13) =  dexp(-0.3223619130191664d+02                           &
 - (  0.4900000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 14)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 14)=-0.1926445199576598d-06
 rk( 14)=-0.1114355480042343d-04+(azi- 0.00d+00) * rk( 14)
 rk( 14)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 14)
 rk( 14)= 0.2129960000000000d+00+(azi- 0.00d+00) * rk( 14)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 14)=-0.3806644012699938d-07
 rk( 14)=-0.1692289039915322d-04+(azi- 0.10d+02) * rk( 14)
 rk( 14)=-0.2806644519957665d-03+(azi- 0.10d+02) * rk( 14)
 rk( 14)= 0.2116890000000000d+00+(azi- 0.10d+02) * rk( 14)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 14)= 0.3591028046562922d-07
 rk( 14)=-0.1806488360296321d-04+(azi- 0.20d+02) * rk( 14)
 rk( 14)=-0.6305421920169308d-03+(azi- 0.20d+02) * rk( 14)
 rk( 14)= 0.2071520000000000d+00+(azi- 0.20d+02) * rk( 14)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 14)=-0.2185746817355116d-06
 rk( 14)=-0.1698757518899435d-04+(azi- 0.30d+02) * rk( 14)
 rk( 14)=-0.9810667799365058d-03+(azi- 0.30d+02) * rk( 14)
 rk( 14)= 0.1990760000000000d+00+(azi- 0.30d+02) * rk( 14)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 14)=-0.2006115535235656d-06
 rk( 14)=-0.2354481564105959d-04+(azi- 0.40d+02) * rk( 14)
 rk( 14)=-0.1386390688237046d-02+(azi- 0.40d+02) * rk( 14)
 rk( 14)= 0.1873480000000000d+00+(azi- 0.40d+02) * rk( 14)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 14)=-0.1055979104170267d-05
 rk( 14)=-0.2956316224676618d-04+(azi- 0.50d+02) * rk( 14)
 rk( 14)=-0.1917470467115309d-02+(azi- 0.50d+02) * rk( 14)
 rk( 14)= 0.1709290000000000d+00+(azi- 0.50d+02) * rk( 14)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 14)=-0.3204720297952934d-06
 rk( 14)=-0.6124253537187832d-04+(azi- 0.60d+02) * rk( 14)
 rk( 14)=-0.2825527443301723d-02+(azi- 0.60d+02) * rk( 14)
 rk( 14)= 0.1477420000000000d+00+(azi- 0.60d+02) * rk( 14)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 14)=-0.6785572971821502d-05
 rk( 14)=-0.7085669626571978d-04+(azi- 0.70d+02) * rk( 14)
 rk( 14)=-0.4146519759677725d-02+(azi- 0.70d+02) * rk( 14)
 rk( 14)= 0.1130420000000000d+00+(azi- 0.70d+02) * rk( 14)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 14)= 0.2349184361304102d-04
 rk( 14)=-0.2337104475894081d-03+(azi- 0.78d+02) * rk( 14)
 rk( 14)=-0.6583056910518970d-02+(azi- 0.78d+02) * rk( 14)
 rk( 14)= 0.7186080000000000d-01+(azi- 0.78d+02) * rk( 14)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 14)= 0.6606749385962285d-04
 rk( 14)= 0.3300937991232988d-03+(azi- 0.86d+02) * rk( 14)
 rk( 14)=-0.5811990098247399d-02+(azi- 0.86d+02) * rk( 14)
 rk( 14)= 0.1626670000000000d-01+(azi- 0.86d+02) * rk( 14)
elseif(azi.ge.90.d0)then
 rk( 14)= 0.2528560000000000d-02
endif
if(att.lt.0.99999) rk( 14) = rk( 14) * att
endif


if(option_photolysis.eq.2) then
 rk( 15)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 15)=-0.3691161034363460d-07
 rk( 15)=-0.1177883896563626d-05+(azi- 0.00d+00) * rk( 15)
 rk( 15)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 15)
 rk( 15)= 0.2656490000000000d-01+(azi- 0.00d+00) * rk( 15)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 15)= 0.1703483103089462d-07
 rk( 15)=-0.2285232206872663d-05+(azi- 0.10d+02) * rk( 15)
 rk( 15)=-0.3463116103436289d-04+(azi- 0.10d+02) * rk( 15)
 rk( 15)= 0.2641020000000000d-01+(azi- 0.10d+02) * rk( 15)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 15)=-0.1532771377994130d-07
 rk( 15)=-0.1774187275945825d-05+(azi- 0.20d+02) * rk( 15)
 rk( 15)=-0.7522535586254776d-04+(azi- 0.20d+02) * rk( 15)
 rk( 15)= 0.2585240000000000d-01+(azi- 0.20d+02) * rk( 15)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 15)=-0.1372397591112858d-07
 rk( 15)=-0.2234018689344064d-05+(azi- 0.30d+02) * rk( 15)
 rk( 15)=-0.1153074155154466d-03+(azi- 0.30d+02) * rk( 15)
 rk( 15)= 0.2490740000000000d-01+(azi- 0.30d+02) * rk( 15)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 15)=-0.2747638257554304d-07
 rk( 15)=-0.2645737966677932d-05+(azi- 0.40d+02) * rk( 15)
 rk( 15)=-0.1641049820756665d-03+(azi- 0.40d+02) * rk( 15)
 rk( 15)= 0.2351720000000000d-01+(azi- 0.40d+02) * rk( 15)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 15)=-0.1322704937867030d-06
 rk( 15)=-0.3470029443944237d-05+(azi- 0.50d+02) * rk( 15)
 rk( 15)=-0.2252626561818883d-03+(azi- 0.50d+02) * rk( 15)
 rk( 15)= 0.2158410000000000d-01+(azi- 0.50d+02) * rk( 15)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 15)=-0.2304164227760796d-07
 rk( 15)=-0.7438144257545598d-05+(azi- 0.60d+02) * rk( 15)
 rk( 15)=-0.3343443931967845d-03+(azi- 0.60d+02) * rk( 15)
 rk( 15)= 0.1885220000000000d-01+(azi- 0.60d+02) * rk( 15)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 15)=-0.1151055949407016d-05
 rk( 15)=-0.8129393525872428d-05+(azi- 0.70d+02) * rk( 15)
 rk( 15)=-0.4900197710309764d-03+(azi- 0.70d+02) * rk( 15)
 rk( 15)= 0.1474190000000000d-01+(azi- 0.70d+02) * rk( 15)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 15)= 0.3051007034753431d-05
 rk( 15)=-0.3575473631163908d-04+(azi- 0.78d+02) * rk( 15)
 rk( 15)=-0.8410928097310894d-03+(azi- 0.78d+02) * rk( 15)
 rk( 15)= 0.9712119999999999d-02+(azi- 0.78d+02) * rk( 15)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 15)= 0.1099207874719760d-04
 rk( 15)= 0.3746943252241897d-04+(azi- 0.86d+02) * rk( 15)
 rk( 15)=-0.8273752400447254d-03+(azi- 0.86d+02) * rk( 15)
 rk( 15)= 0.2257190000000000d-02+(azi- 0.86d+02) * rk( 15)
elseif(azi.ge.90.d0)then
 rk( 15)= 0.2506930000000000d-03
endif
if(att.lt.0.99999) rk( 15) = rk( 15) * att
endif

rk( 16) =  dexp(-0.2492297091482634d+02                           &
 - ( -0.1700000000000000d+03 )/temp)
rk( 17) =  dexp(-0.3073211390514037d+02                           &
 - (  0.1260000000000000d+04 )/temp)
effko =  0.2000000000000000d-29* (temp / 3.d2)                    &
           **(- ( 0.4400000000000000d+01))
rapk =  0.1400000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.7000000000000000d+00))
rk( 18) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk( 19) =  0.1000000000000000d-21
rk( 19) = rk( 19) * ylh2o
effko =  0.1300000000000000d-02* (temp / 3.d2)                    &
           **(- ( 0.3500000000000000d+01))*                       &
           dexp(- 0.1100000000000000d+05/temp)
rapk =  0.9700000000000000d+15* (temp / 3.d2)                     &
            **(- (-0.1000000000000000d+00))*                      &
           dexp(- 0.1108000000000000d+05/temp)
rk( 20) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.4500d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))
rk( 21) =  dexp(-0.8860689615829534d+02                           &
 - ( -0.5300000000000000d+03 )/temp)
rk( 21) = rk( 21) * summ * 0.2d0
rk( 22) =  0.5000000000000000d-39
rk( 22) = rk( 22) * ylh2o
effko =  0.7000000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.2600000000000000d+01))
rapk =  0.3600000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.1000000000000000d+00))
rk( 23) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))

if(option_photolysis.eq.2) then
 rk( 24)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 24)=-0.2887225450832658d-08
 rk( 24)=-0.1810277454916759d-06+(azi- 0.00d+00) * rk( 24)
 rk( 24)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 24)
 rk( 24)= 0.2046710000000000d-02+(azi- 0.00d+00) * rk( 24)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 24)= 0.8216763524985595d-09
 rk( 24)=-0.2676445090166556d-06+(azi- 0.10d+02) * rk( 24)
 rk( 24)=-0.4486722545083316d-05+(azi- 0.10d+02) * rk( 24)
 rk( 24)= 0.2025720000000000d-02+(azi- 0.10d+02) * rk( 24)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 24)=-0.1309479959161308d-08
 rk( 24)=-0.2429942184416991d-06+(azi- 0.20d+02) * rk( 24)
 rk( 24)=-0.9593109819666860d-05+(azi- 0.20d+02) * rk( 24)
 rk( 24)= 0.1954910000000000d-02+(azi- 0.20d+02) * rk( 24)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 24)=-0.1523756515853649d-08
 rk( 24)=-0.2822786172165394d-06+(azi- 0.30d+02) * rk( 24)
 rk( 24)=-0.1484583817624924d-04+(azi- 0.30d+02) * rk( 24)
 rk( 24)= 0.1833370000000000d-02+(azi- 0.30d+02) * rk( 24)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 24)=-0.1745493977424016d-08
 rk( 24)=-0.3279913126921461d-06+(azi- 0.40d+02) * rk( 24)
 rk( 24)=-0.2094853747533609d-04+(azi- 0.40d+02) * rk( 24)
 rk( 24)= 0.1655160000000000d-02+(azi- 0.40d+02) * rk( 24)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 24)=-0.2754267574450560d-08
 rk( 24)=-0.3803561320148624d-06+(azi- 0.50d+02) * rk( 24)
 rk( 24)=-0.2803201192240625d-04+(azi- 0.50d+02) * rk( 24)
 rk( 24)= 0.1411130000000000d-02+(azi- 0.50d+02) * rk( 24)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 24)= 0.1037656427523324d-07
 rk( 24)=-0.4629841592484096d-06+(azi- 0.60d+02) * rk( 24)
 rk( 24)=-0.3646541483503878d-04+(azi- 0.60d+02) * rk( 24)
 rk( 24)= 0.1090020000000000d-02+(azi- 0.60d+02) * rk( 24)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 24)= 0.4335158727139456d-07
 rk( 24)=-0.1516872309913989d-06+(azi- 0.70d+02) * rk( 24)
 rk( 24)=-0.4261212873743828d-04+(azi- 0.70d+02) * rk( 24)
 rk( 24)= 0.6894440000000000d-03+(azi- 0.70d+02) * rk( 24)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 24)=-0.2976581610911796d-07
 rk( 24)= 0.8887508635220705d-06+(azi- 0.78d+02) * rk( 24)
 rk( 24)=-0.3671561967719464d-04+(azi- 0.78d+02) * rk( 24)
 rk( 24)= 0.3610350000000000d-03+(azi- 0.78d+02) * rk( 24)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 24)= 0.5586598403857848d-06
 rk( 24)= 0.1743712769036732d-06+(azi- 0.86d+02) * rk( 24)
 rk( 24)=-0.2821064255377481d-04+(azi- 0.86d+02) * rk( 24)
 rk( 24)= 0.1089500000000000d-03+(azi- 0.86d+02) * rk( 24)
elseif(azi.ge.90.d0)then
 rk( 24)= 0.3465160000000000d-04
endif
if(att.lt.0.99999) rk( 24) = rk( 24) * att
endif

rk( 25) =  dexp(-0.2474064935803238d+02                           &
 - (  0.3900000000000000d+03 )/temp)
rk( 26) =  0.9999999999999999d-20
rk( 27) =  dexp(-0.8322449114623726d+01                           &
 + (-0.3000000000000000d+01 * dlog(temp)) )
effko = 2.4d-14 * dexp(460.0d0 / temp)
rapk = 2.7d-17 * dexp(2199.0d0 / temp)
factor =6.5d-34 * dexp(1335.0d0 / temp) * summ
rk( 28) = effko + factor/(1.0d0 + factor / rapk)
rk( 29) =  dexp(-0.2637825814743318d+02                           &
 - ( -0.2500000000000000d+03 )/temp)
effko =  0.1800000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.3200000000000000d+01))
rapk =  0.4700000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk( 30) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.4100000000000000d-04* (temp / 3.d2)                    &
           **(- ( 0.0000000000000000d+00))*                       &
           dexp(- 0.1065000000000000d+05/temp)
rapk =  0.4800000000000000d+16* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))*                      &
           dexp(- 0.1117000000000000d+05/temp)
rk( 31) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.6000d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))
rk( 32) =  dexp(-0.2736865685146106d+02                           &
 - ( -0.3800000000000000d+03 )/temp)
rk( 33) = 2.3d-13 * dexp(600.0d0 / temp)                          &
            + 1.7d-33* summ * dexp(1000.0d0 / temp)
rk( 34) = 3.22d-34 * dexp(2800.0d0 / temp) +                      &
            2.38d-54 * summ * dexp(3200.0d0 / temp)
rk( 34) = rk( 34) * ylh2o

if(option_photolysis.eq.2) then
 rk( 35)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 35)=-0.1441479345432039d-10
 rk( 35)=-0.1242452065456794d-08+(azi- 0.00d+00) * rk( 35)
 rk( 35)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 35)
 rk( 35)= 0.8394580000000000d-05+(azi- 0.00d+00) * rk( 35)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 35)= 0.8244380362960478d-11
 rk( 35)=-0.1674895869086405d-08+(azi- 0.10d+02) * rk( 35)
 rk( 35)=-0.2917347934543199d-07+(azi- 0.10d+02) * rk( 35)
 rk( 35)= 0.8255920000000000d-05+(azi- 0.10d+02) * rk( 35)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 35)= 0.1172720024787734d-12
 rk( 35)=-0.1427564458197592d-08+(azi- 0.20d+02) * rk( 35)
 rk( 35)=-0.6019808261827194d-07+(azi- 0.20d+02) * rk( 35)
 rk( 35)= 0.7804940000000000d-05+(azi- 0.20d+02) * rk( 35)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 35)= 0.4506531627127201d-11
 rk( 35)=-0.1424046298123240d-08+(azi- 0.30d+02) * rk( 35)
 rk( 35)=-0.8871419018148013d-07+(azi- 0.30d+02) * rk( 35)
 rk( 35)= 0.7060320000000000d-05+(azi- 0.30d+02) * rk( 35)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 35)= 0.1086660148900755d-10
 rk( 35)=-0.1288850349309390d-08+(azi- 0.40d+02) * rk( 35)
 rk( 35)=-0.1158431566558070d-06+(azi- 0.40d+02) * rk( 35)
 rk( 35)= 0.6035280000000000d-05+(azi- 0.40d+02) * rk( 35)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 35)= 0.1803706241686108d-10
 rk( 35)=-0.9628523046392104d-09+(azi- 0.50d+02) * rk( 35)
 rk( 35)=-0.1383601831952927d-06+(azi- 0.50d+02) * rk( 35)
 rk( 35)= 0.4758830000000000d-05+(azi- 0.50d+02) * rk( 35)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 35)= 0.7329514884355585d-10
 rk( 35)=-0.4217404321336693d-09+(azi- 0.60d+02) * rk( 35)
 rk( 35)=-0.1522061105630206d-06+(azi- 0.60d+02) * rk( 35)
 rk( 35)= 0.3296980000000000d-05+(azi- 0.60d+02) * rk( 35)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 35)= 0.6172011386267615d-10
 rk( 35)= 0.1777114033174383d-08+(azi- 0.70d+02) * rk( 35)
 rk( 35)=-0.1386523745526241d-06+(azi- 0.70d+02) * rk( 35)
 rk( 35)= 0.1806040000000000d-05+(azi- 0.70d+02) * rk( 35)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 35)=-0.8279704635596216d-10
 rk( 35)= 0.3258396765877763d-08+(azi- 0.78d+02) * rk( 35)
 rk( 35)=-0.9836828816022045d-07+(azi- 0.78d+02) * rk( 35)
 rk( 35)= 0.8421570000000000d-06+(azi- 0.78d+02) * rk( 35)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 35)= 0.1082517324572734d-08
 rk( 35)= 0.1271267653334671d-08+(azi- 0.86d+02) * rk( 35)
 rk( 35)=-0.6213097280649386d-07+(azi- 0.86d+02) * rk( 35)
 rk( 35)= 0.2213560000000000d-06+(azi- 0.86d+02) * rk( 35)
elseif(azi.ge.90.d0)then
 rk( 35)= 0.6245350000000000d-07
endif
if(att.lt.0.99999) rk( 35) = rk( 35) * att
endif

rk( 36) =  dexp(-0.2656631037893612d+02                           &
 - (  0.1600000000000000d+03 )/temp)
rk( 37) =  0.1100000000000000d-09
rk( 37) = rk( 37) * summ * 5.8d-7
rk( 38) =  dexp(-0.2592627302369012d+02                           &
 - (  0.2000000000000000d+04 )/temp)
rk( 38) = rk( 38) * summ * 5.8d-7
rk( 39) =  dexp(-0.2453997866257023d+02                           &
 - ( -0.1200000000000000d+03 )/temp)
rk( 40) =  dexp(-0.2619593659063922d+02                           &
 - (  0.2400000000000000d+03 )/temp)
effko =  0.6900000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.1000000000000000d+01))
rapk =  0.2600000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk( 41) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk( 42) =  dexp(-0.2375982010502066d+02                           &
 - ( -0.2500000000000000d+03 )/temp)
rk( 43) =  dexp(-0.2422982373426639d+02                           &
 - ( -0.2000000000000000d+03 )/temp)
rk( 44) =  dexp(-0.2729454887930734d+02                           &
 - (  0.2000000000000000d+04 )/temp)
rk( 45) =  0.9999999999999999d-11
rk( 46) =  0.2200000000000000d-10
rk( 47) =  0.3500000000000000d-11
rk( 48) =  0.1000000000000000d-16
rk( 49) =  dexp(-0.2779354004542632d+02                           &
 - (  0.2450000000000000d+04 )/temp)

if(option_photolysis.eq.2) then
 rk( 50)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 50)=-0.1022637521258300d-10
 rk( 50)=-0.1047636247874169d-08+(azi- 0.00d+00) * rk( 50)
 rk( 50)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 50)
 rk( 50)= 0.5038360000000000d-05+(azi- 0.00d+00) * rk( 50)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 50)= 0.9289125637748400d-11
 rk( 50)=-0.1354427504251659d-08+(azi- 0.10d+02) * rk( 50)
 rk( 50)=-0.2402063752125827d-07+(azi- 0.10d+02) * rk( 50)
 rk( 50)= 0.4923370000000000d-05+(azi- 0.10d+02) * rk( 50)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 50)= 0.4979872661590682d-11
 rk( 50)=-0.1075753735119207d-08+(azi- 0.20d+02) * rk( 50)
 rk( 50)=-0.4832244991496692d-07+(azi- 0.20d+02) * rk( 50)
 rk( 50)= 0.4557010000000000d-05+(azi- 0.20d+02) * rk( 50)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 50)= 0.1055138371588688d-10
 rk( 50)=-0.9263575552714849d-09+(azi- 0.30d+02) * rk( 50)
 rk( 50)=-0.6834356281887375d-07+(azi- 0.30d+02) * rk( 50)
 rk( 50)= 0.3971190000000000d-05+(azi- 0.30d+02) * rk( 50)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 50)= 0.1620459247486068d-10
 rk( 50)=-0.6098160437948834d-09+(azi- 0.40d+02) * rk( 50)
 rk( 50)=-0.8370529880953754d-07+(azi- 0.40d+02) * rk( 50)
 rk( 50)= 0.3205670000000000d-05+(azi- 0.40d+02) * rk( 50)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 50)= 0.2215024638467897d-10
 rk( 50)=-0.1236782695490497d-09+(azi- 0.50d+02) * rk( 50)
 rk( 50)=-0.9104024194297703d-07+(azi- 0.50d+02) * rk( 50)
 rk( 50)= 0.2323840000000000d-05+(azi- 0.50d+02) * rk( 50)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 50)= 0.3999042198642357d-10
 rk( 50)= 0.5408291219911870d-09+(azi- 0.60d+02) * rk( 50)
 rk( 50)=-0.8686873341855619d-07+(azi- 0.60d+02) * rk( 50)
 rk( 50)= 0.1423220000000000d-05+(azi- 0.60d+02) * rk( 50)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 50)=-0.2067341717000215d-11
 rk( 50)= 0.1740541781586012d-08+(azi- 0.70d+02) * rk( 50)
 rk( 50)=-0.6405502438280072d-07+(azi- 0.70d+02) * rk( 50)
 rk( 50)= 0.6486060000000000d-06+(azi- 0.70d+02) * rk( 50)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 50)=-0.2916729149837185d-10
 rk( 50)= 0.1690925580378854d-08+(azi- 0.78d+02) * rk( 50)
 rk( 50)=-0.3660328548710213d-07+(azi- 0.78d+02) * rk( 50)
 rk( 50)= 0.2465020000000000d-06+(azi- 0.78d+02) * rk( 50)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 50)= 0.1504439894534147d-09
 rk( 50)= 0.9909105844077646d-09+(azi- 0.86d+02) * rk( 50)
 rk( 50)=-0.1514859616880918d-07+(azi- 0.86d+02) * rk( 50)
 rk( 50)= 0.4696130000000000d-07+(azi- 0.86d+02) * rk( 50)
elseif(azi.ge.90.d0)then
 rk( 50)= 0.1184990000000000d-07
endif
if(att.lt.0.99999) rk( 50) = rk( 50) * att
endif


if(option_photolysis.eq.2) then
 rk( 51)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 51)=-0.1532025548735494d-11
 rk( 51)=-0.1605297445126457d-09+(azi- 0.00d+00) * rk( 51)
 rk( 51)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 51)
 rk( 51)= 0.7475310000000001d-06+(azi- 0.00d+00) * rk( 51)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 51)= 0.1499076646206637d-11
 rk( 51)=-0.2064905109747105d-09+(azi- 0.10d+02) * rk( 51)
 rk( 51)=-0.3670202554873561d-08+(azi- 0.10d+02) * rk( 51)
 rk( 51)= 0.7299460000000000d-06+(azi- 0.10d+02) * rk( 51)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 51)= 0.8827189639089728d-12
 rk( 51)=-0.1615182115885116d-09+(azi- 0.20d+02) * rk( 51)
 rk( 51)=-0.7350289780505779d-08+(azi- 0.20d+02) * rk( 51)
 rk( 51)= 0.6740940000000000d-06+(azi- 0.20d+02) * rk( 51)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 51)= 0.1742047498157359d-11
 rk( 51)=-0.1350366426712420d-09+(azi- 0.30d+02) * rk( 51)
 rk( 51)=-0.1031583832310330d-07+(azi- 0.30d+02) * rk( 51)
 rk( 51)= 0.5853220000000000d-06+(azi- 0.30d+02) * rk( 51)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 51)= 0.2513091043461601d-11
 rk( 51)=-0.8277521772651622d-10+(azi- 0.40d+02) * rk( 51)
 rk( 51)=-0.1249395692708094d-07+(azi- 0.40d+02) * rk( 51)
 rk( 51)= 0.4704020000000000d-06+(azi- 0.40d+02) * rk( 51)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 51)= 0.3194588327996089d-11
 rk( 51)=-0.7382486422669833d-11+(azi- 0.50d+02) * rk( 51)
 rk( 51)=-0.1339553396857280d-07+(azi- 0.50d+02) * rk( 51)
 rk( 51)= 0.3396980000000000d-06+(azi- 0.50d+02) * rk( 51)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 51)= 0.5281855644545232d-11
 rk( 51)= 0.8845516341719630d-10+(azi- 0.60d+02) * rk( 51)
 rk( 51)=-0.1258480719862726d-07+(azi- 0.60d+02) * rk( 51)
 rk( 51)= 0.2081990000000000d-06+(azi- 0.60d+02) * rk( 51)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 51)=-0.2756160173816243d-12
 rk( 51)= 0.2469108327538047d-09+(azi- 0.70d+02) * rk( 51)
 rk( 51)=-0.9231147236917937d-08+(azi- 0.70d+02) * rk( 51)
 rk( 51)= 0.9647830000000000d-07+(azi- 0.70d+02) * rk( 51)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 51)=-0.5270116382809120d-11
 rk( 51)= 0.2402960483368575d-09+(azi- 0.78d+02) * rk( 51)
 rk( 51)=-0.5333492188194333d-08+(azi- 0.78d+02) * rk( 51)
 rk( 51)= 0.3829030000000000d-07+(azi- 0.78d+02) * rk( 51)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 51)= 0.3312732748114380d-10
 rk( 51)= 0.1138132551494386d-09+(azi- 0.86d+02) * rk( 51)
 rk( 51)=-0.2500617760290411d-08+(azi- 0.86d+02) * rk( 51)
 rk( 51)= 0.8303010000000000d-08+(azi- 0.86d+02) * rk( 51)
elseif(azi.ge.90.d0)then
 rk( 51)= 0.2241700000000000d-08
endif
if(att.lt.0.99999) rk( 51) = rk( 51) * att
endif


if(option_photolysis.eq.2) then
 rk( 52)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 52)=-0.1726633761595217d-06
 rk( 52)=-0.3763226238404795d-05+(azi- 0.00d+00) * rk( 52)
 rk( 52)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 52)
 rk( 52)= 0.2758968400000000d-01+(azi- 0.00d+00) * rk( 52)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 52)= 0.8277361284785716d-06
 rk( 52)=-0.8943127523190445d-05+(azi- 0.10d+02) * rk( 52)
 rk( 52)=-0.1270635376159524d-03+(azi- 0.10d+02) * rk( 52)
 rk( 52)= 0.2704069800000000d-01+(azi- 0.10d+02) * rk( 52)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 52)=-0.1363361137754772d-05
 rk( 52)= 0.1588895633116670d-04+(azi- 0.20d+02) * rk( 52)
 rk( 52)=-0.5760524953618990d-04+(azi- 0.20d+02) * rk( 52)
 rk( 52)= 0.2570348600000000d-01+(azi- 0.20d+02) * rk( 52)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 52)= 0.9462274225405240d-06
 rk( 52)=-0.2501187780147645d-04+(azi- 0.30d+02) * rk( 52)
 rk( 52)=-0.1488344642392872d-03+(azi- 0.30d+02) * rk( 52)
 rk( 52)= 0.2535296800000000d-01+(azi- 0.30d+02) * rk( 52)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 52)=-0.2183585524073524d-06
 rk( 52)= 0.3374944874739267d-05+(azi- 0.40d+02) * rk( 52)
 rk( 52)=-0.3652037935066590d-03+(azi- 0.40d+02) * rk( 52)
 rk( 52)= 0.2230966300000000d-01+(azi- 0.40d+02) * rk( 52)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 52)= 0.1378647870888786d-06
 rk( 52)=-0.3175811697481157d-05+(azi- 0.50d+02) * rk( 52)
 rk( 52)=-0.3632124617340762d-03+(azi- 0.50d+02) * rk( 52)
 rk( 52)= 0.1877676100000000d-01+(azi- 0.50d+02) * rk( 52)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 52)=-0.3865635959481289d-06
 rk( 52)= 0.9601319151840079d-06+(azi- 0.60d+02) * rk( 52)
 rk( 52)=-0.3853692595570321d-03+(azi- 0.60d+02) * rk( 52)
 rk( 52)= 0.1496492000000000d-01+(azi- 0.60d+02) * rk( 52)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 52)=-0.1916508555648358d-06
 rk( 52)=-0.1063677596325682d-04+(azi- 0.70d+02) * rk( 52)
 rk( 52)=-0.4821357000377863d-03+(azi- 0.70d+02) * rk( 52)
 rk( 52)= 0.1082067700000000d-01+(azi- 0.70d+02) * rk( 52)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 52)= 0.2540478756918318d-05
 rk( 52)=-0.1523639649680941d-04+(azi- 0.78d+02) * rk( 52)
 rk( 52)=-0.6891210797183578d-03+(azi- 0.78d+02) * rk( 52)
 rk( 52)= 0.6184712500000000d-02+(azi- 0.78d+02) * rk( 52)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 52)= 0.1651057353835306d-05
 rk( 52)= 0.4573509366928574d-04+(azi- 0.86d+02) * rk( 52)
 rk( 52)=-0.4451315023386027d-03+(azi- 0.86d+02) * rk( 52)
 rk( 52)= 0.9973396100000000d-03+(azi- 0.86d+02) * rk( 52)
elseif(azi.ge.90.d0)then
 rk( 52)= 0.5424277000000000d-04
endif
if(att.lt.0.99999) rk( 52) = rk( 52) * att
endif

rk( 53) =  dexp(-0.2667550967090111d+02                           &
 - ( -0.3650000000000000d+03 )/temp)
rk( 54) =  dexp(-0.2667550967090111d+02                           &
 - ( -0.3650000000000000d+03 )/temp)
rk( 55) =  dexp(-0.2791870318838033d+02                           &
 - ( -0.7000000000000000d+03 )/temp)
rk( 56) =  dexp(-0.2791870318838033d+02                           &
 - ( -0.7000000000000000d+03 )/temp)
rk( 57) =  0.6800000000000000d-13
rk( 58) =  0.6800000000000000d-13
rk( 59) =  0.6800000000000000d-13
rk( 60) =  dexp(-0.2815865385801092d+02                           &
 - (  0.3600000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 61)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 61)=-0.6650705923705479d-11
 rk( 61)=-0.6611929407629450d-09+(azi- 0.00d+00) * rk( 61)
 rk( 61)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 61)
 rk( 61)= 0.3379090000000000d-05+(azi- 0.00d+00) * rk( 61)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 61)= 0.5602117771116236d-11
 rk( 61)=-0.8607141184741090d-09+(azi- 0.10d+02) * rk( 61)
 rk( 61)=-0.1521907059237054d-07+(azi- 0.10d+02) * rk( 61)
 rk( 61)= 0.3306320000000000d-05+(azi- 0.10d+02) * rk( 61)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 61)= 0.2622234839240618d-11
 rk( 61)=-0.6926505853406227d-09+(azi- 0.20d+02) * rk( 61)
 rk( 61)=-0.3075271763051784d-07+(azi- 0.20d+02) * rk( 61)
 rk( 61)= 0.3073660000000000d-05+(azi- 0.20d+02) * rk( 61)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 61)= 0.6018942871922055d-11
 rk( 61)=-0.6139835401634017d-09+(azi- 0.30d+02) * rk( 61)
 rk( 61)=-0.4381905888555809d-07+(azi- 0.30d+02) * rk( 61)
 rk( 61)= 0.2699490000000000d-05+(azi- 0.30d+02) * rk( 61)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 61)= 0.9631993673072422d-11
 rk( 61)=-0.4334152540057500d-09+(azi- 0.40d+02) * rk( 61)
 rk( 61)=-0.5429304682724973d-07+(azi- 0.40d+02) * rk( 61)
 rk( 61)= 0.2205920000000000d-05+(azi- 0.40d+02) * rk( 61)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 61)= 0.1346308243578111d-10
 rk( 61)=-0.1444554438135145d-09+(azi- 0.50d+02) * rk( 61)
 rk( 61)=-0.6007175380544298d-07+(azi- 0.50d+02) * rk( 61)
 rk( 61)= 0.1629280000000000d-05+(azi- 0.50d+02) * rk( 61)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 61)= 0.2754267658383055d-10
 rk( 61)= 0.2594370292596410d-09+(azi- 0.60d+02) * rk( 61)
 rk( 61)=-0.5892193795098066d-07+(azi- 0.60d+02) * rk( 61)
 rk( 61)= 0.1027580000000000d-05+(azi- 0.60d+02) * rk( 61)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 61)= 0.3078605881868256d-11
 rk( 61)= 0.1085717326774769d-08+(azi- 0.70d+02) * rk( 61)
 rk( 61)=-0.4547039439064206d-07+(azi- 0.70d+02) * rk( 61)
 rk( 61)= 0.4918470000000000d-06+(azi- 0.70d+02) * rk( 61)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 61)=-0.2153154079143221d-10
 rk( 61)= 0.1159603867942148d-08+(azi- 0.78d+02) * rk( 61)
 rk( 61)=-0.2750782483291689d-07+(azi- 0.78d+02) * rk( 61)
 rk( 61)= 0.1991460000000000d-06+(azi- 0.78d+02) * rk( 61)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 61)= 0.1655300763808969d-09
 rk( 61)= 0.6428468889409992d-09+(azi- 0.86d+02) * rk( 61)
 rk( 61)=-0.1308821877786526d-07+(azi- 0.86d+02) * rk( 61)
 rk( 61)= 0.4227390000000000d-07+(azi- 0.86d+02) * rk( 61)
elseif(azi.ge.90.d0)then
 rk( 61)= 0.1080050000000000d-07
endif
if(att.lt.0.99999) rk( 61) = rk( 61) * att
endif

rk( 62) =  dexp(-0.2652908103716776d+02                           &
 - ( -0.1900000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 63)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 63)=-0.9639086891432908d-11
 rk( 63)=-0.8144091310856724d-09+(azi- 0.00d+00) * rk( 63)
 rk( 63)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 63)
 rk( 63)= 0.5704620000000000d-05+(azi- 0.00d+00) * rk( 63)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 63)= 0.5287260674299008d-11
 rk( 63)=-0.1103581737828659d-08+(azi- 0.10d+02) * rk( 63)
 rk( 63)=-0.1917990868914332d-07+(azi- 0.10d+02) * rk( 63)
 rk( 63)= 0.5613540000000000d-05+(azi- 0.10d+02) * rk( 63)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 63)=-0.2699558057629897d-12
 rk( 63)=-0.9449639175996891d-09+(azi- 0.20d+02) * rk( 63)
 rk( 63)=-0.3966536524342679d-07+(azi- 0.20d+02) * rk( 63)
 rk( 63)= 0.5316670000000000d-05+(azi- 0.20d+02) * rk( 63)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 63)= 0.2382562548753546d-11
 rk( 63)=-0.9530625917725825d-09+(azi- 0.30d+02) * rk( 63)
 rk( 63)=-0.5864563033714949d-07+(azi- 0.30d+02) * rk( 63)
 rk( 63)= 0.4825250000000000d-05+(azi- 0.30d+02) * rk( 63)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 63)= 0.6319705610747424d-11
 rk( 63)=-0.8815857153099513d-09+(azi- 0.40d+02) * rk( 63)
 rk( 63)=-0.7699211340797505d-07+(azi- 0.40d+02) * rk( 63)
 rk( 63)= 0.4145870000000000d-05+(azi- 0.40d+02) * rk( 63)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 63)= 0.1056861500825653d-10
 rk( 63)=-0.6919945469875881d-09+(azi- 0.50d+02) * rk( 63)
 rk( 63)=-0.9272791603094997d-07+(azi- 0.50d+02) * rk( 63)
 rk( 63)= 0.3294110000000000d-05+(azi- 0.50d+02) * rk( 63)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 63)= 0.4850583435620618d-10
 rk( 63)=-0.3749360967397995d-09+(azi- 0.60d+02) * rk( 63)
 rk( 63)=-0.1033972224682254d-06+(azi- 0.60d+02) * rk( 63)
 rk( 63)= 0.2308200000000000d-05+(azi- 0.60d+02) * rk( 63)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 63)= 0.4863918163365898d-10
 rk( 63)= 0.1080238933948239d-08+(azi- 0.70d+02) * rk( 63)
 rk( 63)=-0.9634419409614782d-07+(azi- 0.70d+02) * rk( 63)
 rk( 63)= 0.1285240000000000d-05+(azi- 0.70d+02) * rk( 63)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 63)=-0.6023611040521017d-10
 rk( 63)= 0.2247579293157748d-08+(azi- 0.78d+02) * rk( 63)
 rk( 63)=-0.6972164827933041d-07+(azi- 0.78d+02) * rk( 63)
 rk( 63)= 0.6085250000000000d-06+(azi- 0.78d+02) * rk( 63)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 63)= 0.8106335758155240d-09
 rk( 63)= 0.8019126434123756d-09+(azi- 0.86d+02) * rk( 63)
 rk( 63)=-0.4532571278666100d-07+(azi- 0.86d+02) * rk( 63)
 rk( 63)= 0.1637560000000000d-06+(azi- 0.86d+02) * rk( 63)
elseif(azi.ge.90.d0)then
 rk( 63)= 0.4716430000000000d-07
endif
if(att.lt.0.99999) rk( 63) = rk( 63) * att
endif

rk( 64) = 1.44d-13 * (1.0d0 + 2.381d-20 *                         &
 8.0d-1 * summ)
rk( 65) =  dexp(-0.2673493309137191d+02                           &
 - (  0.1775000000000000d+04 )/temp)
rk( 66) =  dexp(-0.2660140169874739d+02                           &
 - ( -0.3000000000000000d+03 )/temp)
rk( 67) =  dexp(-0.2852261923521233d+02                           &
 - ( -0.7500000000000000d+03 )/temp)
rk( 68) =  dexp(-0.2998489950331015d+02                           &
 - ( -0.3900000000000000d+03 )/temp)
rk( 69) =  dexp(-0.2629602004919621d+02                           &
 - ( -0.2000000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 70)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 70)=-0.9639086891432908d-11
 rk( 70)=-0.8144091310856724d-09+(azi- 0.00d+00) * rk( 70)
 rk( 70)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 70)
 rk( 70)= 0.5704620000000000d-05+(azi- 0.00d+00) * rk( 70)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 70)= 0.5287260674299008d-11
 rk( 70)=-0.1103581737828659d-08+(azi- 0.10d+02) * rk( 70)
 rk( 70)=-0.1917990868914332d-07+(azi- 0.10d+02) * rk( 70)
 rk( 70)= 0.5613540000000000d-05+(azi- 0.10d+02) * rk( 70)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 70)=-0.2699558057629897d-12
 rk( 70)=-0.9449639175996891d-09+(azi- 0.20d+02) * rk( 70)
 rk( 70)=-0.3966536524342679d-07+(azi- 0.20d+02) * rk( 70)
 rk( 70)= 0.5316670000000000d-05+(azi- 0.20d+02) * rk( 70)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 70)= 0.2382562548753546d-11
 rk( 70)=-0.9530625917725825d-09+(azi- 0.30d+02) * rk( 70)
 rk( 70)=-0.5864563033714949d-07+(azi- 0.30d+02) * rk( 70)
 rk( 70)= 0.4825250000000000d-05+(azi- 0.30d+02) * rk( 70)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 70)= 0.6319705610747424d-11
 rk( 70)=-0.8815857153099513d-09+(azi- 0.40d+02) * rk( 70)
 rk( 70)=-0.7699211340797505d-07+(azi- 0.40d+02) * rk( 70)
 rk( 70)= 0.4145870000000000d-05+(azi- 0.40d+02) * rk( 70)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 70)= 0.1056861500825653d-10
 rk( 70)=-0.6919945469875881d-09+(azi- 0.50d+02) * rk( 70)
 rk( 70)=-0.9272791603094997d-07+(azi- 0.50d+02) * rk( 70)
 rk( 70)= 0.3294110000000000d-05+(azi- 0.50d+02) * rk( 70)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 70)= 0.4850583435620618d-10
 rk( 70)=-0.3749360967397995d-09+(azi- 0.60d+02) * rk( 70)
 rk( 70)=-0.1033972224682254d-06+(azi- 0.60d+02) * rk( 70)
 rk( 70)= 0.2308200000000000d-05+(azi- 0.60d+02) * rk( 70)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 70)= 0.4863918163365898d-10
 rk( 70)= 0.1080238933948239d-08+(azi- 0.70d+02) * rk( 70)
 rk( 70)=-0.9634419409614782d-07+(azi- 0.70d+02) * rk( 70)
 rk( 70)= 0.1285240000000000d-05+(azi- 0.70d+02) * rk( 70)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 70)=-0.6023611040521017d-10
 rk( 70)= 0.2247579293157748d-08+(azi- 0.78d+02) * rk( 70)
 rk( 70)=-0.6972164827933041d-07+(azi- 0.78d+02) * rk( 70)
 rk( 70)= 0.6085250000000000d-06+(azi- 0.78d+02) * rk( 70)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 70)= 0.8106335758155240d-09
 rk( 70)= 0.8019126434123756d-09+(azi- 0.86d+02) * rk( 70)
 rk( 70)=-0.4532571278666100d-07+(azi- 0.86d+02) * rk( 70)
 rk( 70)= 0.1637560000000000d-06+(azi- 0.86d+02) * rk( 70)
elseif(azi.ge.90.d0)then
 rk( 70)= 0.4716430000000000d-07
endif
if(att.lt.0.99999) rk( 70) = rk( 70) * att
endif

rk( 71) =  dexp(-0.2564314676777420d+02                           &
 - (  0.6200000000000000d+03 )/temp)
rk( 72) =  0.9000000000000000d-11

if(option_photolysis.eq.2) then
 rk( 73)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 73)=-0.5816837387219519d-10
 rk( 73)=-0.5184316261278087d-08+(azi- 0.00d+00) * rk( 73)
 rk( 73)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 73)
 rk( 73)= 0.3220560000000000d-04+(azi- 0.00d+00) * rk( 73)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 73)= 0.3450512161659949d-10
 rk( 73)=-0.6929367477443941d-08+(azi- 0.10d+02) * rk( 73)
 rk( 73)=-0.1211368373872203d-06+(azi- 0.10d+02) * rk( 73)
 rk( 73)= 0.3162900000000000d-04+(azi- 0.10d+02) * rk( 73)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 73)= 0.3647887405787883d-11
 rk( 73)=-0.5894213828945960d-08+(azi- 0.20d+02) * rk( 73)
 rk( 73)=-0.2493726504511192d-06+(azi- 0.20d+02) * rk( 73)
 rk( 73)= 0.2975920000000000d-04+(azi- 0.20d+02) * rk( 73)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 73)= 0.2530332876025029d-10
 rk( 73)=-0.5784777206772343d-08+(azi- 0.30d+02) * rk( 73)
 rk( 73)=-0.3661625608083018d-06+(azi- 0.30d+02) * rk( 73)
 rk( 73)= 0.2667970000000000d-04+(azi- 0.30d+02) * rk( 73)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 73)= 0.6013879755321277d-10
 rk( 73)=-0.5025677343964861d-08+(azi- 0.40d+02) * rk( 73)
 rk( 73)=-0.4742671063156733d-06+(azi- 0.40d+02) * rk( 73)
 rk( 73)= 0.2246490000000000d-04+(azi- 0.40d+02) * rk( 73)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 73)= 0.1004414810269593d-09
 rk( 73)=-0.3221513417368319d-08+(azi- 0.50d+02) * rk( 73)
 rk( 73)=-0.5567390139290089d-06+(azi- 0.50d+02) * rk( 73)
 rk( 73)= 0.1727980000000000d-04+(azi- 0.50d+02) * rk( 73)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 73)= 0.3107352783389442d-09
 rk( 73)=-0.2082689865616575d-09+(azi- 0.60d+02) * rk( 73)
 rk( 73)=-0.5910368379682989d-06+(azi- 0.60d+02) * rk( 73)
 rk( 73)= 0.1149070000000000d-04+(azi- 0.60d+02) * rk( 73)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 73)= 0.1345128013872392d-09
 rk( 73)= 0.9113789363612173d-08+(azi- 0.70d+02) * rk( 73)
 rk( 73)=-0.5019816341977565d-06+(azi- 0.70d+02) * rk( 73)
 rk( 73)= 0.5870240000000000d-05+(azi- 0.70d+02) * rk( 73)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 73)=-0.1960445509611151d-09
 rk( 73)= 0.1234209659691269d-07+(azi- 0.78d+02) * rk( 73)
 rk( 73)=-0.3303345465135304d-06+(azi- 0.78d+02) * rk( 73)
 rk( 73)= 0.2506540000000000d-05+(azi- 0.78d+02) * rk( 73)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 73)= 0.2279277828298341d-08
 rk( 73)= 0.7637027373778166d-08+(azi- 0.86d+02) * rk( 73)
 rk( 73)=-0.1705015547476783d-06+(azi- 0.86d+02) * rk( 73)
 rk( 73)= 0.5533830000000000d-06+(azi- 0.86d+02) * rk( 73)
elseif(azi.ge.90.d0)then
 rk( 73)= 0.1394430000000000d-06
endif
if(att.lt.0.99999) rk( 73) = rk( 73) * att
endif


if(option_photolysis.eq.2) then
 rk( 74)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 74)=-0.7261045436751545d-10
 rk( 74)=-0.5576895456324842d-08+(azi- 0.00d+00) * rk( 74)
 rk( 74)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 74)
 rk( 74)= 0.4669460000000000d-04+(azi- 0.00d+00) * rk( 74)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 74)= 0.2853136310254372d-10
 rk( 74)=-0.7755209087350302d-08+(azi- 0.10d+02) * rk( 74)
 rk( 74)=-0.1333210454367515d-06+(azi- 0.10d+02) * rk( 74)
 rk( 74)= 0.4606430000000000d-04+(azi- 0.10d+02) * rk( 74)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 74)=-0.1901499804265726d-10
 rk( 74)=-0.6899268194273995d-08+(azi- 0.20d+02) * rk( 74)
 rk( 74)=-0.2798658182529944d-06+(azi- 0.20d+02) * rk( 74)
 rk( 74)= 0.4398410000000000d-04+(azi- 0.20d+02) * rk( 74)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 74)=-0.9471370931908783d-11
 rk( 74)=-0.7469718135553710d-08+(azi- 0.30d+02) * rk( 74)
 rk( 74)=-0.4235556815512711d-06+(azi- 0.30d+02) * rk( 74)
 rk( 74)= 0.4047650000000000d-04+(azi- 0.30d+02) * rk( 74)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 74)= 0.1440048177028887d-10
 rk( 74)=-0.7753859263510966d-08+(azi- 0.40d+02) * rk( 74)
 rk( 74)=-0.5757914555419174d-06+(azi- 0.40d+02) * rk( 74)
 rk( 74)= 0.3548450000000000d-04+(azi- 0.40d+02) * rk( 74)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 74)= 0.3856944385081257d-10
 rk( 74)=-0.7321844810402538d-08+(azi- 0.50d+02) * rk( 74)
 rk( 74)=-0.7265484962810543d-06+(azi- 0.50d+02) * rk( 74)
 rk( 74)= 0.2896560000000000d-04+(azi- 0.50d+02) * rk( 74)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 74)= 0.4136217428262900d-09
 rk( 74)=-0.6164761494878585d-08+(azi- 0.60d+02) * rk( 74)
 rk( 74)=-0.8614145593338596d-06+(azi- 0.60d+02) * rk( 74)
 rk( 74)= 0.2100650000000000d-04+(azi- 0.60d+02) * rk( 74)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 74)= 0.5376701572532502d-09
 rk( 74)= 0.6243890789914774d-08+(azi- 0.70d+02) * rk( 74)
 rk( 74)=-0.8606232663835604d-06+(azi- 0.70d+02) * rk( 74)
 rk( 74)= 0.1218950000000000d-04+(azi- 0.70d+02) * rk( 74)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 74)=-0.5508273899935273d-09
 rk( 74)= 0.1914797456399955d-07+(azi- 0.78d+02) * rk( 74)
 rk( 74)=-0.6574883435524898d-06+(azi- 0.78d+02) * rk( 74)
 rk( 74)= 0.5979410000000000d-05+(azi- 0.78d+02) * rk( 74)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 74)= 0.8530305661922505d-08
 rk( 74)= 0.5928117204100688d-08+(azi- 0.86d+02) * rk( 74)
 rk( 74)=-0.4568796094072541d-06+(azi- 0.86d+02) * rk( 74)
 rk( 74)= 0.1662950000000000d-05+(azi- 0.86d+02) * rk( 74)
elseif(azi.ge.90.d0)then
 rk( 74)= 0.4762210000000000d-06
endif
if(att.lt.0.99999) rk( 74) = rk( 74) * att
endif

rk( 75) =  dexp(-0.2410466059131239d+02                           &
 - (  0.1600000000000000d+04 )/temp)
rk( 76) =  0.5800000000000000d-15
rk( 77) =  dexp(-0.3226665050940135d+02                           &
 - ( -0.6250000000000000d+03 )/temp)
rk( 78) =  dexp( 0.2850648985328245d+02                           &
 - (  0.7000000000000000d+04 )/temp)
rk( 79) =  0.5600000000000000d-11
rk( 80) =  dexp(-0.3281600979716958d+02                           &
 - ( -0.2300000000000000d+04 )/temp)
rk( 81) =  0.4000000000000000d-12
rk( 82) =  dexp(-0.2474064935803238d+02                           &
 - (  0.1100000000000000d+04 )/temp)
rk( 83) =  dexp(-0.2590825451818744d+02                           &
 - ( -0.2700000000000000d+03 )/temp)
rk( 84) =  dexp(-0.2729454887930734d+02                           &
 - (  0.1900000000000000d+04 )/temp)

if(option_photolysis.eq.2) then
 rk( 85)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 85)=-0.1292568102250478d-10
 rk( 85)=-0.1349143189774952d-08+(azi- 0.00d+00) * rk( 85)
 rk( 85)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 85)
 rk( 85)= 0.6105070000000000d-05+(azi- 0.00d+00) * rk( 85)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 85)= 0.1221704306751377d-10
 rk( 85)=-0.1736913620450095d-08+(azi- 0.10d+02) * rk( 85)
 rk( 85)=-0.3086056810225046d-07+(azi- 0.10d+02) * rk( 85)
 rk( 85)= 0.5957230000000000d-05+(azi- 0.10d+02) * rk( 85)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 85)= 0.7227508752451222d-11
 rk( 85)=-0.1370402328424685d-08+(azi- 0.20d+02) * rk( 85)
 rk( 85)=-0.6193372759099823d-07+(azi- 0.20d+02) * rk( 85)
 rk( 85)= 0.5487150000000000d-05+(azi- 0.20d+02) * rk( 85)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 85)= 0.1521292192268060d-10
 rk( 85)=-0.1153577065851153d-08+(azi- 0.30d+02) * rk( 85)
 rk( 85)=-0.8717352153375648d-07+(azi- 0.30d+02) * rk( 85)
 rk( 85)= 0.4738000000000000d-05+(azi- 0.30d+02) * rk( 85)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 85)= 0.2301080355683006d-10
 rk( 85)=-0.6971894081707449d-09+(azi- 0.40d+02) * rk( 85)
 rk( 85)=-0.1056811862739754d-06+(azi- 0.40d+02) * rk( 85)
 rk( 85)= 0.3766120000000000d-05+(azi- 0.40d+02) * rk( 85)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 85)= 0.3106386384999131d-10
 rk( 85)=-0.6865301465823343d-11+(azi- 0.50d+02) * rk( 85)
 rk( 85)=-0.1127217333703412d-06+(azi- 0.50d+02) * rk( 85)
 rk( 85)= 0.2662600000000000d-05+(azi- 0.50d+02) * rk( 85)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 85)= 0.4531874104319860d-10
 rk( 85)= 0.9250506140339690d-09+(azi- 0.60d+02) * rk( 85)
 rk( 85)=-0.1035398802446585d-06+(azi- 0.60d+02) * rk( 85)
 rk( 85)= 0.1565760000000000d-05+(azi- 0.60d+02) * rk( 85)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 85)=-0.1303761111962380d-10
 rk( 85)= 0.2284612845330880d-08+(azi- 0.70d+02) * rk( 85)
 rk( 85)=-0.7144324565100488d-07+(azi- 0.70d+02) * rk( 85)
 rk( 85)= 0.6681850000000000d-06+(azi- 0.70d+02) * rk( 85)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 85)=-0.4077238229824927d-10
 rk( 85)= 0.1971710178462450d-08+(azi- 0.78d+02) * rk( 85)
 rk( 85)=-0.3739266146067857d-07+(azi- 0.78d+02) * rk( 85)
 rk( 85)= 0.2361790000000000d-06+(azi- 0.78d+02) * rk( 85)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 85)= 0.1193377495821847d-09
 rk( 85)= 0.9931730033112436d-09+(azi- 0.86d+02) * rk( 85)
 rk( 85)=-0.1367359600643481d-07+(azi- 0.86d+02) * rk( 85)
 rk( 85)= 0.4235170000000000d-07+(azi- 0.86d+02) * rk( 85)
elseif(azi.ge.90.d0)then
 rk( 85)= 0.1118570000000000d-07
endif
if(att.lt.0.99999) rk( 85) = rk( 85) * att
endif

rk( 86) =  dexp(-0.2553915705425015d+02                           &
 - ( -0.2700000000000000d+03 )/temp)
effko =  0.2700000000000000d-27* (temp / 3.d2)                    &
           **(- ( 0.7100000000000000d+01))
rapk =  0.1200000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.9000000000000000d+00))
rk( 87) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.3000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.4900000000000000d-02* (temp / 3.d2)                    &
           **(- ( 0.0000000000000000d+00))*                       &
           dexp(- 0.1210000000000000d+05/temp)
rapk =  0.5400000000000000d+17* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))*                      &
           dexp(- 0.1383000000000000d+05/temp)
rk( 88) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.3000d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))

if(option_photolysis.eq.2) then
 rk( 89)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 89)=-0.5245828038559129d-12
 rk( 89)=-0.1048523196144091d-10+(azi- 0.00d+00) * rk( 89)
 rk( 89)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 89)
 rk( 89)= 0.4777587000000000d-07+(azi- 0.00d+00) * rk( 89)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 89)= 0.2522760411567752d-11
 rk( 89)=-0.2622271607711829d-10+(azi- 0.10d+02) * rk( 89)
 rk( 89)=-0.3670794803855920d-09+(azi- 0.10d+02) * rk( 89)
 rk( 89)= 0.4620276400000000d-07+(azi- 0.10d+02) * rk( 89)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 89)=-0.3810250842415100d-11
 rk( 89)= 0.4946009626991426d-10+(azi- 0.20d+02) * rk( 89)
 rk( 89)=-0.1347056784576322d-09+(azi- 0.20d+02) * rk( 89)
 rk( 89)= 0.4243245800000000d-07+(azi- 0.20d+02) * rk( 89)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 89)= 0.2339466958092650d-11
 rk( 89)=-0.6484742900253869d-10+(azi- 0.30d+02) * rk( 89)
 rk( 89)=-0.2885790057838769d-09+(azi- 0.30d+02) * rk( 89)
 rk( 89)= 0.4222116000000000d-07+(azi- 0.30d+02) * rk( 89)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 89)= 0.1867501004448772d-13
 rk( 89)= 0.5336579740240855d-11+(azi- 0.40d+02) * rk( 89)
 rk( 89)=-0.8836874984068571d-09+(azi- 0.40d+02) * rk( 89)
 rk( 89)= 0.3519009400000000d-07+(azi- 0.40d+02) * rk( 89)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 89)= 0.8226700172943887d-13
 rk( 89)= 0.5896830041575641d-11+(azi- 0.50d+02) * rk( 89)
 rk( 89)=-0.7713534005886921d-09+(azi- 0.50d+02) * rk( 89)
 rk( 89)= 0.2690555200000000d-07+(azi- 0.50d+02) * rk( 89)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 89)=-0.9853030169627836d-12
 rk( 89)= 0.8364840093458807d-11+(azi- 0.60d+02) * rk( 89)
 rk( 89)=-0.6287366992383443d-09+(azi- 0.60d+02) * rk( 89)
 rk( 89)= 0.1986396800000000d-07+(azi- 0.60d+02) * rk( 89)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 89)= 0.1136116301266380d-11
 rk( 89)=-0.2119425041541064d-10+(azi- 0.70d+02) * rk( 89)
 rk( 89)=-0.7570308024579321d-09+(azi- 0.70d+02) * rk( 89)
 rk( 89)= 0.1342778200000000d-07+(azi- 0.70d+02) * rk( 89)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 89)= 0.1853495648302295d-11
 rk( 89)= 0.6072540815002347d-11+(azi- 0.78d+02) * rk( 89)
 rk( 89)=-0.8780044792615161d-09+(azi- 0.78d+02) * rk( 89)
 rk( 89)= 0.6596795100000000d-08+(azi- 0.78d+02) * rk( 89)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 89)= 0.4275243906997995d-12
 rk( 89)= 0.5055643637423095d-10+(azi- 0.86d+02) * rk( 89)
 rk( 89)=-0.4249726617465909d-09+(azi- 0.86d+02) * rk( 89)
 rk( 89)= 0.9103916500000000d-09+(azi- 0.86d+02) * rk( 89)
elseif(azi.ge.90.d0)then
 rk( 89)= 0.4676554600000000d-10
endif
if(att.lt.0.99999) rk( 89) = rk( 89) * att
endif

rk( 90) =  dexp(-0.2847499118622308d+02                           &
 - ( -0.1040000000000000d+04 )/temp)
rk( 91) =  dexp(-0.2693787393536860d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk( 92) =  dexp(-0.2845200166799838d+02                           &
 - ( -0.1070000000000000d+04 )/temp)
rk( 93) =  dexp(-0.2656631037893612d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk( 94) =  dexp(-0.2854731184780270d+02                           &
 - ( -0.2000000000000000d+03 )/temp)

if(option_photolysis.eq.2) then
 rk( 95)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk( 95)= 0.3439467845737435d-11
 rk( 95)=-0.1604640139173746d-09+(azi- 0.00d+00) * rk( 95)
 rk( 95)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk( 95)
 rk( 95)= 0.6899761847520000d-06+(azi- 0.00d+00) * rk( 95)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk( 95)=-0.1682071517212295d-11
 rk( 95)=-0.5727997854525157d-10+(azi- 0.10d+02) * rk( 95)
 rk( 95)=-0.2177439924626261d-08+(azi- 0.10d+02) * rk( 95)
 rk( 95)= 0.6773692512060000d-06+(azi- 0.10d+02) * rk( 95)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk( 95)=-0.1770967004888075d-11
 rk( 95)=-0.1077421240616206d-09+(azi- 0.20d+02) * rk( 95)
 rk( 95)=-0.3827660950694980d-08+(azi- 0.20d+02) * rk( 95)
 rk( 95)= 0.6481847825880000d-06+(azi- 0.20d+02) * rk( 95)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk( 95)= 0.5197719474764733d-11
 rk( 95)=-0.1608711342082635d-09+(azi- 0.30d+02) * rk( 95)
 rk( 95)=-0.6513793533393824d-08+(azi- 0.30d+02) * rk( 95)
 rk( 95)= 0.5973629936700000d-06+(azi- 0.30d+02) * rk( 95)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk( 95)=-0.1925844006171251d-11
 rk( 95)=-0.4939549965316779d-11+(azi- 0.40d+02) * rk( 95)
 rk( 95)=-0.8171900375129673d-08+(azi- 0.40d+02) * rk( 95)
 rk( 95)= 0.5213356643900000d-06+(azi- 0.40d+02) * rk( 95)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk( 95)= 0.3659679272920429d-11
 rk( 95)=-0.6271487015045512d-10+(azi- 0.50d+02) * rk( 95)
 rk( 95)=-0.8848444576287439d-08+(azi- 0.50d+02) * rk( 95)
 rk( 95)= 0.4371968616360000d-06+(azi- 0.50d+02) * rk( 95)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk( 95)=-0.1519631773950880d-10
 rk( 95)= 0.4707550803713459d-10+(azi- 0.60d+02) * rk( 95)
 rk( 95)=-0.9004838197420313d-08+(azi- 0.60d+02) * rk( 95)
 rk( 95)= 0.3461006081310000d-06+(azi- 0.60d+02) * rk( 95)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk( 95)= 0.7012506971232375d-10
 rk( 95)=-0.4088140241481955d-09+(azi- 0.70d+02) * rk( 95)
 rk( 95)=-0.1262222335853076d-07+(azi- 0.70d+02) * rk( 95)
 rk( 95)= 0.2455634592210000d-06+(azi- 0.70d+02) * rk( 95)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk( 95)=-0.1236221759603351d-09
 rk( 95)= 0.1274187648948739d-08+(azi- 0.78d+02) * rk( 95)
 rk( 95)=-0.5699234360131495d-08+(azi- 0.78d+02) * rk( 95)
 rk( 95)= 0.1543256105000000d-06+(azi- 0.78d+02) * rk( 95)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk( 95)= 0.4706176323768702d-09
 rk( 95)=-0.1692744574097185d-08+(azi- 0.86d+02) * rk( 95)
 rk( 95)=-0.9047689761312286d-08+(azi- 0.86d+02) * rk( 95)
 rk( 95)= 0.1269851910600000d-06+(azi- 0.86d+02) * rk( 95)
elseif(azi.ge.90.d0)then
 rk( 95)= 0.9383004730129999d-07
endif
if(att.lt.0.99999) rk( 95) = rk( 95) * att
endif

rk( 96) =  dexp(-0.2854731184780270d+02                           &
 - ( -0.2000000000000000d+03 )/temp)
rk( 97) =  dexp(-0.2506607175846701d+02                           &
 - (  0.8700000000000000d+03 )/temp)
rk( 98) =  dexp(-0.2600178057619827d+02                           &
 - ( -0.4050000000000000d+03 )/temp)
rk( 99) =  0.6500000000000000d-14

if(option_photolysis.eq.2) then
 rk(100)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(100)=-0.2543209447272390d-06
 rk(100)=-0.4665430552727591d-05+(azi- 0.00d+00) * rk(100)
 rk(100)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(100)
 rk(100)= 0.2672443200000000d-01+(azi- 0.00d+00) * rk(100)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(100)= 0.1160138834181708d-05
 rk(100)=-0.1229505889454476d-04+(azi- 0.10d+02) * rk(100)
 rk(100)=-0.1696048944727235d-03+(azi- 0.10d+02) * rk(100)
 rk(100)= 0.2600356800000000d-01+(azi- 0.10d+02) * rk(100)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(100)=-0.1797915391999585d-05
 rk(100)= 0.2250910613090648d-04+(azi- 0.20d+02) * rk(100)
 rk(100)=-0.6746442210910607d-04+(azi- 0.20d+02) * rk(100)
 rk(100)= 0.2423815200000000d-01+(azi- 0.20d+02) * rk(100)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(100)= 0.1113225733816626d-05
 rk(100)=-0.3142835562908105d-04+(azi- 0.30d+02) * rk(100)
 rk(100)=-0.1566569170908517d-03+(azi- 0.30d+02) * rk(100)
 rk(100)= 0.2401650300000000d-01+(azi- 0.30d+02) * rk(100)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(100)=-0.4892354326693738d-07
 rk(100)= 0.1968416385417769d-05+(azi- 0.40d+02) * rk(100)
 rk(100)=-0.4512563095274845d-03+(azi- 0.40d+02) * rk(100)
 rk(100)= 0.2042032400000000d-01+(azi- 0.40d+02) * rk(100)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(100)= 0.1418524392511286d-06
 rk(100)= 0.5007100874097285d-06+(azi- 0.50d+02) * rk(100)
 rk(100)=-0.4265650447992115d-03+(azi- 0.50d+02) * rk(100)
 rk(100)= 0.1605567900000000d-01+(azi- 0.50d+02) * rk(100)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(100)=-0.6037011137374412d-06
 rk(100)= 0.4756283264942610d-05+(azi- 0.60d+02) * rk(100)
 rk(100)=-0.3739951112756851d-03+(azi- 0.60d+02) * rk(100)
 rk(100)= 0.1198195200000000d-01+(azi- 0.60d+02) * rk(100)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(100)= 0.7324219730526701d-06
 rk(100)=-0.1335475014717716d-04+(azi- 0.70d+02) * rk(100)
 rk(100)=-0.4599797800980132d-03+(azi- 0.70d+02) * rk(100)
 rk(100)= 0.8113928100000001d-02+(azi- 0.70d+02) * rk(100)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(100)= 0.1115677554341943d-05
 rk(100)= 0.4223377206111212d-05+(azi- 0.78d+02) * rk(100)
 rk(100)=-0.5330307636267073d-03+(azi- 0.78d+02) * rk(100)
 rk(100)= 0.3954385900000000d-02+(azi- 0.78d+02) * rk(100)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(100)= 0.6769853777566226d-07
 rk(100)= 0.3099963851027621d-04+(azi- 0.86d+02) * rk(100)
 rk(100)=-0.2512466378952194d-03+(azi- 0.86d+02) * rk(100)
 rk(100)= 0.5316628400000000d-03+(azi- 0.86d+02) * rk(100)
elseif(azi.ge.90.d0)then
 rk(100)= 0.2700321100000000d-04
endif
if(att.lt.0.99999) rk(100) = rk(100) * att
endif

rk(101) =  dexp(-0.2572891358953163d+02                           &
 - ( -0.3400000000000000d+03 )/temp)
effko =  0.2700000000000000d-27* (temp / 3.d2)                    &
           **(- ( 0.7100000000000000d+01))
rapk =  0.1200000000000000d-10* (temp / 3.d2)                     &
            **(- ( 0.9000000000000000d+00))
rk(102) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.3000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
effko =  0.4900000000000000d-02* (temp / 3.d2)                    &
           **(- ( 0.0000000000000000d+00))*                       &
           dexp(- 0.1210000000000000d+05/temp)
rapk =  0.5400000000000000d+17* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))*                      &
           dexp(- 0.1383000000000000d+05/temp)
rk(103) = (effko*summ / ( 1.0d0 + effko * summ /                  &
         rapk)) * 0.3000d+00** (1.0d0 / (1.0d0 +                  &
          (dlog10(effko * summ / rapk))**2))

if(option_photolysis.eq.2) then
 rk(104)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(104)=-0.5245828038559129d-12
 rk(104)=-0.1048523196144091d-10+(azi- 0.00d+00) * rk(104)
 rk(104)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(104)
 rk(104)= 0.4777587000000000d-07+(azi- 0.00d+00) * rk(104)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(104)= 0.2522760411567752d-11
 rk(104)=-0.2622271607711829d-10+(azi- 0.10d+02) * rk(104)
 rk(104)=-0.3670794803855920d-09+(azi- 0.10d+02) * rk(104)
 rk(104)= 0.4620276400000000d-07+(azi- 0.10d+02) * rk(104)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(104)=-0.3810250842415100d-11
 rk(104)= 0.4946009626991426d-10+(azi- 0.20d+02) * rk(104)
 rk(104)=-0.1347056784576322d-09+(azi- 0.20d+02) * rk(104)
 rk(104)= 0.4243245800000000d-07+(azi- 0.20d+02) * rk(104)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(104)= 0.2339466958092650d-11
 rk(104)=-0.6484742900253869d-10+(azi- 0.30d+02) * rk(104)
 rk(104)=-0.2885790057838769d-09+(azi- 0.30d+02) * rk(104)
 rk(104)= 0.4222116000000000d-07+(azi- 0.30d+02) * rk(104)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(104)= 0.1867501004448772d-13
 rk(104)= 0.5336579740240855d-11+(azi- 0.40d+02) * rk(104)
 rk(104)=-0.8836874984068571d-09+(azi- 0.40d+02) * rk(104)
 rk(104)= 0.3519009400000000d-07+(azi- 0.40d+02) * rk(104)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(104)= 0.8226700172943887d-13
 rk(104)= 0.5896830041575641d-11+(azi- 0.50d+02) * rk(104)
 rk(104)=-0.7713534005886921d-09+(azi- 0.50d+02) * rk(104)
 rk(104)= 0.2690555200000000d-07+(azi- 0.50d+02) * rk(104)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(104)=-0.9853030169627836d-12
 rk(104)= 0.8364840093458807d-11+(azi- 0.60d+02) * rk(104)
 rk(104)=-0.6287366992383443d-09+(azi- 0.60d+02) * rk(104)
 rk(104)= 0.1986396800000000d-07+(azi- 0.60d+02) * rk(104)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(104)= 0.1136116301266380d-11
 rk(104)=-0.2119425041541064d-10+(azi- 0.70d+02) * rk(104)
 rk(104)=-0.7570308024579321d-09+(azi- 0.70d+02) * rk(104)
 rk(104)= 0.1342778200000000d-07+(azi- 0.70d+02) * rk(104)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(104)= 0.1853495648302295d-11
 rk(104)= 0.6072540815002347d-11+(azi- 0.78d+02) * rk(104)
 rk(104)=-0.8780044792615161d-09+(azi- 0.78d+02) * rk(104)
 rk(104)= 0.6596795100000000d-08+(azi- 0.78d+02) * rk(104)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(104)= 0.4275243906997995d-12
 rk(104)= 0.5055643637423095d-10+(azi- 0.86d+02) * rk(104)
 rk(104)=-0.4249726617465909d-09+(azi- 0.86d+02) * rk(104)
 rk(104)= 0.9103916500000000d-09+(azi- 0.86d+02) * rk(104)
elseif(azi.ge.90.d0)then
 rk(104)= 0.4676554600000000d-10
endif
if(att.lt.0.99999) rk(104) = rk(104) * att
endif

rk(105) =  0.3000000000000000d-12
rk(106) =  dexp(-0.2847499118622308d+02                           &
 - ( -0.1040000000000000d+04 )/temp)
rk(107) =  dexp(-0.2693787393536860d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk(108) =  dexp(-0.2845200166799838d+02                           &
 - ( -0.1070000000000000d+04 )/temp)
rk(109) =  dexp(-0.2656631037893612d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk(110) =  dexp(-0.2656631037893612d+02                           &
 - ( -0.5000000000000000d+03 )/temp)
rk(111) =  0.8100000000000000d-12
rk(112) =  dexp( 0.3453877639491068d+02                           &
 - (  0.8000000000000000d+04 )/temp)
rk(113) =  0.1600000000000000d+04
rk(114) =  0.1500000000000000d-10
rk(115) =  dexp(-0.2531848569208134d+02                           &
 - (  0.2800000000000000d+03 )/temp)
rk(116) =  0.3200000000000000d-10
rk(117) =  dexp(-0.3266697421800909d+02                           &
 - (  0.1900000000000000d+04 )/temp)
rk(118) =  dexp(-0.2798769605986728d+02                           &
 - (  0.2160000000000000d+04 )/temp)
rk(119) =  dexp(-0.2499196378631329d+02                           &
 - (  0.7920000000000000d+03 )/temp)
effko =  0.1000000000000000d-27* (temp / 3.d2)                    &
           **(- ( 0.8000000000000000d+00))
rapk =  0.8800000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk(120) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk(121) =  dexp(-0.3205386974512268d+02                           &
 - (  0.2630000000000000d+04 )/temp)
rk(122) =  dexp(-0.2643709864745611d+02                           &
 - (  0.2880000000000000d+04 )/temp)
rk(123) =  0.2300000000000000d-10
rk(124) =  dexp(-0.2532843602293450d+02                           &
 - ( -0.5500000000000000d+03 )/temp)
rk(125) =  dexp(-0.3241054468906142d+02                           &
 - (  0.1100000000000000d+04 )/temp)
rk(126) =  dexp(-0.2767184311044880d+02                           &
 - (  0.2700000000000000d+03 )/temp)
rk(127) =  dexp(-0.2704323445102643d+02                           &
 - ( -0.3550000000000000d+03 )/temp)
rk(128) =  0.8100000000000000d-11
rk(129) =  0.4200000000000000d+01
rk(130) =  0.4100000000000000d-10
rk(131) =  0.2200000000000000d-10
rk(132) =  0.1400000000000000d-10
rk(133) =  0.5500000000000000d-11

if(option_photolysis.eq.2) then
 rk(134)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(134)=-0.8584948998100693d-09
 rk(134)=-0.6414505100189953d-07+(azi- 0.00d+00) * rk(134)
 rk(134)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(134)
 rk(134)= 0.5609260000000000d-03+(azi- 0.00d+00) * rk(134)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(134)= 0.3254846994303392d-09
 rk(134)=-0.8989989799620162d-07+(azi- 0.10d+02) * rk(134)
 rk(134)=-0.1540449489981011d-05+(azi- 0.10d+02) * rk(134)
 rk(134)= 0.5536530000000000d-03+(azi- 0.10d+02) * rk(134)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(134)=-0.2454438979115351d-09
 rk(134)=-0.8013535701329145d-07+(azi- 0.20d+02) * rk(134)
 rk(134)=-0.3240802040075941d-05+(azi- 0.20d+02) * rk(134)
 rk(134)= 0.5295840000000000d-03+(azi- 0.20d+02) * rk(134)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(134)=-0.1557091077839579d-09
 rk(134)=-0.8749867395063711d-07+(azi- 0.30d+02) * rk(134)
 rk(134)=-0.4917142349715225d-05+(azi- 0.30d+02) * rk(134)
 rk(134)= 0.4889170000000000d-03+(azi- 0.30d+02) * rk(134)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(134)= 0.5928032904741575d-10
 rk(134)=-0.9216994718415690d-07+(azi- 0.40d+02) * rk(134)
 rk(134)=-0.6713828561063159d-05+(azi- 0.40d+02) * rk(134)
 rk(134)= 0.4308400000000000d-03+(azi- 0.40d+02) * rk(134)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(134)= 0.2595877915944963d-09
 rk(134)=-0.9039153731273486d-07+(azi- 0.50d+02) * rk(134)
 rk(134)=-0.8539443406032087d-05+(azi- 0.50d+02) * rk(134)
 rk(134)= 0.3545440000000000d-03+(azi- 0.50d+02) * rk(134)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(134)= 0.4590368504578084d-08
 rk(134)=-0.8260390356490844d-07+(azi- 0.60d+02) * rk(134)
 rk(134)=-0.1026939781480858d-04+(azi- 0.60d+02) * rk(134)
 rk(134)= 0.2603700000000000d-03+(azi- 0.60d+02) * rk(134)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(134)= 0.7530009721131077d-08
 rk(134)= 0.5510715157249846d-07+(azi- 0.70d+02) * rk(134)
 rk(134)=-0.1054436533473309d-04+(azi- 0.70d+02) * rk(134)
 rk(134)= 0.1540060000000000d-03+(azi- 0.70d+02) * rk(134)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(134)=-0.7917656811272118d-08
 rk(134)= 0.2358273848796985d-06+(azi- 0.78d+02) * rk(134)
 rk(134)=-0.8216889043116161d-05+(azi- 0.78d+02) * rk(134)
 rk(134)= 0.7703330000000000d-04+(azi- 0.78d+02) * rk(134)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(134)= 0.1166127504485630d-06
 rk(134)= 0.4580362140873401d-07+(azi- 0.86d+02) * rk(134)
 rk(134)=-0.5963840992804365d-05+(azi- 0.86d+02) * rk(134)
 rk(134)= 0.2233730000000000d-04+(azi- 0.86d+02) * rk(134)
elseif(azi.ge.90.d0)then
 rk(134)= 0.6678010000000000d-05
endif
if(att.lt.0.99999) rk(134) = rk(134) * att
endif

rk(135) =  0.3000000000000000d-10
rk(136) =  dexp(-0.3745754762732854d+02                           &
 - (  0.5000000000000000d+03 )/temp)
rk(137) =  dexp(-0.2479780777187233d+02                           &
 - ( -0.1160000000000000d+03 )/temp)
rk(138) =  0.1700000000000000d-10

if(option_photolysis.eq.2) then
 rk(139)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(139)=-0.1826021023809213d-09
 rk(139)=-0.1228397897619082d-07+(azi- 0.00d+00) * rk(139)
 rk(139)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(139)
 rk(139)= 0.1204770000000000d-03+(azi- 0.00d+00) * rk(139)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(139)= 0.7680630714277358d-10
 rk(139)=-0.1776204204761847d-07+(azi- 0.10d+02) * rk(139)
 rk(139)=-0.3004602102380929d-06+(azi- 0.10d+02) * rk(139)
 rk(139)= 0.1190660000000000d-03+(azi- 0.10d+02) * rk(139)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(139)=-0.4662312619017368d-10
 rk(139)=-0.1545785283333527d-07+(azi- 0.20d+02) * rk(139)
 rk(139)=-0.6326591590476302d-06+(azi- 0.20d+02) * rk(139)
 rk(139)= 0.1143620000000000d-03+(azi- 0.20d+02) * rk(139)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(139)=-0.3391380238206294d-10
 rk(139)=-0.1685654661904045d-07+(azi- 0.30d+02) * rk(139)
 rk(139)=-0.9558031535713878d-06+(azi- 0.30d+02) * rk(139)
 rk(139)= 0.1064430000000000d-03+(azi- 0.30d+02) * rk(139)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(139)=-0.3482166428159112d-10
 rk(139)=-0.1787396069050238d-07+(azi- 0.40d+02) * rk(139)
 rk(139)=-0.1303108226666818d-05+(azi- 0.40d+02) * rk(139)
 rk(139)= 0.9516540000000000d-04+(azi- 0.40d+02) * rk(139)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(139)=-0.9419954049165213d-10
 rk(139)=-0.1891861061895016d-07+(azi- 0.50d+02) * rk(139)
 rk(139)=-0.1671033939761339d-05+(azi- 0.50d+02) * rk(139)
 rk(139)= 0.8031210000000001d-04+(azi- 0.50d+02) * rk(139)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(139)= 0.4307198262473663d-09
 rk(139)=-0.2174459683369613d-07+(azi- 0.60d+02) * rk(139)
 rk(139)=-0.2077666014287802d-05+(azi- 0.60d+02) * rk(139)
 rk(139)= 0.6161569999999999d-04+(azi- 0.60d+02) * rk(139)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(139)= 0.1932828429018806d-08
 rk(139)=-0.8823002046250573d-08+(azi- 0.70d+02) * rk(139)
 rk(139)=-0.2383342003087316d-05+(azi- 0.70d+02) * rk(139)
 rk(139)= 0.3909530000000000d-04+(azi- 0.70d+02) * rk(139)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(139)= 0.7078805396671869d-10
 rk(139)= 0.3756488025025498d-07+(azi- 0.78d+02) * rk(139)
 rk(139)=-0.2153406977455823d-05+(azi- 0.78d+02) * rk(139)
 rk(139)= 0.2045350000000000d-04+(azi- 0.78d+02) * rk(139)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(139)= 0.2551390080685143d-07
 rk(139)= 0.3926379354523939d-07+(azi- 0.86d+02) * rk(139)
 rk(139)=-0.1538777587090784d-05+(azi- 0.86d+02) * rk(139)
 rk(139)= 0.5666640000000000d-05+(azi- 0.86d+02) * rk(139)
elseif(azi.ge.90.d0)then
 rk(139)= 0.1772640000000000d-05
endif
if(att.lt.0.99999) rk(139) = rk(139) * att
endif

rk(140) =  0.3600000000000000d-10
rk(141) =  dexp(-0.2439627194190406d+02                           &
 - ( -0.4076000000000000d+03 )/temp)
rk(142) =  dexp(-0.3247698978846957d+02                           &
 - (  0.1912000000000000d+04 )/temp)
rk(143) =  dexp(-0.2652245849640727d+02                           &
 - (  0.4480000000000000d+03 )/temp)
rk(144) =  0.3360000000000000d-10
rk(145) =  0.7100000000000001d-17
rk(146) =  0.1000000000000000d-14

if(option_photolysis.eq.2) then
 rk(147)= 0.d0
elseif(option_photolysis.eq.1) then
if(azi.lt.0.d0)then
 stop
elseif(azi.ge. 0.00d+00 .and. azi.lt. 0.10d+02) then
 rk(147)=-0.2724056225766411d-11
 rk(147)=-0.2068594377423350d-09+(azi- 0.00d+00) * rk(147)
 rk(147)= 0.0000000000000000d+00+(azi- 0.00d+00) * rk(147)
 rk(147)= 0.1769770000000000d-05+(azi- 0.00d+00) * rk(147)
elseif(azi.ge. 0.10d+02 .and. azi.lt. 0.20d+02) then
 rk(147)= 0.1042168677298912d-11
 rk(147)=-0.2885811245153273d-09+(azi- 0.10d+02) * rk(147)
 rk(147)=-0.4954405622576623d-08+(azi- 0.10d+02) * rk(147)
 rk(147)= 0.1746360000000000d-05+(azi- 0.10d+02) * rk(147)
elseif(azi.ge. 0.20d+02 .and. azi.lt. 0.30d+02) then
 rk(147)=-0.7346184834290850d-12
 rk(147)=-0.2573160641963604d-09+(azi- 0.20d+02) * rk(147)
 rk(147)=-0.1041337750969350d-07+(azi- 0.20d+02) * rk(147)
 rk(147)= 0.1669000000000000d-05+(azi- 0.20d+02) * rk(147)
elseif(azi.ge. 0.30d+02 .and. azi.lt. 0.40d+02) then
 rk(147)=-0.4136947435822000d-12
 rk(147)=-0.2793546186992334d-09+(azi- 0.30d+02) * rk(147)
 rk(147)=-0.1578008433864942d-07+(azi- 0.30d+02) * rk(147)
 rk(147)= 0.1538400000000000d-05+(azi- 0.30d+02) * rk(147)
elseif(azi.ge. 0.40d+02 .and. azi.lt. 0.50d+02) then
 rk(147)= 0.3593974577559286d-12
 rk(147)=-0.2917654610066926d-09+(azi- 0.40d+02) * rk(147)
 rk(147)=-0.2149128513570870d-07+(azi- 0.40d+02) * rk(147)
 rk(147)= 0.1352250000000000d-05+(azi- 0.40d+02) * rk(147)
elseif(azi.ge. 0.50d+02 .and. azi.lt. 0.60d+02) then
 rk(147)= 0.1158104912559730d-11
 rk(147)=-0.2809835372740098d-09+(azi- 0.50d+02) * rk(147)
 rk(147)=-0.2721877511851595d-07+(azi- 0.50d+02) * rk(147)
 rk(147)= 0.1108520000000000d-05+(azi- 0.50d+02) * rk(147)
elseif(azi.ge. 0.60d+02 .and. azi.lt. 0.70d+02) then
 rk(147)= 0.1481218289200588d-10
 rk(147)=-0.2462403898971914d-09+(azi- 0.60d+02) * rk(147)
 rk(147)=-0.3249101439022863d-07+(azi- 0.60d+02) * rk(147)
 rk(147)= 0.8093920000000000d-06+(azi- 0.60d+02) * rk(147)
elseif(azi.ge. 0.70d+02 .and. azi.lt. 0.78d+02) then
 rk(147)= 0.2251236790098139d-10
 rk(147)= 0.1981250968631836d-09+(azi- 0.70d+02) * rk(147)
 rk(147)=-0.3297216732056966d-07+(azi- 0.70d+02) * rk(147)
 rk(147)= 0.4746700000000000d-06+(azi- 0.70d+02) * rk(147)
elseif(azi.ge. 0.78d+02 .and. azi.lt. 0.86d+02) then
 rk(147)=-0.2282006684570455d-10
 rk(147)= 0.7384219264865253d-09+(azi- 0.78d+02) * rk(147)
 rk(147)=-0.2547979113377114d-07+(azi- 0.78d+02) * rk(147)
 rk(147)= 0.2350990000000000d-06+(azi- 0.78d+02) * rk(147)
elseif(azi.ge. 0.86d+02 .and. azi.lt. 0.90d+02) then
 rk(147)= 0.3441785534773510d-09
 rk(147)= 0.1907403221845338d-09+(azi- 0.86d+02) * rk(147)
 rk(147)=-0.1804649314436201d-07+(azi- 0.86d+02) * rk(147)
 rk(147)= 0.6683580000000001d-07+(azi- 0.86d+02) * rk(147)
elseif(azi.ge.90.d0)then
 rk(147)= 0.1972910000000000d-07
endif
if(att.lt.0.99999) rk(147) = rk(147) * att
endif

rk(148) =  0.3600000000000000d-10
rk(149) =  dexp(-0.2492297091482634d+02                           &
 - ( -0.4490000000000000d+03 )/temp)
rk(150) =  dexp(-0.3435645483811673d+02                           &
 - (  0.8210000000000000d+03 )/temp)
rk(151) =  dexp(-0.2632268829627837d+02                           &
 - ( -0.1750000000000000d+03 )/temp)
effko =  0.3000000000000000d-30* (temp / 3.d2)                    &
           **(- ( 0.3300000000000000d+01))
rapk =  0.1500000000000000d-11* (temp / 3.d2)                     &
            **(- ( 0.0000000000000000d+00))
rk(152) = (effko * summ /                                         &
            ( 1.0d0 + effko * summ / rapk)) *                     &
            0.6000d+00** (1.0d0 / (1.0d0 +                        &
             (dlog10(effko * summ / rapk))**2))
rk(153) =  dexp(-0.2569949970432533d+02                           &
 - (  0.2300000000000000d+03 )/temp)
rk(154) =  dexp(-0.2546769809026801d+02                           &
 - (  0.1070000000000000d+04 )/temp)
rk(155) =  0.1500000000000000d-18

return
end subroutine kinetic_3


!===============================================================================

!> fexchem_3
!> \brief Computation of the chemical production terms
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name              role
!------------------------------------------------------------------------------
!> \param[in]     ns                total number of chemical species
!> \param[in]     nr                total number of chemical reactions
!> \param[in]     y                 concentrations vector
!> \param[in]     rk                kinetic rates
!> \param[in]     zcsourc           source term
!> \param[in]     convers_factor    conversion factors
!> \param[out]    chem              chemical production terms for every species
!______________________________________________________________________________

subroutine fexchem_3(ns,nr,y,rk,zcsourc,convers_factor,chem)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns),chem(ns),zcsourc(ns)
double precision convers_factor(ns)

! Local variables

integer i
double precision w(nr)
double precision conc(ns)

do i=1,ns
 chem(i)=0.d0
enddo

!     Conversion mug/m3 to molecules/cm3.

do i = 1, ns
   conc(i) = y(i) * convers_factor(i)
enddo

!     Compute reaction rates.

call rates_3(ns,nr,rk,conc,w)

!     Chemical production terms.

chem( 43) = chem( 43) + w(  1)
chem( 48) = chem( 48) + w(  1)
chem( 52) = chem( 52) - w(  1)
chem( 43) = chem( 43) - w(  2)
chem( 47) = chem( 47) + w(  2)
chem( 47) = chem( 47) - w(  3)
chem( 48) = chem( 48) - w(  3)
chem( 52) = chem( 52) + w(  3)
chem( 43) = chem( 43) - w(  4)
chem( 48) = chem( 48) + w(  4)
chem( 52) = chem( 52) - w(  4)
chem( 43) = chem( 43) - w(  5)
chem( 44) = chem( 44) + w(  5)
chem( 52) = chem( 52) - w(  5)
chem( 43) = chem( 43) - w(  6)
chem( 48) = chem( 48) - w(  6)
chem( 52) = chem( 52) + w(  6)
chem( 44) = chem( 44) + w(  7)
chem( 47) = chem( 47) - w(  7)
chem( 52) = chem( 52) - w(  7)
chem( 43) = chem( 43) + w(  8)
chem( 47) = chem( 47) - w(  8)
chem(  1) = chem(  1) + w(  9)
chem( 47) = chem( 47) - w(  9)
chem(  1) = chem(  1) - w( 10)
chem( 43) = chem( 43) + w( 10)
chem(  1) = chem(  1) - w( 11)
chem( 42) = chem( 42) +  0.2000000000000000d+01 * w( 11)
chem( 42) = chem( 42) - w( 12)
chem( 47) = chem( 47) - w( 12)
chem( 50) = chem( 50) + w( 12)
chem( 42) = chem( 42) + w( 13)
chem( 47) = chem( 47) - w( 13)
chem( 50) = chem( 50) - w( 13)
chem( 43) = chem( 43) + w( 14)
chem( 44) = chem( 44) - w( 14)
chem( 52) = chem( 52) + w( 14)
chem( 44) = chem( 44) - w( 15)
chem( 48) = chem( 48) + w( 15)
chem( 44) = chem( 44) - w( 16)
chem( 48) = chem( 48) - w( 16)
chem( 52) = chem( 52) +  0.2000000000000000d+01 * w( 16)
chem( 44) = chem( 44) - w( 17)
chem( 48) = chem( 48) + w( 17)
chem(  9) = chem(  9) + w( 18)
chem( 44) = chem( 44) - w( 18)
chem( 52) = chem( 52) - w( 18)
chem(  9) = chem(  9) - w( 19)
chem( 38) = chem( 38) +  0.2000000000000000d+01 * w( 19)
chem(  9) = chem(  9) - w( 20)
chem( 44) = chem( 44) + w( 20)
chem( 52) = chem( 52) + w( 20)
chem( 48) = chem( 48) -  0.2000000000000000d+01 * w( 21)
chem( 52) = chem( 52) +  0.2000000000000000d+01 * w( 21)
chem( 15) = chem( 15) +  0.2000000000000000d+01 * w( 22)
chem( 48) = chem( 48) - w( 22)
chem( 52) = chem( 52) - w( 22)
chem( 15) = chem( 15) + w( 23)
chem( 42) = chem( 42) - w( 23)
chem( 48) = chem( 48) - w( 23)
chem( 15) = chem( 15) - w( 24)
chem( 42) = chem( 42) + w( 24)
chem( 48) = chem( 48) + w( 24)
chem( 15) = chem( 15) - w( 25)
chem( 42) = chem( 42) - w( 25)
chem( 52) = chem( 52) + w( 25)
chem( 15) = chem( 15) -  0.2000000000000000d+01 * w( 26)
chem( 48) = chem( 48) + w( 26)
chem( 52) = chem( 52) + w( 26)
chem( 38) = chem( 38) + w( 27)
chem( 42) = chem( 42) - w( 27)
chem( 52) = chem( 52) - w( 27)
chem( 38) = chem( 38) - w( 28)
chem( 42) = chem( 42) - w( 28)
chem( 44) = chem( 44) + w( 28)
chem( 42) = chem( 42) + w( 29)
chem( 48) = chem( 48) - w( 29)
chem( 50) = chem( 50) - w( 29)
chem( 52) = chem( 52) + w( 29)
chem( 13) = chem( 13) + w( 30)
chem( 50) = chem( 50) - w( 30)
chem( 52) = chem( 52) - w( 30)
chem( 13) = chem( 13) - w( 31)
chem( 50) = chem( 50) + w( 31)
chem( 52) = chem( 52) + w( 31)
chem( 13) = chem( 13) - w( 32)
chem( 42) = chem( 42) - w( 32)
chem( 52) = chem( 52) + w( 32)
chem( 17) = chem( 17) + w( 33)
chem( 50) = chem( 50) -  0.2000000000000000d+01 * w( 33)
chem( 17) = chem( 17) + w( 34)
chem( 50) = chem( 50) -  0.2000000000000000d+01 * w( 34)
chem( 17) = chem( 17) - w( 35)
chem( 42) = chem( 42) +  0.2000000000000000d+01 * w( 35)
chem( 17) = chem( 17) - w( 36)
chem( 42) = chem( 42) - w( 36)
chem( 50) = chem( 50) + w( 36)
chem(  1) = chem(  1) - w( 37)
chem( 42) = chem( 42) + w( 37)
chem( 50) = chem( 50) + w( 37)
chem( 42) = chem( 42) - w( 38)
chem( 50) = chem( 50) + w( 38)
chem( 42) = chem( 42) - w( 39)
chem( 43) = chem( 43) - w( 39)
chem( 50) = chem( 50) + w( 39)
chem( 42) = chem( 42) -  0.2000000000000000d+01 * w( 40)
chem( 43) = chem( 43) + w( 40)
chem( 17) = chem( 17) + w( 41)
chem( 42) = chem( 42) -  0.2000000000000000d+01 * w( 41)
chem( 42) = chem( 42) - w( 42)
chem( 50) = chem( 50) - w( 42)
chem( 42) = chem( 42) + w( 43)
chem( 43) = chem( 43) - w( 43)
chem( 50) = chem( 50) - w( 43)
chem( 17) = chem( 17) - w( 44)
chem( 42) = chem( 42) + w( 44)
chem( 43) = chem( 43) - w( 44)
chem( 50) = chem( 50) + w( 44)
chem( 43) = chem( 43) - w( 45)
chem( 44) = chem( 44) - w( 45)
chem( 52) = chem( 52) + w( 45)
chem( 42) = chem( 42) - w( 46)
chem( 44) = chem( 44) - w( 46)
chem( 50) = chem( 50) + w( 46)
chem( 52) = chem( 52) + w( 46)
chem( 38) = chem( 38) + w( 47)
chem( 44) = chem( 44) - w( 47)
chem( 50) = chem( 50) - w( 47)
chem( 44) = chem( 44) - w( 48)
chem( 47) = chem( 47) - w( 48)
chem( 52) = chem( 52) + w( 48)
chem( 44) = chem( 44) -  0.2000000000000000d+01 * w( 49)
chem( 52) = chem( 52) +  0.2000000000000000d+01 * w( 49)
chem( 13) = chem( 13) - w( 50)
chem( 42) = chem( 42) +  0.3900000000000000d+00 * w( 50)
chem( 44) = chem( 44) +  0.3900000000000000d+00 * w( 50)
chem( 50) = chem( 50) +  0.6100000000000000d+00 * w( 50)
chem( 52) = chem( 52) +  0.6100000000000000d+00 * w( 50)
chem( 38) = chem( 38) - w( 51)
chem( 42) = chem( 42) + w( 51)
chem( 52) = chem( 52) + w( 51)
chem(  9) = chem(  9) - w( 52)
chem( 44) = chem( 44) + w( 52)
chem( 52) = chem( 52) + w( 52)
chem( 48) = chem( 48) - w( 53)
chem( 49) = chem( 49) - w( 53)
chem( 52) = chem( 52) + w( 53)
chem( 33) = chem( 33) + w( 54)
chem( 39) = chem( 39) - w( 54)
chem( 48) = chem( 48) - w( 54)
chem( 21) = chem( 21) + w( 55)
chem( 49) = chem( 49) - w( 55)
chem( 50) = chem( 50) - w( 55)
chem( 21) = chem( 21) + w( 56)
chem( 39) = chem( 39) - w( 56)
chem( 50) = chem( 50) - w( 56)
chem( 49) = chem( 49) -  0.2000000000000000d+01 * w( 57)
chem( 39) = chem( 39) -  0.2000000000000000d+01 * w( 58)
chem( 39) = chem( 39) - w( 59)
chem( 49) = chem( 49) - w( 59)
chem( 33) = chem( 33) - w( 60)
chem( 34) = chem( 34) +  0.6600000000000000d+00 * w( 60)
chem( 35) = chem( 35) +  0.3300000000000000d+00 * w( 60)
chem( 38) = chem( 38) + w( 60)
chem( 41) = chem( 41) +  0.3300000000000000d+00 * w( 60)
chem( 42) = chem( 42) - w( 60)
chem( 45) = chem( 45) +  0.3300000000000000d+00 * w( 60)
chem( 50) = chem( 50) + w( 60)
chem( 33) = chem( 33) - w( 61)
chem( 34) = chem( 34) +  0.6600000000000000d+00 * w( 61)
chem( 35) = chem( 35) +  0.3300000000000000d+00 * w( 61)
chem( 41) = chem( 41) +  0.3300000000000000d+00 * w( 61)
chem( 45) = chem( 45) +  0.3300000000000000d+00 * w( 61)
chem( 50) = chem( 50) + w( 61)
chem( 52) = chem( 52) + w( 61)
chem( 21) = chem( 21) - w( 62)
chem( 35) = chem( 35) +  0.5000000000000000d+00 * w( 62)
chem( 42) = chem( 42) - w( 62)
chem( 45) = chem( 45) +  0.5000000000000000d+00 * w( 62)
chem( 49) = chem( 49) + w( 62)
chem( 21) = chem( 21) - w( 63)
chem( 35) = chem( 35) +  0.5000000000000000d+00 * w( 63)
chem( 42) = chem( 42) + w( 63)
chem( 45) = chem( 45) +  0.5000000000000000d+00 * w( 63)
chem( 50) = chem( 50) + w( 63)
chem( 37) = chem( 37) - w( 64)
chem( 42) = chem( 42) - w( 64)
chem( 50) = chem( 50) + w( 64)
chem(  3) = chem(  3) - w( 65)
chem( 40) = chem( 40) + w( 65)
chem( 42) = chem( 42) - w( 65)
chem( 40) = chem( 40) - w( 66)
chem( 41) = chem( 41) + w( 66)
chem( 48) = chem( 48) - w( 66)
chem( 50) = chem( 50) + w( 66)
chem( 52) = chem( 52) + w( 66)
chem( 19) = chem( 19) + w( 67)
chem( 40) = chem( 40) - w( 67)
chem( 50) = chem( 50) - w( 67)
chem( 11) = chem( 11) +  0.6300000000000000d+00 * w( 68)
chem( 40) = chem( 40) -  0.2000000000000000d+01 * w( 68)
chem( 41) = chem( 41) +  0.1370000000000000d+01 * w( 68)
chem( 50) = chem( 50) +  0.7400000000000000d+00 * w( 68)
chem( 19) = chem( 19) - w( 69)
chem( 40) = chem( 40) +  0.7000000000000000d+00 * w( 69)
chem( 42) = chem( 42) - w( 69)
chem( 49) = chem( 49) +  0.3000000000000000d+00 * w( 69)
chem( 50) = chem( 50) +  0.3000000000000000d-01 * w( 69)
chem( 19) = chem( 19) - w( 70)
chem( 41) = chem( 41) + w( 70)
chem( 42) = chem( 42) + w( 70)
chem( 50) = chem( 50) + w( 70)
chem( 11) = chem( 11) - w( 71)
chem( 41) = chem( 41) + w( 71)
chem( 42) = chem( 42) - w( 71)
chem( 50) = chem( 50) + w( 71)
chem( 37) = chem( 37) + w( 72)
chem( 41) = chem( 41) - w( 72)
chem( 42) = chem( 42) - w( 72)
chem( 50) = chem( 50) + w( 72)
chem( 37) = chem( 37) + w( 73)
chem( 41) = chem( 41) - w( 73)
chem( 50) = chem( 50) +  0.2000000000000000d+01 * w( 73)
chem( 37) = chem( 37) + w( 74)
chem( 41) = chem( 41) - w( 74)
chem( 37) = chem( 37) + w( 75)
chem( 41) = chem( 41) - w( 75)
chem( 42) = chem( 42) + w( 75)
chem( 43) = chem( 43) - w( 75)
chem( 50) = chem( 50) + w( 75)
chem( 37) = chem( 37) + w( 76)
chem( 38) = chem( 38) + w( 76)
chem( 41) = chem( 41) - w( 76)
chem( 44) = chem( 44) - w( 76)
chem( 50) = chem( 50) + w( 76)
chem( 18) = chem( 18) + w( 77)
chem( 41) = chem( 41) - w( 77)
chem( 50) = chem( 50) - w( 77)
chem( 18) = chem( 18) - w( 78)
chem( 41) = chem( 41) + w( 78)
chem( 50) = chem( 50) + w( 78)
chem( 18) = chem( 18) - w( 79)
chem( 20) = chem( 20) + w( 79)
chem( 48) = chem( 48) - w( 79)
chem( 50) = chem( 50) + w( 79)
chem( 52) = chem( 52) + w( 79)
chem( 18) = chem( 18) - w( 80)
chem( 19) = chem( 19) + w( 80)
chem( 50) = chem( 50) - w( 80)
chem( 20) = chem( 20) - w( 81)
chem( 42) = chem( 42) - w( 81)
chem( 50) = chem( 50) + w( 81)
chem( 42) = chem( 42) + w( 82)
chem( 43) = chem( 43) - w( 82)
chem( 45) = chem( 45) - w( 82)
chem( 51) = chem( 51) + w( 82)
chem( 42) = chem( 42) - w( 83)
chem( 45) = chem( 45) - w( 83)
chem( 51) = chem( 51) + w( 83)
chem( 38) = chem( 38) + w( 84)
chem( 44) = chem( 44) - w( 84)
chem( 45) = chem( 45) - w( 84)
chem( 51) = chem( 51) + w( 84)
chem( 37) = chem( 37) + w( 85)
chem( 40) = chem( 40) + w( 85)
chem( 45) = chem( 45) - w( 85)
chem( 50) = chem( 50) + w( 85)
chem( 40) = chem( 40) + w( 86)
chem( 48) = chem( 48) - w( 86)
chem( 51) = chem( 51) - w( 86)
chem( 52) = chem( 52) + w( 86)
chem( 10) = chem( 10) + w( 87)
chem( 51) = chem( 51) - w( 87)
chem( 52) = chem( 52) - w( 87)
chem( 10) = chem( 10) - w( 88)
chem( 51) = chem( 51) + w( 88)
chem( 52) = chem( 52) + w( 88)
chem( 10) = chem( 10) - w( 89)
chem( 51) = chem( 51) + w( 89)
chem( 52) = chem( 52) + w( 89)
chem( 22) = chem( 22) +  0.8000000000000000d+00 * w( 90)
chem( 36) = chem( 36) +  0.2000000000000000d+00 * w( 90)
chem( 47) = chem( 47) +  0.2000000000000000d+00 * w( 90)
chem( 50) = chem( 50) - w( 90)
chem( 51) = chem( 51) - w( 90)
chem( 36) = chem( 36) +  0.1000000000000000d+00 * w( 91)
chem( 40) = chem( 40) -  0.9999999999999998d-01 * w( 91)
chem( 41) = chem( 41) + w( 91)
chem( 50) = chem( 50) +  0.9000000000000000d+00 * w( 91)
chem( 51) = chem( 51) - w( 91)
chem( 36) = chem( 36) +  0.1000000000000000d+00 * w( 92)
chem( 40) = chem( 40) +  0.9000000000000000d+00 * w( 92)
chem( 49) = chem( 49) - w( 92)
chem( 51) = chem( 51) - w( 92)
chem( 40) = chem( 40) +  0.2000000000000000d+01 * w( 93)
chem( 51) = chem( 51) -  0.2000000000000000d+01 * w( 93)
chem( 22) = chem( 22) - w( 94)
chem( 42) = chem( 42) - w( 94)
chem( 51) = chem( 51) + w( 94)
chem( 22) = chem( 22) - w( 95)
chem( 40) = chem( 40) + w( 95)
chem( 42) = chem( 42) + w( 95)
chem( 36) = chem( 36) - w( 96)
chem( 40) = chem( 40) + w( 96)
chem( 42) = chem( 42) - w( 96)
chem( 35) = chem( 35) - w( 97)
chem( 42) = chem( 42) + w( 97)
chem( 43) = chem( 43) - w( 97)
chem( 46) = chem( 46) + w( 97)
chem( 35) = chem( 35) - w( 98)
chem( 42) = chem( 42) - w( 98)
chem( 46) = chem( 46) + w( 98)
chem( 35) = chem( 35) - w( 99)
chem( 38) = chem( 38) + w( 99)
chem( 44) = chem( 44) - w( 99)
chem( 46) = chem( 46) + w( 99)
chem( 35) = chem( 35) - w(100)
chem( 37) = chem( 37) + w(100)
chem( 40) = chem( 40) + w(100)
chem( 50) = chem( 50) + w(100)
chem( 45) = chem( 45) + w(101)
chem( 46) = chem( 46) - w(101)
chem( 48) = chem( 48) - w(101)
chem( 49) = chem( 49) + w(101)
chem( 50) = chem( 50) + w(101)
chem( 52) = chem( 52) + w(101)
chem( 14) = chem( 14) + w(102)
chem( 46) = chem( 46) - w(102)
chem( 52) = chem( 52) - w(102)
chem( 14) = chem( 14) - w(103)
chem( 46) = chem( 46) + w(103)
chem( 52) = chem( 52) + w(103)
chem( 14) = chem( 14) - w(104)
chem( 46) = chem( 46) + w(104)
chem( 52) = chem( 52) + w(104)
chem( 14) = chem( 14) - w(105)
chem( 42) = chem( 42) - w(105)
chem( 45) = chem( 45) + w(105)
chem( 52) = chem( 52) + w(105)
chem( 22) = chem( 22) +  0.8000000000000000d+00 * w(106)
chem( 36) = chem( 36) +  0.2000000000000000d+00 * w(106)
chem( 46) = chem( 46) - w(106)
chem( 47) = chem( 47) +  0.2000000000000000d+00 * w(106)
chem( 50) = chem( 50) - w(106)
chem( 36) = chem( 36) +  0.1000000000000000d+00 * w(107)
chem( 40) = chem( 40) - w(107)
chem( 41) = chem( 41) +  0.1000000000000000d+00 * w(107)
chem( 45) = chem( 45) +  0.9000000000000000d+00 * w(107)
chem( 49) = chem( 49) +  0.9000000000000000d+00 * w(107)
chem( 50) = chem( 50) + w(107)
chem( 51) = chem( 51) - w(107)
chem( 36) = chem( 36) +  0.1000000000000000d+00 * w(108)
chem( 45) = chem( 45) +  0.9000000000000000d+00 * w(108)
chem( 49) = chem( 49) - w(108)
chem( 51) = chem( 51) - w(108)
chem( 45) = chem( 45) +  0.2000000000000000d+01 * w(109)
chem( 46) = chem( 46) -  0.2000000000000000d+01 * w(109)
chem( 49) = chem( 49) +  0.2000000000000000d+01 * w(109)
chem( 50) = chem( 50) +  0.2000000000000000d+01 * w(109)
chem( 40) = chem( 40) + w(110)
chem( 45) = chem( 45) + w(110)
chem( 46) = chem( 46) - w(110)
chem( 49) = chem( 49) + w(110)
chem( 50) = chem( 50) + w(110)
chem( 51) = chem( 51) - w(110)
chem( 16) = chem( 16) +  0.7600000000000000d+00 * w(111)
chem( 34) = chem( 34) -  0.8900000000000000d+00 * w(111)
chem( 35) = chem( 35) +  0.5000000000000000d-01 * w(111)
chem( 39) = chem( 39) +  0.1300000000000000d+00 * w(111)
chem( 42) = chem( 42) - w(111)
chem( 45) = chem( 45) +  0.6000000000000000d-01 * w(111)
chem( 49) = chem( 49) +  0.8700000000000000d+00 * w(111)
chem( 50) = chem( 50) +  0.1100000000000000d+00 * w(111)
chem( 16) = chem( 16) -  0.9800000000000000d+00 * w(112)
chem( 34) = chem( 34) +  0.2100000000000000d+01 * w(112)
chem( 35) = chem( 35) +  0.5000000000000000d+00 * w(112)
chem( 39) = chem( 39) +  0.4000000000000000d-01 * w(112)
chem( 45) = chem( 45) +  0.6000000000000000d+00 * w(112)
chem( 49) = chem( 49) +  0.9600000000000000d+00 * w(112)
chem( 50) = chem( 50) +  0.9399999999999999d+00 * w(112)
chem( 16) = chem( 16) - w(113)
chem( 50) = chem( 50) + w(113)
chem( 16) = chem( 16) - w(114)
chem( 33) = chem( 33) + w(114)
chem( 52) = chem( 52) - w(114)
chem( 24) = chem( 24) - w(115)
chem( 34) = chem( 34) +  0.2000000000000000d+00 * w(115)
chem( 35) = chem( 35) +  0.3000000000000000d+00 * w(115)
chem( 37) = chem( 37) +  0.2000000000000000d+00 * w(115)
chem( 39) = chem( 39) +  0.1000000000000000d-01 * w(115)
chem( 41) = chem( 41) +  0.2000000000000000d+00 * w(115)
chem( 42) = chem( 42) +  0.1000000000000000d+00 * w(115)
chem( 43) = chem( 43) - w(115)
chem( 45) = chem( 45) +  0.2000000000000000d+00 * w(115)
chem( 49) = chem( 49) +  0.2000000000000000d+00 * w(115)
chem( 50) = chem( 50) +  0.3000000000000000d+00 * w(115)
chem( 24) = chem( 24) - w(116)
chem( 34) = chem( 34) +  0.7000000000000000d+00 * w(116)
chem( 35) = chem( 35) +  0.6200000000000000d+00 * w(116)
chem( 41) = chem( 41) +  0.8000000000000000d+00 * w(116)
chem( 42) = chem( 42) - w(116)
chem( 45) = chem( 45) +  0.3300000000000000d+00 * w(116)
chem( 49) = chem( 49) +  0.8000000000000000d+00 * w(116)
chem( 50) = chem( 50) +  0.9500000000000000d+00 * w(116)
chem( 24) = chem( 24) - w(117)
chem( 34) = chem( 34) + w(117)
chem( 35) = chem( 35) +  0.3200000000000000d+00 * w(117)
chem( 37) = chem( 37) +  0.3300000000000000d+00 * w(117)
chem( 41) = chem( 41) +  0.7400000000000000d+00 * w(117)
chem( 42) = chem( 42) +  0.1000000000000000d+00 * w(117)
chem( 45) = chem( 45) +  0.1800000000000000d+00 * w(117)
chem( 47) = chem( 47) - w(117)
chem( 49) = chem( 49) +  0.2200000000000000d+00 * w(117)
chem( 50) = chem( 50) +  0.4400000000000000d+00 * w(117)
chem( 24) = chem( 24) - w(118)
chem( 34) = chem( 34) + w(118)
chem( 35) = chem( 35) +  0.5600000000000001d+00 * w(118)
chem( 39) = chem( 39) +  0.9000000000000000d-01 * w(118)
chem( 41) = chem( 41) + w(118)
chem( 44) = chem( 44) - w(118)
chem( 45) = chem( 45) +  0.3500000000000000d+00 * w(118)
chem( 49) = chem( 49) +  0.9100000000000000d+00 * w(118)
chem( 52) = chem( 52) + w(118)
chem( 23) = chem( 23) - w(119)
chem( 37) = chem( 37) + w(119)
chem( 41) = chem( 41) + w(119)
chem( 42) = chem( 42) +  0.3000000000000000d+00 * w(119)
chem( 43) = chem( 43) - w(119)
chem( 49) = chem( 49) +  0.7000000000000000d+00 * w(119)
chem( 50) = chem( 50) +  0.1700000000000000d+01 * w(119)
chem( 23) = chem( 23) - w(120)
chem( 35) = chem( 35) +  0.2200000000000000d+00 * w(120)
chem( 41) = chem( 41) +  0.1560000000000000d+01 * w(120)
chem( 42) = chem( 42) - w(120)
chem( 49) = chem( 49) + w(120)
chem( 50) = chem( 50) + w(120)
chem( 20) = chem( 20) +  0.3700000000000000d+00 * w(121)
chem( 23) = chem( 23) - w(121)
chem( 37) = chem( 37) +  0.6300000000000000d+00 * w(121)
chem( 41) = chem( 41) + w(121)
chem( 42) = chem( 42) +  0.1300000000000000d+00 * w(121)
chem( 47) = chem( 47) - w(121)
chem( 50) = chem( 50) +  0.1300000000000000d+00 * w(121)
chem( 23) = chem( 23) - w(122)
chem( 41) = chem( 41) +  0.2000000000000000d+01 * w(122)
chem( 44) = chem( 44) - w(122)
chem( 49) = chem( 49) + w(122)
chem( 52) = chem( 52) + w(122)
chem( 25) = chem( 25) - w(123)
chem( 34) = chem( 34) +  0.1000000000000000d+00 * w(123)
chem( 35) = chem( 35) +  0.6600000000000000d+00 * w(123)
chem( 37) = chem( 37) +  0.1000000000000000d+00 * w(123)
chem( 43) = chem( 43) - w(123)
chem( 45) = chem( 45) +  0.1240000000000000d+01 * w(123)
chem( 49) = chem( 49) +  0.1000000000000000d+00 * w(123)
chem( 50) = chem( 50) +  0.1000000000000000d+00 * w(123)
chem( 25) = chem( 25) - w(124)
chem( 35) = chem( 35) +  0.7000000000000000d+00 * w(124)
chem( 42) = chem( 42) - w(124)
chem( 45) = chem( 45) +  0.1300000000000000d+01 * w(124)
chem( 49) = chem( 49) + w(124)
chem( 50) = chem( 50) + w(124)
chem( 25) = chem( 25) - w(125)
chem( 35) = chem( 35) +  0.3500000000000000d+00 * w(125)
chem( 37) = chem( 37) +  0.2500000000000000d+00 * w(125)
chem( 41) = chem( 41) +  0.2500000000000000d+00 * w(125)
chem( 42) = chem( 42) +  0.5000000000000000d+00 * w(125)
chem( 43) = chem( 43) +  0.5000000000000000d+00 * w(125)
chem( 45) = chem( 45) +  0.6500000000000000d+00 * w(125)
chem( 47) = chem( 47) - w(125)
chem( 50) = chem( 50) +  0.5000000000000000d+00 * w(125)
chem( 25) = chem( 25) - w(126)
chem( 35) = chem( 35) +  0.6400000000000000d+00 * w(126)
chem( 44) = chem( 44) - w(126)
chem( 45) = chem( 45) +  0.1180000000000000d+01 * w(126)
chem( 50) = chem( 50) + w(126)
chem( 52) = chem( 52) + w(126)
chem(  5) = chem(  5) - w(127)
chem( 12) = chem( 12) +  0.5600000000000001d+00 * w(127)
chem( 27) = chem( 27) +  0.3600000000000000d+00 * w(127)
chem( 42) = chem( 42) - w(127)
chem( 49) = chem( 49) +  0.8000000000000000d-01 * w(127)
chem( 50) = chem( 50) +  0.4400000000000000d+00 * w(127)
chem( 12) = chem( 12) - w(128)
chem( 28) = chem( 28) +  0.9000000000000000d+00 * w(128)
chem( 33) = chem( 33) +  0.1000000000000000d+00 * w(128)
chem( 48) = chem( 48) - w(128)
chem( 50) = chem( 50) +  0.9000000000000000d+00 * w(128)
chem( 52) = chem( 52) +  0.9000000000000000d+00 * w(128)
chem( 12) = chem( 12) - w(129)
chem( 27) = chem( 27) + w(129)
chem( 50) = chem( 50) + w(129)
chem( 27) = chem( 27) - w(130)
chem( 28) = chem( 28) +  0.3000000000000000d+00 * w(130)
chem( 30) = chem( 30) +  0.4000000000000000d+00 * w(130)
chem( 42) = chem( 42) - w(130)
chem( 49) = chem( 49) +  0.6000000000000000d+00 * w(130)
chem( 50) = chem( 50) +  0.6000000000000000d+00 * w(130)
chem( 27) = chem( 27) - w(131)
chem( 30) = chem( 30) + w(131)
chem( 38) = chem( 38) + w(131)
chem( 44) = chem( 44) - w(131)
chem( 30) = chem( 30) - w(132)
chem( 33) = chem( 33) + w(132)
chem( 52) = chem( 52) - w(132)
chem( 27) = chem( 27) + w(133)
chem( 30) = chem( 30) - w(133)
chem( 50) = chem( 50) - w(133)
chem( 28) = chem( 28) - w(134)
chem( 37) = chem( 37) + w(134)
chem( 50) = chem( 50) + w(134)
chem( 51) = chem( 51) + w(134)
chem( 28) = chem( 28) - w(135)
chem( 37) = chem( 37) +  0.2000000000000000d+01 * w(135)
chem( 41) = chem( 41) + w(135)
chem( 42) = chem( 42) - w(135)
chem( 49) = chem( 49) + w(135)
chem( 50) = chem( 50) +  0.2000000000000000d+01 * w(135)
chem( 51) = chem( 51) + w(135)
chem( 28) = chem( 28) - w(136)
chem( 29) = chem( 29) +  0.2000000000000000d+00 * w(136)
chem( 35) = chem( 35) +  0.3000000000000000d-01 * w(136)
chem( 37) = chem( 37) +  0.6899999999999999d+00 * w(136)
chem( 41) = chem( 41) +  0.7000000000000000d+00 * w(136)
chem( 42) = chem( 42) +  0.8000000000000000d-01 * w(136)
chem( 47) = chem( 47) - w(136)
chem( 49) = chem( 49) +  0.3000000000000000d-01 * w(136)
chem( 50) = chem( 50) +  0.7600000000000000d+00 * w(136)
chem( 51) = chem( 51) +  0.6200000000000000d+00 * w(136)
chem(  6) = chem(  6) - w(137)
chem( 12) = chem( 12) +  0.3000000000000000d+00 * w(137)
chem( 27) = chem( 27) +  0.2000000000000000d-01 * w(137)
chem( 29) = chem( 29) +  0.8000000000000000d+00 * w(137)
chem( 34) = chem( 34) +  0.1100000000000000d+01 * w(137)
chem( 42) = chem( 42) - w(137)
chem( 49) = chem( 49) +  0.5000000000000000d+00 * w(137)
chem( 50) = chem( 50) +  0.7000000000000000d+00 * w(137)
chem( 29) = chem( 29) - w(138)
chem( 42) = chem( 42) - w(138)
chem( 49) = chem( 49) + w(138)
chem( 51) = chem( 51) + w(138)
chem( 29) = chem( 29) - w(139)
chem( 37) = chem( 37) + w(139)
chem( 50) = chem( 50) + w(139)
chem( 51) = chem( 51) + w(139)
chem( 31) = chem( 31) - w(140)
chem( 32) = chem( 32) +  0.7500000000000000d+00 * w(140)
chem( 34) = chem( 34) +  0.2500000000000000d+00 * w(140)
chem( 41) = chem( 41) +  0.5000000000000000d+00 * w(140)
chem( 43) = chem( 43) - w(140)
chem( 46) = chem( 46) +  0.2500000000000000d+00 * w(140)
chem( 49) = chem( 49) +  0.2500000000000000d+00 * w(140)
chem( 50) = chem( 50) +  0.2500000000000000d+00 * w(140)
chem( 31) = chem( 31) - w(141)
chem( 32) = chem( 32) +  0.9120000000000000d+00 * w(141)
chem( 39) = chem( 39) +  0.8799999999999999d-01 * w(141)
chem( 41) = chem( 41) +  0.6290000000000000d+00 * w(141)
chem( 42) = chem( 42) - w(141)
chem( 49) = chem( 49) +  0.9910000000000000d+00 * w(141)
chem( 50) = chem( 50) +  0.9120000000000000d+00 * w(141)
chem( 31) = chem( 31) - w(142)
chem( 32) = chem( 32) +  0.6500000000000000d+00 * w(142)
chem( 34) = chem( 34) +  0.3500000000000000d+00 * w(142)
chem( 35) = chem( 35) +  0.1500000000000000d+00 * w(142)
chem( 37) = chem( 37) +  0.6600000000000000d-01 * w(142)
chem( 41) = chem( 41) +  0.6000000000000000d+00 * w(142)
chem( 42) = chem( 42) +  0.2660000000000000d+00 * w(142)
chem( 46) = chem( 46) +  0.2000000000000000d+00 * w(142)
chem( 47) = chem( 47) - w(142)
chem( 49) = chem( 49) +  0.2000000000000000d+00 * w(142)
chem( 50) = chem( 50) +  0.6600000000000000d-01 * w(142)
chem( 31) = chem( 31) - w(143)
chem( 32) = chem( 32) +  0.2000000000000000d+00 * w(143)
chem( 33) = chem( 33) +  0.8000000000000000d+00 * w(143)
chem( 34) = chem( 34) +  0.2400000000000000d+01 * w(143)
chem( 35) = chem( 35) +  0.8000000000000000d+00 * w(143)
chem( 44) = chem( 44) - w(143)
chem( 49) = chem( 49) + w(143)
chem( 50) = chem( 50) +  0.8000000000000000d+00 * w(143)
chem( 52) = chem( 52) +  0.2000000000000000d+00 * w(143)
chem( 29) = chem( 29) +  0.1680000000000000d+00 * w(144)
chem( 32) = chem( 32) - w(144)
chem( 34) = chem( 34) +  0.1565000000000000d+01 * w(144)
chem( 35) = chem( 35) +  0.1200000000000000d+00 * w(144)
chem( 37) = chem( 37) +  0.3340000000000000d+00 * w(144)
chem( 41) = chem( 41) +  0.1670000000000000d+00 * w(144)
chem( 42) = chem( 42) - w(144)
chem( 45) = chem( 45) +  0.2520000000000000d+00 * w(144)
chem( 46) = chem( 46) +  0.2500000000000000d+00 * w(144)
chem( 49) = chem( 49) +  0.7130000000000000d+00 * w(144)
chem( 50) = chem( 50) +  0.5030000000000000d+00 * w(144)
chem( 51) = chem( 51) +  0.2100000000000000d+00 * w(144)
chem( 29) = chem( 29) +  0.8500000000000000d+00 * w(145)
chem( 32) = chem( 32) - w(145)
chem( 34) = chem( 34) +  0.3600000000000000d+00 * w(145)
chem( 37) = chem( 37) +  0.2250000000000000d+00 * w(145)
chem( 41) = chem( 41) +  0.1500000000000000d+00 * w(145)
chem( 42) = chem( 42) +  0.2680000000000000d+00 * w(145)
chem( 45) = chem( 45) +  0.2000000000000000d-01 * w(145)
chem( 47) = chem( 47) - w(145)
chem( 49) = chem( 49) +  0.6400000000000000d-01 * w(145)
chem( 50) = chem( 50) +  0.1540000000000000d+00 * w(145)
chem( 51) = chem( 51) +  0.1140000000000000d+00 * w(145)
chem( 32) = chem( 32) - w(146)
chem( 33) = chem( 33) +  0.8500000000000000d+00 * w(146)
chem( 34) = chem( 34) +  0.1282000000000000d+01 * w(146)
chem( 35) = chem( 35) +  0.3570000000000000d+00 * w(146)
chem( 37) = chem( 37) +  0.6430000000000000d+00 * w(146)
chem( 38) = chem( 38) +  0.1500000000000000d+00 * w(146)
chem( 41) = chem( 41) +  0.2820000000000000d+00 * w(146)
chem( 44) = chem( 44) - w(146)
chem( 46) = chem( 46) +  0.7500000000000000d-01 * w(146)
chem( 49) = chem( 49) +  0.7500000000000000d-01 * w(146)
chem( 50) = chem( 50) +  0.9250000000000000d+00 * w(146)
chem( 32) = chem( 32) - w(147)
chem( 34) = chem( 34) +  0.8320000000000000d+00 * w(147)
chem( 37) = chem( 37) +  0.3330000000000000d+00 * w(147)
chem( 41) = chem( 41) +  0.9000000000000000d+00 * w(147)
chem( 45) = chem( 45) +  0.6700000000000000d-01 * w(147)
chem( 49) = chem( 49) +  0.7000000000000000d+00 * w(147)
chem( 50) = chem( 50) +  0.1033000000000000d+01 * w(147)
chem( 51) = chem( 51) +  0.9670000000000000d+00 * w(147)
chem( 26) = chem( 26) - w(148)
chem( 34) = chem( 34) +  0.5120000000000000d+01 * w(148)
chem( 35) = chem( 35) +  0.1500000000000000d+00 * w(148)
chem( 43) = chem( 43) - w(148)
chem( 26) = chem( 26) - w(149)
chem( 34) = chem( 34) +  0.1660000000000000d+01 * w(149)
chem( 35) = chem( 35) +  0.4700000000000000d+00 * w(149)
chem( 39) = chem( 39) +  0.2500000000000000d+00 * w(149)
chem( 41) = chem( 41) +  0.2800000000000000d+00 * w(149)
chem( 42) = chem( 42) - w(149)
chem( 49) = chem( 49) +  0.1250000000000000d+01 * w(149)
chem( 50) = chem( 50) +  0.7500000000000000d+00 * w(149)
chem( 26) = chem( 26) - w(150)
chem( 34) = chem( 34) +  0.7000000000000000d+01 * w(150)
chem( 35) = chem( 35) +  0.2100000000000000d+00 * w(150)
chem( 37) = chem( 37) +  0.1000000000000000d-02 * w(150)
chem( 39) = chem( 39) +  0.1800000000000000d+00 * w(150)
chem( 41) = chem( 41) +  0.2400000000000000d+00 * w(150)
chem( 42) = chem( 42) +  0.5700000000000000d+00 * w(150)
chem( 46) = chem( 46) +  0.3900000000000000d+00 * w(150)
chem( 47) = chem( 47) - w(150)
chem( 49) = chem( 49) +  0.7600000000000000d+00 * w(150)
chem( 50) = chem( 50) +  0.7000000000000001d-01 * w(150)
chem( 26) = chem( 26) - w(151)
chem( 33) = chem( 33) +  0.5300000000000000d+00 * w(151)
chem( 35) = chem( 35) +  0.4700000000000000d+00 * w(151)
chem( 39) = chem( 39) +  0.2500000000000000d+00 * w(151)
chem( 44) = chem( 44) - w(151)
chem( 49) = chem( 49) +  0.1030000000000000d+01 * w(151)
chem( 50) = chem( 50) +  0.2800000000000000d+00 * w(151)
chem( 52) = chem( 52) +  0.4700000000000000d+00 * w(151)
chem(  7) = chem(  7) - w(152)
chem(  8) = chem(  8) + w(152)
chem( 42) = chem( 42) - w(152)
chem( 50) = chem( 50) + w(152)
chem(  2) = chem(  2) - w(153)
chem( 35) = chem( 35) +  0.5000000000000000d-01 * w(153)
chem( 41) = chem( 41) +  0.1000000000000000d+00 * w(153)
chem( 42) = chem( 42) - w(153)
chem( 45) = chem( 45) +  0.9000000000000000d+00 * w(153)
chem( 49) = chem( 49) +  0.1000000000000000d+00 * w(153)
chem( 50) = chem( 50) + w(153)
chem(  4) = chem(  4) - w(154)
chem( 39) = chem( 39) +  0.8999999999999999d-02 * w(154)
chem( 42) = chem( 42) - w(154)
chem( 45) = chem( 45) +  0.9910000000000000d+00 * w(154)
chem( 49) = chem( 49) +  0.9910000000000000d+00 * w(154)
chem( 50) = chem( 50) + w(154)
chem( 31) = chem( 31) - w(155)
chem( 32) = chem( 32) +  0.2000000000000000d+00 * w(155)
chem( 33) = chem( 33) +  0.8000000000000000d+00 * w(155)
chem( 34) = chem( 34) +  0.2400000000000000d+01 * w(155)
chem( 35) = chem( 35) +  0.8000000000000000d+00 * w(155)
chem( 48) = chem( 48) +  0.2000000000000000d+00 * w(155)
chem( 49) = chem( 49) + w(155)
chem( 50) = chem( 50) +  0.8000000000000000d+00 * w(155)
chem( 52) = chem( 52) - w(155)

!    Conversion molecules/cm3 to mug/m3.

do i = 1, ns
   chem(i) = chem(i) / convers_factor(i)
enddo

!     Volumic source terms.

do i=1,ns
chem(i)=chem(i)+zcsourc(i)
enddo

return
end subroutine fexchem_3


!===============================================================================

!> jacdchemdc_3
!> \brief Computation of the Jacobian matrix for atmospheric chemistry
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     ns                 total number of chemical species
!> \param[in]     nr                 total number of chemical reactions
!> \param[in]     y                  concentrations vector
!> \param[in]     convers_factor     conversion factors of mug/m3
!>                                   to molecules/cm3
!> \param[in]     convers_factor_jac conversion factors for the Jacobian matrix
!>                                   (Wmol(i)/Wmol(j))
!> \param[in]     rk                 kinetic rates
!> \param[out]    jacc               Jacobian matrix
!______________________________________________________________________________

subroutine jacdchemdc_3(ns,nr,y,convers_factor,                     &
                    convers_factor_jac,rk,jacc)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns),jacc(ns,ns)
double precision convers_factor(ns)
double precision convers_factor_jac(ns,ns)

! Local variables

integer i,j
double precision dw(nr,ns)
double precision conc(ns)

do j=1,ns
 do i=1,ns
  jacc(i,j)=0.d0
 enddo
enddo

!     Conversion mug/m3 to molecules/cm3.

do i = 1, ns
   conc(i) = y(i) * convers_factor(i)
enddo

call dratedc_3(ns,nr,rk,conc,dw)

jacc( 43, 52) = jacc( 43, 52) + dw(  1, 52)
jacc( 48, 52) = jacc( 48, 52) + dw(  1, 52)
jacc( 52, 52) = jacc( 52, 52) - dw(  1, 52)
jacc( 43, 43) = jacc( 43, 43) - dw(  2, 43)
jacc( 47, 43) = jacc( 47, 43) + dw(  2, 43)
jacc( 47, 47) = jacc( 47, 47) - dw(  3, 47)
jacc( 47, 48) = jacc( 47, 48) - dw(  3, 48)
jacc( 48, 47) = jacc( 48, 47) - dw(  3, 47)
jacc( 48, 48) = jacc( 48, 48) - dw(  3, 48)
jacc( 52, 47) = jacc( 52, 47) + dw(  3, 47)
jacc( 52, 48) = jacc( 52, 48) + dw(  3, 48)
jacc( 43, 43) = jacc( 43, 43) - dw(  4, 43)
jacc( 43, 52) = jacc( 43, 52) - dw(  4, 52)
jacc( 48, 43) = jacc( 48, 43) + dw(  4, 43)
jacc( 48, 52) = jacc( 48, 52) + dw(  4, 52)
jacc( 52, 43) = jacc( 52, 43) - dw(  4, 43)
jacc( 52, 52) = jacc( 52, 52) - dw(  4, 52)
jacc( 43, 43) = jacc( 43, 43) - dw(  5, 43)
jacc( 43, 52) = jacc( 43, 52) - dw(  5, 52)
jacc( 44, 43) = jacc( 44, 43) + dw(  5, 43)
jacc( 44, 52) = jacc( 44, 52) + dw(  5, 52)
jacc( 52, 43) = jacc( 52, 43) - dw(  5, 43)
jacc( 52, 52) = jacc( 52, 52) - dw(  5, 52)
jacc( 43, 43) = jacc( 43, 43) - dw(  6, 43)
jacc( 43, 48) = jacc( 43, 48) - dw(  6, 48)
jacc( 48, 43) = jacc( 48, 43) - dw(  6, 43)
jacc( 48, 48) = jacc( 48, 48) - dw(  6, 48)
jacc( 52, 43) = jacc( 52, 43) + dw(  6, 43)
jacc( 52, 48) = jacc( 52, 48) + dw(  6, 48)
jacc( 44, 52) = jacc( 44, 52) + dw(  7, 52)
jacc( 44, 47) = jacc( 44, 47) + dw(  7, 47)
jacc( 47, 52) = jacc( 47, 52) - dw(  7, 52)
jacc( 47, 47) = jacc( 47, 47) - dw(  7, 47)
jacc( 52, 52) = jacc( 52, 52) - dw(  7, 52)
jacc( 52, 47) = jacc( 52, 47) - dw(  7, 47)
jacc( 43, 47) = jacc( 43, 47) + dw(  8, 47)
jacc( 47, 47) = jacc( 47, 47) - dw(  8, 47)
jacc(  1, 47) = jacc(  1, 47) + dw(  9, 47)
jacc( 47, 47) = jacc( 47, 47) - dw(  9, 47)
jacc(  1,  1) = jacc(  1,  1) - dw( 10,  1)
jacc( 43,  1) = jacc( 43,  1) + dw( 10,  1)
jacc(  1,  1) = jacc(  1,  1) - dw( 11,  1)
jacc( 42,  1) = jacc( 42,  1)+ 0.2000000000000000d+01*dw( 11,  1)
jacc( 42, 47) = jacc( 42, 47) - dw( 12, 47)
jacc( 42, 42) = jacc( 42, 42) - dw( 12, 42)
jacc( 47, 47) = jacc( 47, 47) - dw( 12, 47)
jacc( 47, 42) = jacc( 47, 42) - dw( 12, 42)
jacc( 50, 47) = jacc( 50, 47) + dw( 12, 47)
jacc( 50, 42) = jacc( 50, 42) + dw( 12, 42)
jacc( 42, 47) = jacc( 42, 47) + dw( 13, 47)
jacc( 42, 50) = jacc( 42, 50) + dw( 13, 50)
jacc( 47, 47) = jacc( 47, 47) - dw( 13, 47)
jacc( 47, 50) = jacc( 47, 50) - dw( 13, 50)
jacc( 50, 47) = jacc( 50, 47) - dw( 13, 47)
jacc( 50, 50) = jacc( 50, 50) - dw( 13, 50)
jacc( 43, 44) = jacc( 43, 44) + dw( 14, 44)
jacc( 44, 44) = jacc( 44, 44) - dw( 14, 44)
jacc( 52, 44) = jacc( 52, 44) + dw( 14, 44)
jacc( 44, 44) = jacc( 44, 44) - dw( 15, 44)
jacc( 48, 44) = jacc( 48, 44) + dw( 15, 44)
jacc( 44, 44) = jacc( 44, 44) - dw( 16, 44)
jacc( 44, 48) = jacc( 44, 48) - dw( 16, 48)
jacc( 48, 44) = jacc( 48, 44) - dw( 16, 44)
jacc( 48, 48) = jacc( 48, 48) - dw( 16, 48)
jacc( 52, 44) = jacc( 52, 44)+ 0.2000000000000000d+01*dw( 16, 44)
jacc( 52, 48) = jacc( 52, 48)+ 0.2000000000000000d+01*dw( 16, 48)
jacc( 44, 44) = jacc( 44, 44) - dw( 17, 44)
jacc( 44, 52) = jacc( 44, 52) - dw( 17, 52)
jacc( 48, 44) = jacc( 48, 44) + dw( 17, 44)
jacc( 48, 52) = jacc( 48, 52) + dw( 17, 52)
jacc(  9, 44) = jacc(  9, 44) + dw( 18, 44)
jacc(  9, 52) = jacc(  9, 52) + dw( 18, 52)
jacc( 44, 44) = jacc( 44, 44) - dw( 18, 44)
jacc( 44, 52) = jacc( 44, 52) - dw( 18, 52)
jacc( 52, 44) = jacc( 52, 44) - dw( 18, 44)
jacc( 52, 52) = jacc( 52, 52) - dw( 18, 52)
jacc(  9,  9) = jacc(  9,  9) - dw( 19,  9)
jacc( 38,  9) = jacc( 38,  9)+ 0.2000000000000000d+01*dw( 19,  9)
jacc(  9,  9) = jacc(  9,  9) - dw( 20,  9)
jacc( 44,  9) = jacc( 44,  9) + dw( 20,  9)
jacc( 52,  9) = jacc( 52,  9) + dw( 20,  9)
jacc( 48, 48) = jacc( 48, 48)- 0.2000000000000000d+01*dw( 21, 48)
jacc( 48, 48) = jacc( 48, 48)- 0.2000000000000000d+01*dw( 21, 48)
jacc( 52, 48) = jacc( 52, 48)+ 0.2000000000000000d+01*dw( 21, 48)
jacc( 52, 48) = jacc( 52, 48)+ 0.2000000000000000d+01*dw( 21, 48)
jacc( 15, 48) = jacc( 15, 48)+ 0.2000000000000000d+01*dw( 22, 48)
jacc( 15, 52) = jacc( 15, 52)+ 0.2000000000000000d+01*dw( 22, 52)
jacc( 48, 48) = jacc( 48, 48) - dw( 22, 48)
jacc( 48, 52) = jacc( 48, 52) - dw( 22, 52)
jacc( 52, 48) = jacc( 52, 48) - dw( 22, 48)
jacc( 52, 52) = jacc( 52, 52) - dw( 22, 52)
jacc( 15, 42) = jacc( 15, 42) + dw( 23, 42)
jacc( 15, 48) = jacc( 15, 48) + dw( 23, 48)
jacc( 42, 42) = jacc( 42, 42) - dw( 23, 42)
jacc( 42, 48) = jacc( 42, 48) - dw( 23, 48)
jacc( 48, 42) = jacc( 48, 42) - dw( 23, 42)
jacc( 48, 48) = jacc( 48, 48) - dw( 23, 48)
jacc( 15, 15) = jacc( 15, 15) - dw( 24, 15)
jacc( 42, 15) = jacc( 42, 15) + dw( 24, 15)
jacc( 48, 15) = jacc( 48, 15) + dw( 24, 15)
jacc( 15, 42) = jacc( 15, 42) - dw( 25, 42)
jacc( 15, 15) = jacc( 15, 15) - dw( 25, 15)
jacc( 42, 42) = jacc( 42, 42) - dw( 25, 42)
jacc( 42, 15) = jacc( 42, 15) - dw( 25, 15)
jacc( 52, 42) = jacc( 52, 42) + dw( 25, 42)
jacc( 52, 15) = jacc( 52, 15) + dw( 25, 15)
jacc( 15, 15) = jacc( 15, 15)- 0.2000000000000000d+01*dw( 26, 15)
jacc( 15, 15) = jacc( 15, 15)- 0.2000000000000000d+01*dw( 26, 15)
jacc( 48, 15) = jacc( 48, 15) + dw( 26, 15)
jacc( 48, 15) = jacc( 48, 15) + dw( 26, 15)
jacc( 52, 15) = jacc( 52, 15) + dw( 26, 15)
jacc( 52, 15) = jacc( 52, 15) + dw( 26, 15)
jacc( 38, 52) = jacc( 38, 52) + dw( 27, 52)
jacc( 38, 42) = jacc( 38, 42) + dw( 27, 42)
jacc( 42, 52) = jacc( 42, 52) - dw( 27, 52)
jacc( 42, 42) = jacc( 42, 42) - dw( 27, 42)
jacc( 52, 52) = jacc( 52, 52) - dw( 27, 52)
jacc( 52, 42) = jacc( 52, 42) - dw( 27, 42)
jacc( 38, 42) = jacc( 38, 42) - dw( 28, 42)
jacc( 38, 38) = jacc( 38, 38) - dw( 28, 38)
jacc( 42, 42) = jacc( 42, 42) - dw( 28, 42)
jacc( 42, 38) = jacc( 42, 38) - dw( 28, 38)
jacc( 44, 42) = jacc( 44, 42) + dw( 28, 42)
jacc( 44, 38) = jacc( 44, 38) + dw( 28, 38)
jacc( 42, 50) = jacc( 42, 50) + dw( 29, 50)
jacc( 42, 48) = jacc( 42, 48) + dw( 29, 48)
jacc( 48, 50) = jacc( 48, 50) - dw( 29, 50)
jacc( 48, 48) = jacc( 48, 48) - dw( 29, 48)
jacc( 50, 50) = jacc( 50, 50) - dw( 29, 50)
jacc( 50, 48) = jacc( 50, 48) - dw( 29, 48)
jacc( 52, 50) = jacc( 52, 50) + dw( 29, 50)
jacc( 52, 48) = jacc( 52, 48) + dw( 29, 48)
jacc( 13, 50) = jacc( 13, 50) + dw( 30, 50)
jacc( 13, 52) = jacc( 13, 52) + dw( 30, 52)
jacc( 50, 50) = jacc( 50, 50) - dw( 30, 50)
jacc( 50, 52) = jacc( 50, 52) - dw( 30, 52)
jacc( 52, 50) = jacc( 52, 50) - dw( 30, 50)
jacc( 52, 52) = jacc( 52, 52) - dw( 30, 52)
jacc( 13, 13) = jacc( 13, 13) - dw( 31, 13)
jacc( 50, 13) = jacc( 50, 13) + dw( 31, 13)
jacc( 52, 13) = jacc( 52, 13) + dw( 31, 13)
jacc( 13, 42) = jacc( 13, 42) - dw( 32, 42)
jacc( 13, 13) = jacc( 13, 13) - dw( 32, 13)
jacc( 42, 42) = jacc( 42, 42) - dw( 32, 42)
jacc( 42, 13) = jacc( 42, 13) - dw( 32, 13)
jacc( 52, 42) = jacc( 52, 42) + dw( 32, 42)
jacc( 52, 13) = jacc( 52, 13) + dw( 32, 13)
jacc( 17, 50) = jacc( 17, 50) + dw( 33, 50)
jacc( 17, 50) = jacc( 17, 50) + dw( 33, 50)
jacc( 50, 50) = jacc( 50, 50)- 0.2000000000000000d+01*dw( 33, 50)
jacc( 50, 50) = jacc( 50, 50)- 0.2000000000000000d+01*dw( 33, 50)
jacc( 17, 50) = jacc( 17, 50) + dw( 34, 50)
jacc( 17, 50) = jacc( 17, 50) + dw( 34, 50)
jacc( 50, 50) = jacc( 50, 50)- 0.2000000000000000d+01*dw( 34, 50)
jacc( 50, 50) = jacc( 50, 50)- 0.2000000000000000d+01*dw( 34, 50)
jacc( 17, 17) = jacc( 17, 17) - dw( 35, 17)
jacc( 42, 17) = jacc( 42, 17)+ 0.2000000000000000d+01*dw( 35, 17)
jacc( 17, 42) = jacc( 17, 42) - dw( 36, 42)
jacc( 17, 17) = jacc( 17, 17) - dw( 36, 17)
jacc( 42, 42) = jacc( 42, 42) - dw( 36, 42)
jacc( 42, 17) = jacc( 42, 17) - dw( 36, 17)
jacc( 50, 42) = jacc( 50, 42) + dw( 36, 42)
jacc( 50, 17) = jacc( 50, 17) + dw( 36, 17)
jacc(  1,  1) = jacc(  1,  1) - dw( 37,  1)
jacc( 42,  1) = jacc( 42,  1) + dw( 37,  1)
jacc( 50,  1) = jacc( 50,  1) + dw( 37,  1)
jacc( 42, 42) = jacc( 42, 42) - dw( 38, 42)
jacc( 50, 42) = jacc( 50, 42) + dw( 38, 42)
jacc( 42, 42) = jacc( 42, 42) - dw( 39, 42)
jacc( 42, 43) = jacc( 42, 43) - dw( 39, 43)
jacc( 43, 42) = jacc( 43, 42) - dw( 39, 42)
jacc( 43, 43) = jacc( 43, 43) - dw( 39, 43)
jacc( 50, 42) = jacc( 50, 42) + dw( 39, 42)
jacc( 50, 43) = jacc( 50, 43) + dw( 39, 43)
jacc( 42, 42) = jacc( 42, 42)- 0.2000000000000000d+01*dw( 40, 42)
jacc( 42, 42) = jacc( 42, 42)- 0.2000000000000000d+01*dw( 40, 42)
jacc( 43, 42) = jacc( 43, 42) + dw( 40, 42)
jacc( 43, 42) = jacc( 43, 42) + dw( 40, 42)
jacc( 17, 42) = jacc( 17, 42) + dw( 41, 42)
jacc( 17, 42) = jacc( 17, 42) + dw( 41, 42)
jacc( 42, 42) = jacc( 42, 42)- 0.2000000000000000d+01*dw( 41, 42)
jacc( 42, 42) = jacc( 42, 42)- 0.2000000000000000d+01*dw( 41, 42)
jacc( 42, 42) = jacc( 42, 42) - dw( 42, 42)
jacc( 42, 50) = jacc( 42, 50) - dw( 42, 50)
jacc( 50, 42) = jacc( 50, 42) - dw( 42, 42)
jacc( 50, 50) = jacc( 50, 50) - dw( 42, 50)
jacc( 42, 50) = jacc( 42, 50) + dw( 43, 50)
jacc( 42, 43) = jacc( 42, 43) + dw( 43, 43)
jacc( 43, 50) = jacc( 43, 50) - dw( 43, 50)
jacc( 43, 43) = jacc( 43, 43) - dw( 43, 43)
jacc( 50, 50) = jacc( 50, 50) - dw( 43, 50)
jacc( 50, 43) = jacc( 50, 43) - dw( 43, 43)
jacc( 17, 17) = jacc( 17, 17) - dw( 44, 17)
jacc( 17, 43) = jacc( 17, 43) - dw( 44, 43)
jacc( 42, 17) = jacc( 42, 17) + dw( 44, 17)
jacc( 42, 43) = jacc( 42, 43) + dw( 44, 43)
jacc( 43, 17) = jacc( 43, 17) - dw( 44, 17)
jacc( 43, 43) = jacc( 43, 43) - dw( 44, 43)
jacc( 50, 17) = jacc( 50, 17) + dw( 44, 17)
jacc( 50, 43) = jacc( 50, 43) + dw( 44, 43)
jacc( 43, 44) = jacc( 43, 44) - dw( 45, 44)
jacc( 43, 43) = jacc( 43, 43) - dw( 45, 43)
jacc( 44, 44) = jacc( 44, 44) - dw( 45, 44)
jacc( 44, 43) = jacc( 44, 43) - dw( 45, 43)
jacc( 52, 44) = jacc( 52, 44) + dw( 45, 44)
jacc( 52, 43) = jacc( 52, 43) + dw( 45, 43)
jacc( 42, 44) = jacc( 42, 44) - dw( 46, 44)
jacc( 42, 42) = jacc( 42, 42) - dw( 46, 42)
jacc( 44, 44) = jacc( 44, 44) - dw( 46, 44)
jacc( 44, 42) = jacc( 44, 42) - dw( 46, 42)
jacc( 50, 44) = jacc( 50, 44) + dw( 46, 44)
jacc( 50, 42) = jacc( 50, 42) + dw( 46, 42)
jacc( 52, 44) = jacc( 52, 44) + dw( 46, 44)
jacc( 52, 42) = jacc( 52, 42) + dw( 46, 42)
jacc( 38, 44) = jacc( 38, 44) + dw( 47, 44)
jacc( 38, 50) = jacc( 38, 50) + dw( 47, 50)
jacc( 44, 44) = jacc( 44, 44) - dw( 47, 44)
jacc( 44, 50) = jacc( 44, 50) - dw( 47, 50)
jacc( 50, 44) = jacc( 50, 44) - dw( 47, 44)
jacc( 50, 50) = jacc( 50, 50) - dw( 47, 50)
jacc( 44, 44) = jacc( 44, 44) - dw( 48, 44)
jacc( 44, 47) = jacc( 44, 47) - dw( 48, 47)
jacc( 47, 44) = jacc( 47, 44) - dw( 48, 44)
jacc( 47, 47) = jacc( 47, 47) - dw( 48, 47)
jacc( 52, 44) = jacc( 52, 44) + dw( 48, 44)
jacc( 52, 47) = jacc( 52, 47) + dw( 48, 47)
jacc( 44, 44) = jacc( 44, 44)- 0.2000000000000000d+01*dw( 49, 44)
jacc( 44, 44) = jacc( 44, 44)- 0.2000000000000000d+01*dw( 49, 44)
jacc( 52, 44) = jacc( 52, 44)+ 0.2000000000000000d+01*dw( 49, 44)
jacc( 52, 44) = jacc( 52, 44)+ 0.2000000000000000d+01*dw( 49, 44)
jacc( 13, 13) = jacc( 13, 13) - dw( 50, 13)
jacc( 42, 13) = jacc( 42, 13)+ 0.3900000000000000d+00*dw( 50, 13)
jacc( 44, 13) = jacc( 44, 13)+ 0.3900000000000000d+00*dw( 50, 13)
jacc( 50, 13) = jacc( 50, 13)+ 0.6100000000000000d+00*dw( 50, 13)
jacc( 52, 13) = jacc( 52, 13)+ 0.6100000000000000d+00*dw( 50, 13)
jacc( 38, 38) = jacc( 38, 38) - dw( 51, 38)
jacc( 42, 38) = jacc( 42, 38) + dw( 51, 38)
jacc( 52, 38) = jacc( 52, 38) + dw( 51, 38)
jacc(  9,  9) = jacc(  9,  9) - dw( 52,  9)
jacc( 44,  9) = jacc( 44,  9) + dw( 52,  9)
jacc( 52,  9) = jacc( 52,  9) + dw( 52,  9)
jacc( 48, 49) = jacc( 48, 49) - dw( 53, 49)
jacc( 48, 48) = jacc( 48, 48) - dw( 53, 48)
jacc( 49, 49) = jacc( 49, 49) - dw( 53, 49)
jacc( 49, 48) = jacc( 49, 48) - dw( 53, 48)
jacc( 52, 49) = jacc( 52, 49) + dw( 53, 49)
jacc( 52, 48) = jacc( 52, 48) + dw( 53, 48)
jacc( 33, 39) = jacc( 33, 39) + dw( 54, 39)
jacc( 33, 48) = jacc( 33, 48) + dw( 54, 48)
jacc( 39, 39) = jacc( 39, 39) - dw( 54, 39)
jacc( 39, 48) = jacc( 39, 48) - dw( 54, 48)
jacc( 48, 39) = jacc( 48, 39) - dw( 54, 39)
jacc( 48, 48) = jacc( 48, 48) - dw( 54, 48)
jacc( 21, 49) = jacc( 21, 49) + dw( 55, 49)
jacc( 21, 50) = jacc( 21, 50) + dw( 55, 50)
jacc( 49, 49) = jacc( 49, 49) - dw( 55, 49)
jacc( 49, 50) = jacc( 49, 50) - dw( 55, 50)
jacc( 50, 49) = jacc( 50, 49) - dw( 55, 49)
jacc( 50, 50) = jacc( 50, 50) - dw( 55, 50)
jacc( 21, 39) = jacc( 21, 39) + dw( 56, 39)
jacc( 21, 50) = jacc( 21, 50) + dw( 56, 50)
jacc( 39, 39) = jacc( 39, 39) - dw( 56, 39)
jacc( 39, 50) = jacc( 39, 50) - dw( 56, 50)
jacc( 50, 39) = jacc( 50, 39) - dw( 56, 39)
jacc( 50, 50) = jacc( 50, 50) - dw( 56, 50)
jacc( 49, 49) = jacc( 49, 49)- 0.2000000000000000d+01*dw( 57, 49)
jacc( 49, 49) = jacc( 49, 49)- 0.2000000000000000d+01*dw( 57, 49)
jacc( 39, 39) = jacc( 39, 39)- 0.2000000000000000d+01*dw( 58, 39)
jacc( 39, 39) = jacc( 39, 39)- 0.2000000000000000d+01*dw( 58, 39)
jacc( 39, 49) = jacc( 39, 49) - dw( 59, 49)
jacc( 39, 39) = jacc( 39, 39) - dw( 59, 39)
jacc( 49, 49) = jacc( 49, 49) - dw( 59, 49)
jacc( 49, 39) = jacc( 49, 39) - dw( 59, 39)
jacc( 33, 33) = jacc( 33, 33) - dw( 60, 33)
jacc( 33, 42) = jacc( 33, 42) - dw( 60, 42)
jacc( 34, 33) = jacc( 34, 33)+ 0.6600000000000000d+00*dw( 60, 33)
jacc( 34, 42) = jacc( 34, 42)+ 0.6600000000000000d+00*dw( 60, 42)
jacc( 35, 33) = jacc( 35, 33)+ 0.3300000000000000d+00*dw( 60, 33)
jacc( 35, 42) = jacc( 35, 42)+ 0.3300000000000000d+00*dw( 60, 42)
jacc( 38, 33) = jacc( 38, 33) + dw( 60, 33)
jacc( 38, 42) = jacc( 38, 42) + dw( 60, 42)
jacc( 41, 33) = jacc( 41, 33)+ 0.3300000000000000d+00*dw( 60, 33)
jacc( 41, 42) = jacc( 41, 42)+ 0.3300000000000000d+00*dw( 60, 42)
jacc( 42, 33) = jacc( 42, 33) - dw( 60, 33)
jacc( 42, 42) = jacc( 42, 42) - dw( 60, 42)
jacc( 45, 33) = jacc( 45, 33)+ 0.3300000000000000d+00*dw( 60, 33)
jacc( 45, 42) = jacc( 45, 42)+ 0.3300000000000000d+00*dw( 60, 42)
jacc( 50, 33) = jacc( 50, 33) + dw( 60, 33)
jacc( 50, 42) = jacc( 50, 42) + dw( 60, 42)
jacc( 33, 33) = jacc( 33, 33) - dw( 61, 33)
jacc( 34, 33) = jacc( 34, 33)+ 0.6600000000000000d+00*dw( 61, 33)
jacc( 35, 33) = jacc( 35, 33)+ 0.3300000000000000d+00*dw( 61, 33)
jacc( 41, 33) = jacc( 41, 33)+ 0.3300000000000000d+00*dw( 61, 33)
jacc( 45, 33) = jacc( 45, 33)+ 0.3300000000000000d+00*dw( 61, 33)
jacc( 50, 33) = jacc( 50, 33) + dw( 61, 33)
jacc( 52, 33) = jacc( 52, 33) + dw( 61, 33)
jacc( 21, 21) = jacc( 21, 21) - dw( 62, 21)
jacc( 21, 42) = jacc( 21, 42) - dw( 62, 42)
jacc( 35, 21) = jacc( 35, 21)+ 0.5000000000000000d+00*dw( 62, 21)
jacc( 35, 42) = jacc( 35, 42)+ 0.5000000000000000d+00*dw( 62, 42)
jacc( 42, 21) = jacc( 42, 21) - dw( 62, 21)
jacc( 42, 42) = jacc( 42, 42) - dw( 62, 42)
jacc( 45, 21) = jacc( 45, 21)+ 0.5000000000000000d+00*dw( 62, 21)
jacc( 45, 42) = jacc( 45, 42)+ 0.5000000000000000d+00*dw( 62, 42)
jacc( 49, 21) = jacc( 49, 21) + dw( 62, 21)
jacc( 49, 42) = jacc( 49, 42) + dw( 62, 42)
jacc( 21, 21) = jacc( 21, 21) - dw( 63, 21)
jacc( 35, 21) = jacc( 35, 21)+ 0.5000000000000000d+00*dw( 63, 21)
jacc( 42, 21) = jacc( 42, 21) + dw( 63, 21)
jacc( 45, 21) = jacc( 45, 21)+ 0.5000000000000000d+00*dw( 63, 21)
jacc( 50, 21) = jacc( 50, 21) + dw( 63, 21)
jacc( 37, 42) = jacc( 37, 42) - dw( 64, 42)
jacc( 37, 37) = jacc( 37, 37) - dw( 64, 37)
jacc( 42, 42) = jacc( 42, 42) - dw( 64, 42)
jacc( 42, 37) = jacc( 42, 37) - dw( 64, 37)
jacc( 50, 42) = jacc( 50, 42) + dw( 64, 42)
jacc( 50, 37) = jacc( 50, 37) + dw( 64, 37)
jacc(  3, 42) = jacc(  3, 42) - dw( 65, 42)
jacc(  3,  3) = jacc(  3,  3) - dw( 65,  3)
jacc( 40, 42) = jacc( 40, 42) + dw( 65, 42)
jacc( 40,  3) = jacc( 40,  3) + dw( 65,  3)
jacc( 42, 42) = jacc( 42, 42) - dw( 65, 42)
jacc( 42,  3) = jacc( 42,  3) - dw( 65,  3)
jacc( 40, 40) = jacc( 40, 40) - dw( 66, 40)
jacc( 40, 48) = jacc( 40, 48) - dw( 66, 48)
jacc( 41, 40) = jacc( 41, 40) + dw( 66, 40)
jacc( 41, 48) = jacc( 41, 48) + dw( 66, 48)
jacc( 48, 40) = jacc( 48, 40) - dw( 66, 40)
jacc( 48, 48) = jacc( 48, 48) - dw( 66, 48)
jacc( 50, 40) = jacc( 50, 40) + dw( 66, 40)
jacc( 50, 48) = jacc( 50, 48) + dw( 66, 48)
jacc( 52, 40) = jacc( 52, 40) + dw( 66, 40)
jacc( 52, 48) = jacc( 52, 48) + dw( 66, 48)
jacc( 19, 40) = jacc( 19, 40) + dw( 67, 40)
jacc( 19, 50) = jacc( 19, 50) + dw( 67, 50)
jacc( 40, 40) = jacc( 40, 40) - dw( 67, 40)
jacc( 40, 50) = jacc( 40, 50) - dw( 67, 50)
jacc( 50, 40) = jacc( 50, 40) - dw( 67, 40)
jacc( 50, 50) = jacc( 50, 50) - dw( 67, 50)
jacc( 11, 40) = jacc( 11, 40)+ 0.6300000000000000d+00*dw( 68, 40)
jacc( 11, 40) = jacc( 11, 40)+ 0.6300000000000000d+00*dw( 68, 40)
jacc( 40, 40) = jacc( 40, 40)- 0.2000000000000000d+01*dw( 68, 40)
jacc( 40, 40) = jacc( 40, 40)- 0.2000000000000000d+01*dw( 68, 40)
jacc( 41, 40) = jacc( 41, 40)+ 0.1370000000000000d+01*dw( 68, 40)
jacc( 41, 40) = jacc( 41, 40)+ 0.1370000000000000d+01*dw( 68, 40)
jacc( 50, 40) = jacc( 50, 40)+ 0.7400000000000000d+00*dw( 68, 40)
jacc( 50, 40) = jacc( 50, 40)+ 0.7400000000000000d+00*dw( 68, 40)
jacc( 19, 19) = jacc( 19, 19) - dw( 69, 19)
jacc( 19, 42) = jacc( 19, 42) - dw( 69, 42)
jacc( 40, 19) = jacc( 40, 19)+ 0.7000000000000000d+00*dw( 69, 19)
jacc( 40, 42) = jacc( 40, 42)+ 0.7000000000000000d+00*dw( 69, 42)
jacc( 42, 19) = jacc( 42, 19) - dw( 69, 19)
jacc( 42, 42) = jacc( 42, 42) - dw( 69, 42)
jacc( 49, 19) = jacc( 49, 19)+ 0.3000000000000000d+00*dw( 69, 19)
jacc( 49, 42) = jacc( 49, 42)+ 0.3000000000000000d+00*dw( 69, 42)
jacc( 50, 19) = jacc( 50, 19)+ 0.3000000000000000d-01*dw( 69, 19)
jacc( 50, 42) = jacc( 50, 42)+ 0.3000000000000000d-01*dw( 69, 42)
jacc( 19, 19) = jacc( 19, 19) - dw( 70, 19)
jacc( 41, 19) = jacc( 41, 19) + dw( 70, 19)
jacc( 42, 19) = jacc( 42, 19) + dw( 70, 19)
jacc( 50, 19) = jacc( 50, 19) + dw( 70, 19)
jacc( 11, 11) = jacc( 11, 11) - dw( 71, 11)
jacc( 11, 42) = jacc( 11, 42) - dw( 71, 42)
jacc( 41, 11) = jacc( 41, 11) + dw( 71, 11)
jacc( 41, 42) = jacc( 41, 42) + dw( 71, 42)
jacc( 42, 11) = jacc( 42, 11) - dw( 71, 11)
jacc( 42, 42) = jacc( 42, 42) - dw( 71, 42)
jacc( 50, 11) = jacc( 50, 11) + dw( 71, 11)
jacc( 50, 42) = jacc( 50, 42) + dw( 71, 42)
jacc( 37, 41) = jacc( 37, 41) + dw( 72, 41)
jacc( 37, 42) = jacc( 37, 42) + dw( 72, 42)
jacc( 41, 41) = jacc( 41, 41) - dw( 72, 41)
jacc( 41, 42) = jacc( 41, 42) - dw( 72, 42)
jacc( 42, 41) = jacc( 42, 41) - dw( 72, 41)
jacc( 42, 42) = jacc( 42, 42) - dw( 72, 42)
jacc( 50, 41) = jacc( 50, 41) + dw( 72, 41)
jacc( 50, 42) = jacc( 50, 42) + dw( 72, 42)
jacc( 37, 41) = jacc( 37, 41) + dw( 73, 41)
jacc( 41, 41) = jacc( 41, 41) - dw( 73, 41)
jacc( 50, 41) = jacc( 50, 41)+ 0.2000000000000000d+01*dw( 73, 41)
jacc( 37, 41) = jacc( 37, 41) + dw( 74, 41)
jacc( 41, 41) = jacc( 41, 41) - dw( 74, 41)
jacc( 37, 41) = jacc( 37, 41) + dw( 75, 41)
jacc( 37, 43) = jacc( 37, 43) + dw( 75, 43)
jacc( 41, 41) = jacc( 41, 41) - dw( 75, 41)
jacc( 41, 43) = jacc( 41, 43) - dw( 75, 43)
jacc( 42, 41) = jacc( 42, 41) + dw( 75, 41)
jacc( 42, 43) = jacc( 42, 43) + dw( 75, 43)
jacc( 43, 41) = jacc( 43, 41) - dw( 75, 41)
jacc( 43, 43) = jacc( 43, 43) - dw( 75, 43)
jacc( 50, 41) = jacc( 50, 41) + dw( 75, 41)
jacc( 50, 43) = jacc( 50, 43) + dw( 75, 43)
jacc( 37, 41) = jacc( 37, 41) + dw( 76, 41)
jacc( 37, 44) = jacc( 37, 44) + dw( 76, 44)
jacc( 38, 41) = jacc( 38, 41) + dw( 76, 41)
jacc( 38, 44) = jacc( 38, 44) + dw( 76, 44)
jacc( 41, 41) = jacc( 41, 41) - dw( 76, 41)
jacc( 41, 44) = jacc( 41, 44) - dw( 76, 44)
jacc( 44, 41) = jacc( 44, 41) - dw( 76, 41)
jacc( 44, 44) = jacc( 44, 44) - dw( 76, 44)
jacc( 50, 41) = jacc( 50, 41) + dw( 76, 41)
jacc( 50, 44) = jacc( 50, 44) + dw( 76, 44)
jacc( 18, 41) = jacc( 18, 41) + dw( 77, 41)
jacc( 18, 50) = jacc( 18, 50) + dw( 77, 50)
jacc( 41, 41) = jacc( 41, 41) - dw( 77, 41)
jacc( 41, 50) = jacc( 41, 50) - dw( 77, 50)
jacc( 50, 41) = jacc( 50, 41) - dw( 77, 41)
jacc( 50, 50) = jacc( 50, 50) - dw( 77, 50)
jacc( 18, 18) = jacc( 18, 18) - dw( 78, 18)
jacc( 41, 18) = jacc( 41, 18) + dw( 78, 18)
jacc( 50, 18) = jacc( 50, 18) + dw( 78, 18)
jacc( 18, 18) = jacc( 18, 18) - dw( 79, 18)
jacc( 18, 48) = jacc( 18, 48) - dw( 79, 48)
jacc( 20, 18) = jacc( 20, 18) + dw( 79, 18)
jacc( 20, 48) = jacc( 20, 48) + dw( 79, 48)
jacc( 48, 18) = jacc( 48, 18) - dw( 79, 18)
jacc( 48, 48) = jacc( 48, 48) - dw( 79, 48)
jacc( 50, 18) = jacc( 50, 18) + dw( 79, 18)
jacc( 50, 48) = jacc( 50, 48) + dw( 79, 48)
jacc( 52, 18) = jacc( 52, 18) + dw( 79, 18)
jacc( 52, 48) = jacc( 52, 48) + dw( 79, 48)
jacc( 18, 18) = jacc( 18, 18) - dw( 80, 18)
jacc( 18, 50) = jacc( 18, 50) - dw( 80, 50)
jacc( 19, 18) = jacc( 19, 18) + dw( 80, 18)
jacc( 19, 50) = jacc( 19, 50) + dw( 80, 50)
jacc( 50, 18) = jacc( 50, 18) - dw( 80, 18)
jacc( 50, 50) = jacc( 50, 50) - dw( 80, 50)
jacc( 20, 20) = jacc( 20, 20) - dw( 81, 20)
jacc( 20, 42) = jacc( 20, 42) - dw( 81, 42)
jacc( 42, 20) = jacc( 42, 20) - dw( 81, 20)
jacc( 42, 42) = jacc( 42, 42) - dw( 81, 42)
jacc( 50, 20) = jacc( 50, 20) + dw( 81, 20)
jacc( 50, 42) = jacc( 50, 42) + dw( 81, 42)
jacc( 42, 45) = jacc( 42, 45) + dw( 82, 45)
jacc( 42, 43) = jacc( 42, 43) + dw( 82, 43)
jacc( 43, 45) = jacc( 43, 45) - dw( 82, 45)
jacc( 43, 43) = jacc( 43, 43) - dw( 82, 43)
jacc( 45, 45) = jacc( 45, 45) - dw( 82, 45)
jacc( 45, 43) = jacc( 45, 43) - dw( 82, 43)
jacc( 51, 45) = jacc( 51, 45) + dw( 82, 45)
jacc( 51, 43) = jacc( 51, 43) + dw( 82, 43)
jacc( 42, 45) = jacc( 42, 45) - dw( 83, 45)
jacc( 42, 42) = jacc( 42, 42) - dw( 83, 42)
jacc( 45, 45) = jacc( 45, 45) - dw( 83, 45)
jacc( 45, 42) = jacc( 45, 42) - dw( 83, 42)
jacc( 51, 45) = jacc( 51, 45) + dw( 83, 45)
jacc( 51, 42) = jacc( 51, 42) + dw( 83, 42)
jacc( 38, 45) = jacc( 38, 45) + dw( 84, 45)
jacc( 38, 44) = jacc( 38, 44) + dw( 84, 44)
jacc( 44, 45) = jacc( 44, 45) - dw( 84, 45)
jacc( 44, 44) = jacc( 44, 44) - dw( 84, 44)
jacc( 45, 45) = jacc( 45, 45) - dw( 84, 45)
jacc( 45, 44) = jacc( 45, 44) - dw( 84, 44)
jacc( 51, 45) = jacc( 51, 45) + dw( 84, 45)
jacc( 51, 44) = jacc( 51, 44) + dw( 84, 44)
jacc( 37, 45) = jacc( 37, 45) + dw( 85, 45)
jacc( 40, 45) = jacc( 40, 45) + dw( 85, 45)
jacc( 45, 45) = jacc( 45, 45) - dw( 85, 45)
jacc( 50, 45) = jacc( 50, 45) + dw( 85, 45)
jacc( 40, 51) = jacc( 40, 51) + dw( 86, 51)
jacc( 40, 48) = jacc( 40, 48) + dw( 86, 48)
jacc( 48, 51) = jacc( 48, 51) - dw( 86, 51)
jacc( 48, 48) = jacc( 48, 48) - dw( 86, 48)
jacc( 51, 51) = jacc( 51, 51) - dw( 86, 51)
jacc( 51, 48) = jacc( 51, 48) - dw( 86, 48)
jacc( 52, 51) = jacc( 52, 51) + dw( 86, 51)
jacc( 52, 48) = jacc( 52, 48) + dw( 86, 48)
jacc( 10, 51) = jacc( 10, 51) + dw( 87, 51)
jacc( 10, 52) = jacc( 10, 52) + dw( 87, 52)
jacc( 51, 51) = jacc( 51, 51) - dw( 87, 51)
jacc( 51, 52) = jacc( 51, 52) - dw( 87, 52)
jacc( 52, 51) = jacc( 52, 51) - dw( 87, 51)
jacc( 52, 52) = jacc( 52, 52) - dw( 87, 52)
jacc( 10, 10) = jacc( 10, 10) - dw( 88, 10)
jacc( 51, 10) = jacc( 51, 10) + dw( 88, 10)
jacc( 52, 10) = jacc( 52, 10) + dw( 88, 10)
jacc( 10, 10) = jacc( 10, 10) - dw( 89, 10)
jacc( 51, 10) = jacc( 51, 10) + dw( 89, 10)
jacc( 52, 10) = jacc( 52, 10) + dw( 89, 10)
jacc( 22, 51) = jacc( 22, 51)+ 0.8000000000000000d+00*dw( 90, 51)
jacc( 22, 50) = jacc( 22, 50)+ 0.8000000000000000d+00*dw( 90, 50)
jacc( 36, 51) = jacc( 36, 51)+ 0.2000000000000000d+00*dw( 90, 51)
jacc( 36, 50) = jacc( 36, 50)+ 0.2000000000000000d+00*dw( 90, 50)
jacc( 47, 51) = jacc( 47, 51)+ 0.2000000000000000d+00*dw( 90, 51)
jacc( 47, 50) = jacc( 47, 50)+ 0.2000000000000000d+00*dw( 90, 50)
jacc( 50, 51) = jacc( 50, 51) - dw( 90, 51)
jacc( 50, 50) = jacc( 50, 50) - dw( 90, 50)
jacc( 51, 51) = jacc( 51, 51) - dw( 90, 51)
jacc( 51, 50) = jacc( 51, 50) - dw( 90, 50)
jacc( 36, 51) = jacc( 36, 51)+ 0.1000000000000000d+00*dw( 91, 51)
jacc( 36, 40) = jacc( 36, 40)+ 0.1000000000000000d+00*dw( 91, 40)
jacc( 40, 51) = jacc( 40, 51)- 0.9999999999999998d-01*dw( 91, 51)
jacc( 40, 40) = jacc( 40, 40)- 0.9999999999999998d-01*dw( 91, 40)
jacc( 41, 51) = jacc( 41, 51) + dw( 91, 51)
jacc( 41, 40) = jacc( 41, 40) + dw( 91, 40)
jacc( 50, 51) = jacc( 50, 51)+ 0.9000000000000000d+00*dw( 91, 51)
jacc( 50, 40) = jacc( 50, 40)+ 0.9000000000000000d+00*dw( 91, 40)
jacc( 51, 51) = jacc( 51, 51) - dw( 91, 51)
jacc( 51, 40) = jacc( 51, 40) - dw( 91, 40)
jacc( 36, 51) = jacc( 36, 51)+ 0.1000000000000000d+00*dw( 92, 51)
jacc( 36, 49) = jacc( 36, 49)+ 0.1000000000000000d+00*dw( 92, 49)
jacc( 40, 51) = jacc( 40, 51)+ 0.9000000000000000d+00*dw( 92, 51)
jacc( 40, 49) = jacc( 40, 49)+ 0.9000000000000000d+00*dw( 92, 49)
jacc( 49, 51) = jacc( 49, 51) - dw( 92, 51)
jacc( 49, 49) = jacc( 49, 49) - dw( 92, 49)
jacc( 51, 51) = jacc( 51, 51) - dw( 92, 51)
jacc( 51, 49) = jacc( 51, 49) - dw( 92, 49)
jacc( 40, 51) = jacc( 40, 51)+ 0.2000000000000000d+01*dw( 93, 51)
jacc( 40, 51) = jacc( 40, 51)+ 0.2000000000000000d+01*dw( 93, 51)
jacc( 51, 51) = jacc( 51, 51)- 0.2000000000000000d+01*dw( 93, 51)
jacc( 51, 51) = jacc( 51, 51)- 0.2000000000000000d+01*dw( 93, 51)
jacc( 22, 22) = jacc( 22, 22) - dw( 94, 22)
jacc( 22, 42) = jacc( 22, 42) - dw( 94, 42)
jacc( 42, 22) = jacc( 42, 22) - dw( 94, 22)
jacc( 42, 42) = jacc( 42, 42) - dw( 94, 42)
jacc( 51, 22) = jacc( 51, 22) + dw( 94, 22)
jacc( 51, 42) = jacc( 51, 42) + dw( 94, 42)
jacc( 22, 22) = jacc( 22, 22) - dw( 95, 22)
jacc( 40, 22) = jacc( 40, 22) + dw( 95, 22)
jacc( 42, 22) = jacc( 42, 22) + dw( 95, 22)
jacc( 36, 36) = jacc( 36, 36) - dw( 96, 36)
jacc( 36, 42) = jacc( 36, 42) - dw( 96, 42)
jacc( 40, 36) = jacc( 40, 36) + dw( 96, 36)
jacc( 40, 42) = jacc( 40, 42) + dw( 96, 42)
jacc( 42, 36) = jacc( 42, 36) - dw( 96, 36)
jacc( 42, 42) = jacc( 42, 42) - dw( 96, 42)
jacc( 35, 35) = jacc( 35, 35) - dw( 97, 35)
jacc( 35, 43) = jacc( 35, 43) - dw( 97, 43)
jacc( 42, 35) = jacc( 42, 35) + dw( 97, 35)
jacc( 42, 43) = jacc( 42, 43) + dw( 97, 43)
jacc( 43, 35) = jacc( 43, 35) - dw( 97, 35)
jacc( 43, 43) = jacc( 43, 43) - dw( 97, 43)
jacc( 46, 35) = jacc( 46, 35) + dw( 97, 35)
jacc( 46, 43) = jacc( 46, 43) + dw( 97, 43)
jacc( 35, 35) = jacc( 35, 35) - dw( 98, 35)
jacc( 35, 42) = jacc( 35, 42) - dw( 98, 42)
jacc( 42, 35) = jacc( 42, 35) - dw( 98, 35)
jacc( 42, 42) = jacc( 42, 42) - dw( 98, 42)
jacc( 46, 35) = jacc( 46, 35) + dw( 98, 35)
jacc( 46, 42) = jacc( 46, 42) + dw( 98, 42)
jacc( 35, 35) = jacc( 35, 35) - dw( 99, 35)
jacc( 35, 44) = jacc( 35, 44) - dw( 99, 44)
jacc( 38, 35) = jacc( 38, 35) + dw( 99, 35)
jacc( 38, 44) = jacc( 38, 44) + dw( 99, 44)
jacc( 44, 35) = jacc( 44, 35) - dw( 99, 35)
jacc( 44, 44) = jacc( 44, 44) - dw( 99, 44)
jacc( 46, 35) = jacc( 46, 35) + dw( 99, 35)
jacc( 46, 44) = jacc( 46, 44) + dw( 99, 44)
jacc( 35, 35) = jacc( 35, 35) - dw(100, 35)
jacc( 37, 35) = jacc( 37, 35) + dw(100, 35)
jacc( 40, 35) = jacc( 40, 35) + dw(100, 35)
jacc( 50, 35) = jacc( 50, 35) + dw(100, 35)
jacc( 45, 46) = jacc( 45, 46) + dw(101, 46)
jacc( 45, 48) = jacc( 45, 48) + dw(101, 48)
jacc( 46, 46) = jacc( 46, 46) - dw(101, 46)
jacc( 46, 48) = jacc( 46, 48) - dw(101, 48)
jacc( 48, 46) = jacc( 48, 46) - dw(101, 46)
jacc( 48, 48) = jacc( 48, 48) - dw(101, 48)
jacc( 49, 46) = jacc( 49, 46) + dw(101, 46)
jacc( 49, 48) = jacc( 49, 48) + dw(101, 48)
jacc( 50, 46) = jacc( 50, 46) + dw(101, 46)
jacc( 50, 48) = jacc( 50, 48) + dw(101, 48)
jacc( 52, 46) = jacc( 52, 46) + dw(101, 46)
jacc( 52, 48) = jacc( 52, 48) + dw(101, 48)
jacc( 14, 46) = jacc( 14, 46) + dw(102, 46)
jacc( 14, 52) = jacc( 14, 52) + dw(102, 52)
jacc( 46, 46) = jacc( 46, 46) - dw(102, 46)
jacc( 46, 52) = jacc( 46, 52) - dw(102, 52)
jacc( 52, 46) = jacc( 52, 46) - dw(102, 46)
jacc( 52, 52) = jacc( 52, 52) - dw(102, 52)
jacc( 14, 14) = jacc( 14, 14) - dw(103, 14)
jacc( 46, 14) = jacc( 46, 14) + dw(103, 14)
jacc( 52, 14) = jacc( 52, 14) + dw(103, 14)
jacc( 14, 14) = jacc( 14, 14) - dw(104, 14)
jacc( 46, 14) = jacc( 46, 14) + dw(104, 14)
jacc( 52, 14) = jacc( 52, 14) + dw(104, 14)
jacc( 14, 14) = jacc( 14, 14) - dw(105, 14)
jacc( 14, 42) = jacc( 14, 42) - dw(105, 42)
jacc( 42, 14) = jacc( 42, 14) - dw(105, 14)
jacc( 42, 42) = jacc( 42, 42) - dw(105, 42)
jacc( 45, 14) = jacc( 45, 14) + dw(105, 14)
jacc( 45, 42) = jacc( 45, 42) + dw(105, 42)
jacc( 52, 14) = jacc( 52, 14) + dw(105, 14)
jacc( 52, 42) = jacc( 52, 42) + dw(105, 42)
jacc( 22, 46) = jacc( 22, 46)+ 0.8000000000000000d+00*dw(106, 46)
jacc( 22, 50) = jacc( 22, 50)+ 0.8000000000000000d+00*dw(106, 50)
jacc( 36, 46) = jacc( 36, 46)+ 0.2000000000000000d+00*dw(106, 46)
jacc( 36, 50) = jacc( 36, 50)+ 0.2000000000000000d+00*dw(106, 50)
jacc( 46, 46) = jacc( 46, 46) - dw(106, 46)
jacc( 46, 50) = jacc( 46, 50) - dw(106, 50)
jacc( 47, 46) = jacc( 47, 46)+ 0.2000000000000000d+00*dw(106, 46)
jacc( 47, 50) = jacc( 47, 50)+ 0.2000000000000000d+00*dw(106, 50)
jacc( 50, 46) = jacc( 50, 46) - dw(106, 46)
jacc( 50, 50) = jacc( 50, 50) - dw(106, 50)
jacc( 36, 51) = jacc( 36, 51)+ 0.1000000000000000d+00*dw(107, 51)
jacc( 36, 40) = jacc( 36, 40)+ 0.1000000000000000d+00*dw(107, 40)
jacc( 40, 51) = jacc( 40, 51) - dw(107, 51)
jacc( 40, 40) = jacc( 40, 40) - dw(107, 40)
jacc( 41, 51) = jacc( 41, 51)+ 0.1000000000000000d+00*dw(107, 51)
jacc( 41, 40) = jacc( 41, 40)+ 0.1000000000000000d+00*dw(107, 40)
jacc( 45, 51) = jacc( 45, 51)+ 0.9000000000000000d+00*dw(107, 51)
jacc( 45, 40) = jacc( 45, 40)+ 0.9000000000000000d+00*dw(107, 40)
jacc( 49, 51) = jacc( 49, 51)+ 0.9000000000000000d+00*dw(107, 51)
jacc( 49, 40) = jacc( 49, 40)+ 0.9000000000000000d+00*dw(107, 40)
jacc( 50, 51) = jacc( 50, 51) + dw(107, 51)
jacc( 50, 40) = jacc( 50, 40) + dw(107, 40)
jacc( 51, 51) = jacc( 51, 51) - dw(107, 51)
jacc( 51, 40) = jacc( 51, 40) - dw(107, 40)
jacc( 36, 51) = jacc( 36, 51)+ 0.1000000000000000d+00*dw(108, 51)
jacc( 36, 49) = jacc( 36, 49)+ 0.1000000000000000d+00*dw(108, 49)
jacc( 45, 51) = jacc( 45, 51)+ 0.9000000000000000d+00*dw(108, 51)
jacc( 45, 49) = jacc( 45, 49)+ 0.9000000000000000d+00*dw(108, 49)
jacc( 49, 51) = jacc( 49, 51) - dw(108, 51)
jacc( 49, 49) = jacc( 49, 49) - dw(108, 49)
jacc( 51, 51) = jacc( 51, 51) - dw(108, 51)
jacc( 51, 49) = jacc( 51, 49) - dw(108, 49)
jacc( 45, 46) = jacc( 45, 46)+ 0.2000000000000000d+01*dw(109, 46)
jacc( 45, 46) = jacc( 45, 46)+ 0.2000000000000000d+01*dw(109, 46)
jacc( 46, 46) = jacc( 46, 46)- 0.2000000000000000d+01*dw(109, 46)
jacc( 46, 46) = jacc( 46, 46)- 0.2000000000000000d+01*dw(109, 46)
jacc( 49, 46) = jacc( 49, 46)+ 0.2000000000000000d+01*dw(109, 46)
jacc( 49, 46) = jacc( 49, 46)+ 0.2000000000000000d+01*dw(109, 46)
jacc( 50, 46) = jacc( 50, 46)+ 0.2000000000000000d+01*dw(109, 46)
jacc( 50, 46) = jacc( 50, 46)+ 0.2000000000000000d+01*dw(109, 46)
jacc( 40, 46) = jacc( 40, 46) + dw(110, 46)
jacc( 40, 51) = jacc( 40, 51) + dw(110, 51)
jacc( 45, 46) = jacc( 45, 46) + dw(110, 46)
jacc( 45, 51) = jacc( 45, 51) + dw(110, 51)
jacc( 46, 46) = jacc( 46, 46) - dw(110, 46)
jacc( 46, 51) = jacc( 46, 51) - dw(110, 51)
jacc( 49, 46) = jacc( 49, 46) + dw(110, 46)
jacc( 49, 51) = jacc( 49, 51) + dw(110, 51)
jacc( 50, 46) = jacc( 50, 46) + dw(110, 46)
jacc( 50, 51) = jacc( 50, 51) + dw(110, 51)
jacc( 51, 46) = jacc( 51, 46) - dw(110, 46)
jacc( 51, 51) = jacc( 51, 51) - dw(110, 51)
jacc( 16, 34) = jacc( 16, 34)+ 0.7600000000000000d+00*dw(111, 34)
jacc( 16, 42) = jacc( 16, 42)+ 0.7600000000000000d+00*dw(111, 42)
jacc( 34, 34) = jacc( 34, 34)- 0.8900000000000000d+00*dw(111, 34)
jacc( 34, 42) = jacc( 34, 42)- 0.8900000000000000d+00*dw(111, 42)
jacc( 35, 34) = jacc( 35, 34)+ 0.5000000000000000d-01*dw(111, 34)
jacc( 35, 42) = jacc( 35, 42)+ 0.5000000000000000d-01*dw(111, 42)
jacc( 39, 34) = jacc( 39, 34)+ 0.1300000000000000d+00*dw(111, 34)
jacc( 39, 42) = jacc( 39, 42)+ 0.1300000000000000d+00*dw(111, 42)
jacc( 42, 34) = jacc( 42, 34) - dw(111, 34)
jacc( 42, 42) = jacc( 42, 42) - dw(111, 42)
jacc( 45, 34) = jacc( 45, 34)+ 0.6000000000000000d-01*dw(111, 34)
jacc( 45, 42) = jacc( 45, 42)+ 0.6000000000000000d-01*dw(111, 42)
jacc( 49, 34) = jacc( 49, 34)+ 0.8700000000000000d+00*dw(111, 34)
jacc( 49, 42) = jacc( 49, 42)+ 0.8700000000000000d+00*dw(111, 42)
jacc( 50, 34) = jacc( 50, 34)+ 0.1100000000000000d+00*dw(111, 34)
jacc( 50, 42) = jacc( 50, 42)+ 0.1100000000000000d+00*dw(111, 42)
jacc( 16, 16) = jacc( 16, 16)- 0.9800000000000000d+00*dw(112, 16)
jacc( 34, 16) = jacc( 34, 16)+ 0.2100000000000000d+01*dw(112, 16)
jacc( 35, 16) = jacc( 35, 16)+ 0.5000000000000000d+00*dw(112, 16)
jacc( 39, 16) = jacc( 39, 16)+ 0.4000000000000000d-01*dw(112, 16)
jacc( 45, 16) = jacc( 45, 16)+ 0.6000000000000000d+00*dw(112, 16)
jacc( 49, 16) = jacc( 49, 16)+ 0.9600000000000000d+00*dw(112, 16)
jacc( 50, 16) = jacc( 50, 16)+ 0.9399999999999999d+00*dw(112, 16)
jacc( 16, 16) = jacc( 16, 16) - dw(113, 16)
jacc( 50, 16) = jacc( 50, 16) + dw(113, 16)
jacc( 16, 16) = jacc( 16, 16) - dw(114, 16)
jacc( 16, 52) = jacc( 16, 52) - dw(114, 52)
jacc( 33, 16) = jacc( 33, 16) + dw(114, 16)
jacc( 33, 52) = jacc( 33, 52) + dw(114, 52)
jacc( 52, 16) = jacc( 52, 16) - dw(114, 16)
jacc( 52, 52) = jacc( 52, 52) - dw(114, 52)
jacc( 24, 43) = jacc( 24, 43) - dw(115, 43)
jacc( 24, 24) = jacc( 24, 24) - dw(115, 24)
jacc( 34, 43) = jacc( 34, 43)+ 0.2000000000000000d+00*dw(115, 43)
jacc( 34, 24) = jacc( 34, 24)+ 0.2000000000000000d+00*dw(115, 24)
jacc( 35, 43) = jacc( 35, 43)+ 0.3000000000000000d+00*dw(115, 43)
jacc( 35, 24) = jacc( 35, 24)+ 0.3000000000000000d+00*dw(115, 24)
jacc( 37, 43) = jacc( 37, 43)+ 0.2000000000000000d+00*dw(115, 43)
jacc( 37, 24) = jacc( 37, 24)+ 0.2000000000000000d+00*dw(115, 24)
jacc( 39, 43) = jacc( 39, 43)+ 0.1000000000000000d-01*dw(115, 43)
jacc( 39, 24) = jacc( 39, 24)+ 0.1000000000000000d-01*dw(115, 24)
jacc( 41, 43) = jacc( 41, 43)+ 0.2000000000000000d+00*dw(115, 43)
jacc( 41, 24) = jacc( 41, 24)+ 0.2000000000000000d+00*dw(115, 24)
jacc( 42, 43) = jacc( 42, 43)+ 0.1000000000000000d+00*dw(115, 43)
jacc( 42, 24) = jacc( 42, 24)+ 0.1000000000000000d+00*dw(115, 24)
jacc( 43, 43) = jacc( 43, 43) - dw(115, 43)
jacc( 43, 24) = jacc( 43, 24) - dw(115, 24)
jacc( 45, 43) = jacc( 45, 43)+ 0.2000000000000000d+00*dw(115, 43)
jacc( 45, 24) = jacc( 45, 24)+ 0.2000000000000000d+00*dw(115, 24)
jacc( 49, 43) = jacc( 49, 43)+ 0.2000000000000000d+00*dw(115, 43)
jacc( 49, 24) = jacc( 49, 24)+ 0.2000000000000000d+00*dw(115, 24)
jacc( 50, 43) = jacc( 50, 43)+ 0.3000000000000000d+00*dw(115, 43)
jacc( 50, 24) = jacc( 50, 24)+ 0.3000000000000000d+00*dw(115, 24)
jacc( 24, 42) = jacc( 24, 42) - dw(116, 42)
jacc( 24, 24) = jacc( 24, 24) - dw(116, 24)
jacc( 34, 42) = jacc( 34, 42)+ 0.7000000000000000d+00*dw(116, 42)
jacc( 34, 24) = jacc( 34, 24)+ 0.7000000000000000d+00*dw(116, 24)
jacc( 35, 42) = jacc( 35, 42)+ 0.6200000000000000d+00*dw(116, 42)
jacc( 35, 24) = jacc( 35, 24)+ 0.6200000000000000d+00*dw(116, 24)
jacc( 41, 42) = jacc( 41, 42)+ 0.8000000000000000d+00*dw(116, 42)
jacc( 41, 24) = jacc( 41, 24)+ 0.8000000000000000d+00*dw(116, 24)
jacc( 42, 42) = jacc( 42, 42) - dw(116, 42)
jacc( 42, 24) = jacc( 42, 24) - dw(116, 24)
jacc( 45, 42) = jacc( 45, 42)+ 0.3300000000000000d+00*dw(116, 42)
jacc( 45, 24) = jacc( 45, 24)+ 0.3300000000000000d+00*dw(116, 24)
jacc( 49, 42) = jacc( 49, 42)+ 0.8000000000000000d+00*dw(116, 42)
jacc( 49, 24) = jacc( 49, 24)+ 0.8000000000000000d+00*dw(116, 24)
jacc( 50, 42) = jacc( 50, 42)+ 0.9500000000000000d+00*dw(116, 42)
jacc( 50, 24) = jacc( 50, 24)+ 0.9500000000000000d+00*dw(116, 24)
jacc( 24, 47) = jacc( 24, 47) - dw(117, 47)
jacc( 24, 24) = jacc( 24, 24) - dw(117, 24)
jacc( 34, 47) = jacc( 34, 47) + dw(117, 47)
jacc( 34, 24) = jacc( 34, 24) + dw(117, 24)
jacc( 35, 47) = jacc( 35, 47)+ 0.3200000000000000d+00*dw(117, 47)
jacc( 35, 24) = jacc( 35, 24)+ 0.3200000000000000d+00*dw(117, 24)
jacc( 37, 47) = jacc( 37, 47)+ 0.3300000000000000d+00*dw(117, 47)
jacc( 37, 24) = jacc( 37, 24)+ 0.3300000000000000d+00*dw(117, 24)
jacc( 41, 47) = jacc( 41, 47)+ 0.7400000000000000d+00*dw(117, 47)
jacc( 41, 24) = jacc( 41, 24)+ 0.7400000000000000d+00*dw(117, 24)
jacc( 42, 47) = jacc( 42, 47)+ 0.1000000000000000d+00*dw(117, 47)
jacc( 42, 24) = jacc( 42, 24)+ 0.1000000000000000d+00*dw(117, 24)
jacc( 45, 47) = jacc( 45, 47)+ 0.1800000000000000d+00*dw(117, 47)
jacc( 45, 24) = jacc( 45, 24)+ 0.1800000000000000d+00*dw(117, 24)
jacc( 47, 47) = jacc( 47, 47) - dw(117, 47)
jacc( 47, 24) = jacc( 47, 24) - dw(117, 24)
jacc( 49, 47) = jacc( 49, 47)+ 0.2200000000000000d+00*dw(117, 47)
jacc( 49, 24) = jacc( 49, 24)+ 0.2200000000000000d+00*dw(117, 24)
jacc( 50, 47) = jacc( 50, 47)+ 0.4400000000000000d+00*dw(117, 47)
jacc( 50, 24) = jacc( 50, 24)+ 0.4400000000000000d+00*dw(117, 24)
jacc( 24, 44) = jacc( 24, 44) - dw(118, 44)
jacc( 24, 24) = jacc( 24, 24) - dw(118, 24)
jacc( 34, 44) = jacc( 34, 44) + dw(118, 44)
jacc( 34, 24) = jacc( 34, 24) + dw(118, 24)
jacc( 35, 44) = jacc( 35, 44)+ 0.5600000000000001d+00*dw(118, 44)
jacc( 35, 24) = jacc( 35, 24)+ 0.5600000000000001d+00*dw(118, 24)
jacc( 39, 44) = jacc( 39, 44)+ 0.9000000000000000d-01*dw(118, 44)
jacc( 39, 24) = jacc( 39, 24)+ 0.9000000000000000d-01*dw(118, 24)
jacc( 41, 44) = jacc( 41, 44) + dw(118, 44)
jacc( 41, 24) = jacc( 41, 24) + dw(118, 24)
jacc( 44, 44) = jacc( 44, 44) - dw(118, 44)
jacc( 44, 24) = jacc( 44, 24) - dw(118, 24)
jacc( 45, 44) = jacc( 45, 44)+ 0.3500000000000000d+00*dw(118, 44)
jacc( 45, 24) = jacc( 45, 24)+ 0.3500000000000000d+00*dw(118, 24)
jacc( 49, 44) = jacc( 49, 44)+ 0.9100000000000000d+00*dw(118, 44)
jacc( 49, 24) = jacc( 49, 24)+ 0.9100000000000000d+00*dw(118, 24)
jacc( 52, 44) = jacc( 52, 44) + dw(118, 44)
jacc( 52, 24) = jacc( 52, 24) + dw(118, 24)
jacc( 23, 43) = jacc( 23, 43) - dw(119, 43)
jacc( 23, 23) = jacc( 23, 23) - dw(119, 23)
jacc( 37, 43) = jacc( 37, 43) + dw(119, 43)
jacc( 37, 23) = jacc( 37, 23) + dw(119, 23)
jacc( 41, 43) = jacc( 41, 43) + dw(119, 43)
jacc( 41, 23) = jacc( 41, 23) + dw(119, 23)
jacc( 42, 43) = jacc( 42, 43)+ 0.3000000000000000d+00*dw(119, 43)
jacc( 42, 23) = jacc( 42, 23)+ 0.3000000000000000d+00*dw(119, 23)
jacc( 43, 43) = jacc( 43, 43) - dw(119, 43)
jacc( 43, 23) = jacc( 43, 23) - dw(119, 23)
jacc( 49, 43) = jacc( 49, 43)+ 0.7000000000000000d+00*dw(119, 43)
jacc( 49, 23) = jacc( 49, 23)+ 0.7000000000000000d+00*dw(119, 23)
jacc( 50, 43) = jacc( 50, 43)+ 0.1700000000000000d+01*dw(119, 43)
jacc( 50, 23) = jacc( 50, 23)+ 0.1700000000000000d+01*dw(119, 23)
jacc( 23, 42) = jacc( 23, 42) - dw(120, 42)
jacc( 23, 23) = jacc( 23, 23) - dw(120, 23)
jacc( 35, 42) = jacc( 35, 42)+ 0.2200000000000000d+00*dw(120, 42)
jacc( 35, 23) = jacc( 35, 23)+ 0.2200000000000000d+00*dw(120, 23)
jacc( 41, 42) = jacc( 41, 42)+ 0.1560000000000000d+01*dw(120, 42)
jacc( 41, 23) = jacc( 41, 23)+ 0.1560000000000000d+01*dw(120, 23)
jacc( 42, 42) = jacc( 42, 42) - dw(120, 42)
jacc( 42, 23) = jacc( 42, 23) - dw(120, 23)
jacc( 49, 42) = jacc( 49, 42) + dw(120, 42)
jacc( 49, 23) = jacc( 49, 23) + dw(120, 23)
jacc( 50, 42) = jacc( 50, 42) + dw(120, 42)
jacc( 50, 23) = jacc( 50, 23) + dw(120, 23)
jacc( 20, 47) = jacc( 20, 47)+ 0.3700000000000000d+00*dw(121, 47)
jacc( 20, 23) = jacc( 20, 23)+ 0.3700000000000000d+00*dw(121, 23)
jacc( 23, 47) = jacc( 23, 47) - dw(121, 47)
jacc( 23, 23) = jacc( 23, 23) - dw(121, 23)
jacc( 37, 47) = jacc( 37, 47)+ 0.6300000000000000d+00*dw(121, 47)
jacc( 37, 23) = jacc( 37, 23)+ 0.6300000000000000d+00*dw(121, 23)
jacc( 41, 47) = jacc( 41, 47) + dw(121, 47)
jacc( 41, 23) = jacc( 41, 23) + dw(121, 23)
jacc( 42, 47) = jacc( 42, 47)+ 0.1300000000000000d+00*dw(121, 47)
jacc( 42, 23) = jacc( 42, 23)+ 0.1300000000000000d+00*dw(121, 23)
jacc( 47, 47) = jacc( 47, 47) - dw(121, 47)
jacc( 47, 23) = jacc( 47, 23) - dw(121, 23)
jacc( 50, 47) = jacc( 50, 47)+ 0.1300000000000000d+00*dw(121, 47)
jacc( 50, 23) = jacc( 50, 23)+ 0.1300000000000000d+00*dw(121, 23)
jacc( 23, 44) = jacc( 23, 44) - dw(122, 44)
jacc( 23, 23) = jacc( 23, 23) - dw(122, 23)
jacc( 41, 44) = jacc( 41, 44)+ 0.2000000000000000d+01*dw(122, 44)
jacc( 41, 23) = jacc( 41, 23)+ 0.2000000000000000d+01*dw(122, 23)
jacc( 44, 44) = jacc( 44, 44) - dw(122, 44)
jacc( 44, 23) = jacc( 44, 23) - dw(122, 23)
jacc( 49, 44) = jacc( 49, 44) + dw(122, 44)
jacc( 49, 23) = jacc( 49, 23) + dw(122, 23)
jacc( 52, 44) = jacc( 52, 44) + dw(122, 44)
jacc( 52, 23) = jacc( 52, 23) + dw(122, 23)
jacc( 25, 25) = jacc( 25, 25) - dw(123, 25)
jacc( 25, 43) = jacc( 25, 43) - dw(123, 43)
jacc( 34, 25) = jacc( 34, 25)+ 0.1000000000000000d+00*dw(123, 25)
jacc( 34, 43) = jacc( 34, 43)+ 0.1000000000000000d+00*dw(123, 43)
jacc( 35, 25) = jacc( 35, 25)+ 0.6600000000000000d+00*dw(123, 25)
jacc( 35, 43) = jacc( 35, 43)+ 0.6600000000000000d+00*dw(123, 43)
jacc( 37, 25) = jacc( 37, 25)+ 0.1000000000000000d+00*dw(123, 25)
jacc( 37, 43) = jacc( 37, 43)+ 0.1000000000000000d+00*dw(123, 43)
jacc( 43, 25) = jacc( 43, 25) - dw(123, 25)
jacc( 43, 43) = jacc( 43, 43) - dw(123, 43)
jacc( 45, 25) = jacc( 45, 25)+ 0.1240000000000000d+01*dw(123, 25)
jacc( 45, 43) = jacc( 45, 43)+ 0.1240000000000000d+01*dw(123, 43)
jacc( 49, 25) = jacc( 49, 25)+ 0.1000000000000000d+00*dw(123, 25)
jacc( 49, 43) = jacc( 49, 43)+ 0.1000000000000000d+00*dw(123, 43)
jacc( 50, 25) = jacc( 50, 25)+ 0.1000000000000000d+00*dw(123, 25)
jacc( 50, 43) = jacc( 50, 43)+ 0.1000000000000000d+00*dw(123, 43)
jacc( 25, 25) = jacc( 25, 25) - dw(124, 25)
jacc( 25, 42) = jacc( 25, 42) - dw(124, 42)
jacc( 35, 25) = jacc( 35, 25)+ 0.7000000000000000d+00*dw(124, 25)
jacc( 35, 42) = jacc( 35, 42)+ 0.7000000000000000d+00*dw(124, 42)
jacc( 42, 25) = jacc( 42, 25) - dw(124, 25)
jacc( 42, 42) = jacc( 42, 42) - dw(124, 42)
jacc( 45, 25) = jacc( 45, 25)+ 0.1300000000000000d+01*dw(124, 25)
jacc( 45, 42) = jacc( 45, 42)+ 0.1300000000000000d+01*dw(124, 42)
jacc( 49, 25) = jacc( 49, 25) + dw(124, 25)
jacc( 49, 42) = jacc( 49, 42) + dw(124, 42)
jacc( 50, 25) = jacc( 50, 25) + dw(124, 25)
jacc( 50, 42) = jacc( 50, 42) + dw(124, 42)
jacc( 25, 25) = jacc( 25, 25) - dw(125, 25)
jacc( 25, 47) = jacc( 25, 47) - dw(125, 47)
jacc( 35, 25) = jacc( 35, 25)+ 0.3500000000000000d+00*dw(125, 25)
jacc( 35, 47) = jacc( 35, 47)+ 0.3500000000000000d+00*dw(125, 47)
jacc( 37, 25) = jacc( 37, 25)+ 0.2500000000000000d+00*dw(125, 25)
jacc( 37, 47) = jacc( 37, 47)+ 0.2500000000000000d+00*dw(125, 47)
jacc( 41, 25) = jacc( 41, 25)+ 0.2500000000000000d+00*dw(125, 25)
jacc( 41, 47) = jacc( 41, 47)+ 0.2500000000000000d+00*dw(125, 47)
jacc( 42, 25) = jacc( 42, 25)+ 0.5000000000000000d+00*dw(125, 25)
jacc( 42, 47) = jacc( 42, 47)+ 0.5000000000000000d+00*dw(125, 47)
jacc( 43, 25) = jacc( 43, 25)+ 0.5000000000000000d+00*dw(125, 25)
jacc( 43, 47) = jacc( 43, 47)+ 0.5000000000000000d+00*dw(125, 47)
jacc( 45, 25) = jacc( 45, 25)+ 0.6500000000000000d+00*dw(125, 25)
jacc( 45, 47) = jacc( 45, 47)+ 0.6500000000000000d+00*dw(125, 47)
jacc( 47, 25) = jacc( 47, 25) - dw(125, 25)
jacc( 47, 47) = jacc( 47, 47) - dw(125, 47)
jacc( 50, 25) = jacc( 50, 25)+ 0.5000000000000000d+00*dw(125, 25)
jacc( 50, 47) = jacc( 50, 47)+ 0.5000000000000000d+00*dw(125, 47)
jacc( 25, 25) = jacc( 25, 25) - dw(126, 25)
jacc( 25, 44) = jacc( 25, 44) - dw(126, 44)
jacc( 35, 25) = jacc( 35, 25)+ 0.6400000000000000d+00*dw(126, 25)
jacc( 35, 44) = jacc( 35, 44)+ 0.6400000000000000d+00*dw(126, 44)
jacc( 44, 25) = jacc( 44, 25) - dw(126, 25)
jacc( 44, 44) = jacc( 44, 44) - dw(126, 44)
jacc( 45, 25) = jacc( 45, 25)+ 0.1180000000000000d+01*dw(126, 25)
jacc( 45, 44) = jacc( 45, 44)+ 0.1180000000000000d+01*dw(126, 44)
jacc( 50, 25) = jacc( 50, 25) + dw(126, 25)
jacc( 50, 44) = jacc( 50, 44) + dw(126, 44)
jacc( 52, 25) = jacc( 52, 25) + dw(126, 25)
jacc( 52, 44) = jacc( 52, 44) + dw(126, 44)
jacc(  5,  5) = jacc(  5,  5) - dw(127,  5)
jacc(  5, 42) = jacc(  5, 42) - dw(127, 42)
jacc( 12,  5) = jacc( 12,  5)+ 0.5600000000000001d+00*dw(127,  5)
jacc( 12, 42) = jacc( 12, 42)+ 0.5600000000000001d+00*dw(127, 42)
jacc( 27,  5) = jacc( 27,  5)+ 0.3600000000000000d+00*dw(127,  5)
jacc( 27, 42) = jacc( 27, 42)+ 0.3600000000000000d+00*dw(127, 42)
jacc( 42,  5) = jacc( 42,  5) - dw(127,  5)
jacc( 42, 42) = jacc( 42, 42) - dw(127, 42)
jacc( 49,  5) = jacc( 49,  5)+ 0.8000000000000000d-01*dw(127,  5)
jacc( 49, 42) = jacc( 49, 42)+ 0.8000000000000000d-01*dw(127, 42)
jacc( 50,  5) = jacc( 50,  5)+ 0.4400000000000000d+00*dw(127,  5)
jacc( 50, 42) = jacc( 50, 42)+ 0.4400000000000000d+00*dw(127, 42)
jacc( 12, 12) = jacc( 12, 12) - dw(128, 12)
jacc( 12, 48) = jacc( 12, 48) - dw(128, 48)
jacc( 28, 12) = jacc( 28, 12)+ 0.9000000000000000d+00*dw(128, 12)
jacc( 28, 48) = jacc( 28, 48)+ 0.9000000000000000d+00*dw(128, 48)
jacc( 33, 12) = jacc( 33, 12)+ 0.1000000000000000d+00*dw(128, 12)
jacc( 33, 48) = jacc( 33, 48)+ 0.1000000000000000d+00*dw(128, 48)
jacc( 48, 12) = jacc( 48, 12) - dw(128, 12)
jacc( 48, 48) = jacc( 48, 48) - dw(128, 48)
jacc( 50, 12) = jacc( 50, 12)+ 0.9000000000000000d+00*dw(128, 12)
jacc( 50, 48) = jacc( 50, 48)+ 0.9000000000000000d+00*dw(128, 48)
jacc( 52, 12) = jacc( 52, 12)+ 0.9000000000000000d+00*dw(128, 12)
jacc( 52, 48) = jacc( 52, 48)+ 0.9000000000000000d+00*dw(128, 48)
jacc( 12, 12) = jacc( 12, 12) - dw(129, 12)
jacc( 27, 12) = jacc( 27, 12) + dw(129, 12)
jacc( 50, 12) = jacc( 50, 12) + dw(129, 12)
jacc( 27, 42) = jacc( 27, 42) - dw(130, 42)
jacc( 27, 27) = jacc( 27, 27) - dw(130, 27)
jacc( 28, 42) = jacc( 28, 42)+ 0.3000000000000000d+00*dw(130, 42)
jacc( 28, 27) = jacc( 28, 27)+ 0.3000000000000000d+00*dw(130, 27)
jacc( 30, 42) = jacc( 30, 42)+ 0.4000000000000000d+00*dw(130, 42)
jacc( 30, 27) = jacc( 30, 27)+ 0.4000000000000000d+00*dw(130, 27)
jacc( 42, 42) = jacc( 42, 42) - dw(130, 42)
jacc( 42, 27) = jacc( 42, 27) - dw(130, 27)
jacc( 49, 42) = jacc( 49, 42)+ 0.6000000000000000d+00*dw(130, 42)
jacc( 49, 27) = jacc( 49, 27)+ 0.6000000000000000d+00*dw(130, 27)
jacc( 50, 42) = jacc( 50, 42)+ 0.6000000000000000d+00*dw(130, 42)
jacc( 50, 27) = jacc( 50, 27)+ 0.6000000000000000d+00*dw(130, 27)
jacc( 27, 27) = jacc( 27, 27) - dw(131, 27)
jacc( 27, 44) = jacc( 27, 44) - dw(131, 44)
jacc( 30, 27) = jacc( 30, 27) + dw(131, 27)
jacc( 30, 44) = jacc( 30, 44) + dw(131, 44)
jacc( 38, 27) = jacc( 38, 27) + dw(131, 27)
jacc( 38, 44) = jacc( 38, 44) + dw(131, 44)
jacc( 44, 27) = jacc( 44, 27) - dw(131, 27)
jacc( 44, 44) = jacc( 44, 44) - dw(131, 44)
jacc( 30, 30) = jacc( 30, 30) - dw(132, 30)
jacc( 30, 52) = jacc( 30, 52) - dw(132, 52)
jacc( 33, 30) = jacc( 33, 30) + dw(132, 30)
jacc( 33, 52) = jacc( 33, 52) + dw(132, 52)
jacc( 52, 30) = jacc( 52, 30) - dw(132, 30)
jacc( 52, 52) = jacc( 52, 52) - dw(132, 52)
jacc( 27, 30) = jacc( 27, 30) + dw(133, 30)
jacc( 27, 50) = jacc( 27, 50) + dw(133, 50)
jacc( 30, 30) = jacc( 30, 30) - dw(133, 30)
jacc( 30, 50) = jacc( 30, 50) - dw(133, 50)
jacc( 50, 30) = jacc( 50, 30) - dw(133, 30)
jacc( 50, 50) = jacc( 50, 50) - dw(133, 50)
jacc( 28, 28) = jacc( 28, 28) - dw(134, 28)
jacc( 37, 28) = jacc( 37, 28) + dw(134, 28)
jacc( 50, 28) = jacc( 50, 28) + dw(134, 28)
jacc( 51, 28) = jacc( 51, 28) + dw(134, 28)
jacc( 28, 28) = jacc( 28, 28) - dw(135, 28)
jacc( 28, 42) = jacc( 28, 42) - dw(135, 42)
jacc( 37, 28) = jacc( 37, 28)+ 0.2000000000000000d+01*dw(135, 28)
jacc( 37, 42) = jacc( 37, 42)+ 0.2000000000000000d+01*dw(135, 42)
jacc( 41, 28) = jacc( 41, 28) + dw(135, 28)
jacc( 41, 42) = jacc( 41, 42) + dw(135, 42)
jacc( 42, 28) = jacc( 42, 28) - dw(135, 28)
jacc( 42, 42) = jacc( 42, 42) - dw(135, 42)
jacc( 49, 28) = jacc( 49, 28) + dw(135, 28)
jacc( 49, 42) = jacc( 49, 42) + dw(135, 42)
jacc( 50, 28) = jacc( 50, 28)+ 0.2000000000000000d+01*dw(135, 28)
jacc( 50, 42) = jacc( 50, 42)+ 0.2000000000000000d+01*dw(135, 42)
jacc( 51, 28) = jacc( 51, 28) + dw(135, 28)
jacc( 51, 42) = jacc( 51, 42) + dw(135, 42)
jacc( 28, 28) = jacc( 28, 28) - dw(136, 28)
jacc( 28, 47) = jacc( 28, 47) - dw(136, 47)
jacc( 29, 28) = jacc( 29, 28)+ 0.2000000000000000d+00*dw(136, 28)
jacc( 29, 47) = jacc( 29, 47)+ 0.2000000000000000d+00*dw(136, 47)
jacc( 35, 28) = jacc( 35, 28)+ 0.3000000000000000d-01*dw(136, 28)
jacc( 35, 47) = jacc( 35, 47)+ 0.3000000000000000d-01*dw(136, 47)
jacc( 37, 28) = jacc( 37, 28)+ 0.6899999999999999d+00*dw(136, 28)
jacc( 37, 47) = jacc( 37, 47)+ 0.6899999999999999d+00*dw(136, 47)
jacc( 41, 28) = jacc( 41, 28)+ 0.7000000000000000d+00*dw(136, 28)
jacc( 41, 47) = jacc( 41, 47)+ 0.7000000000000000d+00*dw(136, 47)
jacc( 42, 28) = jacc( 42, 28)+ 0.8000000000000000d-01*dw(136, 28)
jacc( 42, 47) = jacc( 42, 47)+ 0.8000000000000000d-01*dw(136, 47)
jacc( 47, 28) = jacc( 47, 28) - dw(136, 28)
jacc( 47, 47) = jacc( 47, 47) - dw(136, 47)
jacc( 49, 28) = jacc( 49, 28)+ 0.3000000000000000d-01*dw(136, 28)
jacc( 49, 47) = jacc( 49, 47)+ 0.3000000000000000d-01*dw(136, 47)
jacc( 50, 28) = jacc( 50, 28)+ 0.7600000000000000d+00*dw(136, 28)
jacc( 50, 47) = jacc( 50, 47)+ 0.7600000000000000d+00*dw(136, 47)
jacc( 51, 28) = jacc( 51, 28)+ 0.6200000000000000d+00*dw(136, 28)
jacc( 51, 47) = jacc( 51, 47)+ 0.6200000000000000d+00*dw(136, 47)
jacc(  6, 42) = jacc(  6, 42) - dw(137, 42)
jacc(  6,  6) = jacc(  6,  6) - dw(137,  6)
jacc( 12, 42) = jacc( 12, 42)+ 0.3000000000000000d+00*dw(137, 42)
jacc( 12,  6) = jacc( 12,  6)+ 0.3000000000000000d+00*dw(137,  6)
jacc( 27, 42) = jacc( 27, 42)+ 0.2000000000000000d-01*dw(137, 42)
jacc( 27,  6) = jacc( 27,  6)+ 0.2000000000000000d-01*dw(137,  6)
jacc( 29, 42) = jacc( 29, 42)+ 0.8000000000000000d+00*dw(137, 42)
jacc( 29,  6) = jacc( 29,  6)+ 0.8000000000000000d+00*dw(137,  6)
jacc( 34, 42) = jacc( 34, 42)+ 0.1100000000000000d+01*dw(137, 42)
jacc( 34,  6) = jacc( 34,  6)+ 0.1100000000000000d+01*dw(137,  6)
jacc( 42, 42) = jacc( 42, 42) - dw(137, 42)
jacc( 42,  6) = jacc( 42,  6) - dw(137,  6)
jacc( 49, 42) = jacc( 49, 42)+ 0.5000000000000000d+00*dw(137, 42)
jacc( 49,  6) = jacc( 49,  6)+ 0.5000000000000000d+00*dw(137,  6)
jacc( 50, 42) = jacc( 50, 42)+ 0.7000000000000000d+00*dw(137, 42)
jacc( 50,  6) = jacc( 50,  6)+ 0.7000000000000000d+00*dw(137,  6)
jacc( 29, 42) = jacc( 29, 42) - dw(138, 42)
jacc( 29, 29) = jacc( 29, 29) - dw(138, 29)
jacc( 42, 42) = jacc( 42, 42) - dw(138, 42)
jacc( 42, 29) = jacc( 42, 29) - dw(138, 29)
jacc( 49, 42) = jacc( 49, 42) + dw(138, 42)
jacc( 49, 29) = jacc( 49, 29) + dw(138, 29)
jacc( 51, 42) = jacc( 51, 42) + dw(138, 42)
jacc( 51, 29) = jacc( 51, 29) + dw(138, 29)
jacc( 29, 29) = jacc( 29, 29) - dw(139, 29)
jacc( 37, 29) = jacc( 37, 29) + dw(139, 29)
jacc( 50, 29) = jacc( 50, 29) + dw(139, 29)
jacc( 51, 29) = jacc( 51, 29) + dw(139, 29)
jacc( 31, 43) = jacc( 31, 43) - dw(140, 43)
jacc( 31, 31) = jacc( 31, 31) - dw(140, 31)
jacc( 32, 43) = jacc( 32, 43)+ 0.7500000000000000d+00*dw(140, 43)
jacc( 32, 31) = jacc( 32, 31)+ 0.7500000000000000d+00*dw(140, 31)
jacc( 34, 43) = jacc( 34, 43)+ 0.2500000000000000d+00*dw(140, 43)
jacc( 34, 31) = jacc( 34, 31)+ 0.2500000000000000d+00*dw(140, 31)
jacc( 41, 43) = jacc( 41, 43)+ 0.5000000000000000d+00*dw(140, 43)
jacc( 41, 31) = jacc( 41, 31)+ 0.5000000000000000d+00*dw(140, 31)
jacc( 43, 43) = jacc( 43, 43) - dw(140, 43)
jacc( 43, 31) = jacc( 43, 31) - dw(140, 31)
jacc( 46, 43) = jacc( 46, 43)+ 0.2500000000000000d+00*dw(140, 43)
jacc( 46, 31) = jacc( 46, 31)+ 0.2500000000000000d+00*dw(140, 31)
jacc( 49, 43) = jacc( 49, 43)+ 0.2500000000000000d+00*dw(140, 43)
jacc( 49, 31) = jacc( 49, 31)+ 0.2500000000000000d+00*dw(140, 31)
jacc( 50, 43) = jacc( 50, 43)+ 0.2500000000000000d+00*dw(140, 43)
jacc( 50, 31) = jacc( 50, 31)+ 0.2500000000000000d+00*dw(140, 31)
jacc( 31, 42) = jacc( 31, 42) - dw(141, 42)
jacc( 31, 31) = jacc( 31, 31) - dw(141, 31)
jacc( 32, 42) = jacc( 32, 42)+ 0.9120000000000000d+00*dw(141, 42)
jacc( 32, 31) = jacc( 32, 31)+ 0.9120000000000000d+00*dw(141, 31)
jacc( 39, 42) = jacc( 39, 42)+ 0.8799999999999999d-01*dw(141, 42)
jacc( 39, 31) = jacc( 39, 31)+ 0.8799999999999999d-01*dw(141, 31)
jacc( 41, 42) = jacc( 41, 42)+ 0.6290000000000000d+00*dw(141, 42)
jacc( 41, 31) = jacc( 41, 31)+ 0.6290000000000000d+00*dw(141, 31)
jacc( 42, 42) = jacc( 42, 42) - dw(141, 42)
jacc( 42, 31) = jacc( 42, 31) - dw(141, 31)
jacc( 49, 42) = jacc( 49, 42)+ 0.9910000000000000d+00*dw(141, 42)
jacc( 49, 31) = jacc( 49, 31)+ 0.9910000000000000d+00*dw(141, 31)
jacc( 50, 42) = jacc( 50, 42)+ 0.9120000000000000d+00*dw(141, 42)
jacc( 50, 31) = jacc( 50, 31)+ 0.9120000000000000d+00*dw(141, 31)
jacc( 31, 47) = jacc( 31, 47) - dw(142, 47)
jacc( 31, 31) = jacc( 31, 31) - dw(142, 31)
jacc( 32, 47) = jacc( 32, 47)+ 0.6500000000000000d+00*dw(142, 47)
jacc( 32, 31) = jacc( 32, 31)+ 0.6500000000000000d+00*dw(142, 31)
jacc( 34, 47) = jacc( 34, 47)+ 0.3500000000000000d+00*dw(142, 47)
jacc( 34, 31) = jacc( 34, 31)+ 0.3500000000000000d+00*dw(142, 31)
jacc( 35, 47) = jacc( 35, 47)+ 0.1500000000000000d+00*dw(142, 47)
jacc( 35, 31) = jacc( 35, 31)+ 0.1500000000000000d+00*dw(142, 31)
jacc( 37, 47) = jacc( 37, 47)+ 0.6600000000000000d-01*dw(142, 47)
jacc( 37, 31) = jacc( 37, 31)+ 0.6600000000000000d-01*dw(142, 31)
jacc( 41, 47) = jacc( 41, 47)+ 0.6000000000000000d+00*dw(142, 47)
jacc( 41, 31) = jacc( 41, 31)+ 0.6000000000000000d+00*dw(142, 31)
jacc( 42, 47) = jacc( 42, 47)+ 0.2660000000000000d+00*dw(142, 47)
jacc( 42, 31) = jacc( 42, 31)+ 0.2660000000000000d+00*dw(142, 31)
jacc( 46, 47) = jacc( 46, 47)+ 0.2000000000000000d+00*dw(142, 47)
jacc( 46, 31) = jacc( 46, 31)+ 0.2000000000000000d+00*dw(142, 31)
jacc( 47, 47) = jacc( 47, 47) - dw(142, 47)
jacc( 47, 31) = jacc( 47, 31) - dw(142, 31)
jacc( 49, 47) = jacc( 49, 47)+ 0.2000000000000000d+00*dw(142, 47)
jacc( 49, 31) = jacc( 49, 31)+ 0.2000000000000000d+00*dw(142, 31)
jacc( 50, 47) = jacc( 50, 47)+ 0.6600000000000000d-01*dw(142, 47)
jacc( 50, 31) = jacc( 50, 31)+ 0.6600000000000000d-01*dw(142, 31)
jacc( 31, 44) = jacc( 31, 44) - dw(143, 44)
jacc( 31, 31) = jacc( 31, 31) - dw(143, 31)
jacc( 32, 44) = jacc( 32, 44)+ 0.2000000000000000d+00*dw(143, 44)
jacc( 32, 31) = jacc( 32, 31)+ 0.2000000000000000d+00*dw(143, 31)
jacc( 33, 44) = jacc( 33, 44)+ 0.8000000000000000d+00*dw(143, 44)
jacc( 33, 31) = jacc( 33, 31)+ 0.8000000000000000d+00*dw(143, 31)
jacc( 34, 44) = jacc( 34, 44)+ 0.2400000000000000d+01*dw(143, 44)
jacc( 34, 31) = jacc( 34, 31)+ 0.2400000000000000d+01*dw(143, 31)
jacc( 35, 44) = jacc( 35, 44)+ 0.8000000000000000d+00*dw(143, 44)
jacc( 35, 31) = jacc( 35, 31)+ 0.8000000000000000d+00*dw(143, 31)
jacc( 44, 44) = jacc( 44, 44) - dw(143, 44)
jacc( 44, 31) = jacc( 44, 31) - dw(143, 31)
jacc( 49, 44) = jacc( 49, 44) + dw(143, 44)
jacc( 49, 31) = jacc( 49, 31) + dw(143, 31)
jacc( 50, 44) = jacc( 50, 44)+ 0.8000000000000000d+00*dw(143, 44)
jacc( 50, 31) = jacc( 50, 31)+ 0.8000000000000000d+00*dw(143, 31)
jacc( 52, 44) = jacc( 52, 44)+ 0.2000000000000000d+00*dw(143, 44)
jacc( 52, 31) = jacc( 52, 31)+ 0.2000000000000000d+00*dw(143, 31)
jacc( 29, 42) = jacc( 29, 42)+ 0.1680000000000000d+00*dw(144, 42)
jacc( 29, 32) = jacc( 29, 32)+ 0.1680000000000000d+00*dw(144, 32)
jacc( 32, 42) = jacc( 32, 42) - dw(144, 42)
jacc( 32, 32) = jacc( 32, 32) - dw(144, 32)
jacc( 34, 42) = jacc( 34, 42)+ 0.1565000000000000d+01*dw(144, 42)
jacc( 34, 32) = jacc( 34, 32)+ 0.1565000000000000d+01*dw(144, 32)
jacc( 35, 42) = jacc( 35, 42)+ 0.1200000000000000d+00*dw(144, 42)
jacc( 35, 32) = jacc( 35, 32)+ 0.1200000000000000d+00*dw(144, 32)
jacc( 37, 42) = jacc( 37, 42)+ 0.3340000000000000d+00*dw(144, 42)
jacc( 37, 32) = jacc( 37, 32)+ 0.3340000000000000d+00*dw(144, 32)
jacc( 41, 42) = jacc( 41, 42)+ 0.1670000000000000d+00*dw(144, 42)
jacc( 41, 32) = jacc( 41, 32)+ 0.1670000000000000d+00*dw(144, 32)
jacc( 42, 42) = jacc( 42, 42) - dw(144, 42)
jacc( 42, 32) = jacc( 42, 32) - dw(144, 32)
jacc( 45, 42) = jacc( 45, 42)+ 0.2520000000000000d+00*dw(144, 42)
jacc( 45, 32) = jacc( 45, 32)+ 0.2520000000000000d+00*dw(144, 32)
jacc( 46, 42) = jacc( 46, 42)+ 0.2500000000000000d+00*dw(144, 42)
jacc( 46, 32) = jacc( 46, 32)+ 0.2500000000000000d+00*dw(144, 32)
jacc( 49, 42) = jacc( 49, 42)+ 0.7130000000000000d+00*dw(144, 42)
jacc( 49, 32) = jacc( 49, 32)+ 0.7130000000000000d+00*dw(144, 32)
jacc( 50, 42) = jacc( 50, 42)+ 0.5030000000000000d+00*dw(144, 42)
jacc( 50, 32) = jacc( 50, 32)+ 0.5030000000000000d+00*dw(144, 32)
jacc( 51, 42) = jacc( 51, 42)+ 0.2100000000000000d+00*dw(144, 42)
jacc( 51, 32) = jacc( 51, 32)+ 0.2100000000000000d+00*dw(144, 32)
jacc( 29, 47) = jacc( 29, 47)+ 0.8500000000000000d+00*dw(145, 47)
jacc( 29, 32) = jacc( 29, 32)+ 0.8500000000000000d+00*dw(145, 32)
jacc( 32, 47) = jacc( 32, 47) - dw(145, 47)
jacc( 32, 32) = jacc( 32, 32) - dw(145, 32)
jacc( 34, 47) = jacc( 34, 47)+ 0.3600000000000000d+00*dw(145, 47)
jacc( 34, 32) = jacc( 34, 32)+ 0.3600000000000000d+00*dw(145, 32)
jacc( 37, 47) = jacc( 37, 47)+ 0.2250000000000000d+00*dw(145, 47)
jacc( 37, 32) = jacc( 37, 32)+ 0.2250000000000000d+00*dw(145, 32)
jacc( 41, 47) = jacc( 41, 47)+ 0.1500000000000000d+00*dw(145, 47)
jacc( 41, 32) = jacc( 41, 32)+ 0.1500000000000000d+00*dw(145, 32)
jacc( 42, 47) = jacc( 42, 47)+ 0.2680000000000000d+00*dw(145, 47)
jacc( 42, 32) = jacc( 42, 32)+ 0.2680000000000000d+00*dw(145, 32)
jacc( 45, 47) = jacc( 45, 47)+ 0.2000000000000000d-01*dw(145, 47)
jacc( 45, 32) = jacc( 45, 32)+ 0.2000000000000000d-01*dw(145, 32)
jacc( 47, 47) = jacc( 47, 47) - dw(145, 47)
jacc( 47, 32) = jacc( 47, 32) - dw(145, 32)
jacc( 49, 47) = jacc( 49, 47)+ 0.6400000000000000d-01*dw(145, 47)
jacc( 49, 32) = jacc( 49, 32)+ 0.6400000000000000d-01*dw(145, 32)
jacc( 50, 47) = jacc( 50, 47)+ 0.1540000000000000d+00*dw(145, 47)
jacc( 50, 32) = jacc( 50, 32)+ 0.1540000000000000d+00*dw(145, 32)
jacc( 51, 47) = jacc( 51, 47)+ 0.1140000000000000d+00*dw(145, 47)
jacc( 51, 32) = jacc( 51, 32)+ 0.1140000000000000d+00*dw(145, 32)
jacc( 32, 44) = jacc( 32, 44) - dw(146, 44)
jacc( 32, 32) = jacc( 32, 32) - dw(146, 32)
jacc( 33, 44) = jacc( 33, 44)+ 0.8500000000000000d+00*dw(146, 44)
jacc( 33, 32) = jacc( 33, 32)+ 0.8500000000000000d+00*dw(146, 32)
jacc( 34, 44) = jacc( 34, 44)+ 0.1282000000000000d+01*dw(146, 44)
jacc( 34, 32) = jacc( 34, 32)+ 0.1282000000000000d+01*dw(146, 32)
jacc( 35, 44) = jacc( 35, 44)+ 0.3570000000000000d+00*dw(146, 44)
jacc( 35, 32) = jacc( 35, 32)+ 0.3570000000000000d+00*dw(146, 32)
jacc( 37, 44) = jacc( 37, 44)+ 0.6430000000000000d+00*dw(146, 44)
jacc( 37, 32) = jacc( 37, 32)+ 0.6430000000000000d+00*dw(146, 32)
jacc( 38, 44) = jacc( 38, 44)+ 0.1500000000000000d+00*dw(146, 44)
jacc( 38, 32) = jacc( 38, 32)+ 0.1500000000000000d+00*dw(146, 32)
jacc( 41, 44) = jacc( 41, 44)+ 0.2820000000000000d+00*dw(146, 44)
jacc( 41, 32) = jacc( 41, 32)+ 0.2820000000000000d+00*dw(146, 32)
jacc( 44, 44) = jacc( 44, 44) - dw(146, 44)
jacc( 44, 32) = jacc( 44, 32) - dw(146, 32)
jacc( 46, 44) = jacc( 46, 44)+ 0.7500000000000000d-01*dw(146, 44)
jacc( 46, 32) = jacc( 46, 32)+ 0.7500000000000000d-01*dw(146, 32)
jacc( 49, 44) = jacc( 49, 44)+ 0.7500000000000000d-01*dw(146, 44)
jacc( 49, 32) = jacc( 49, 32)+ 0.7500000000000000d-01*dw(146, 32)
jacc( 50, 44) = jacc( 50, 44)+ 0.9250000000000000d+00*dw(146, 44)
jacc( 50, 32) = jacc( 50, 32)+ 0.9250000000000000d+00*dw(146, 32)
jacc( 32, 32) = jacc( 32, 32) - dw(147, 32)
jacc( 34, 32) = jacc( 34, 32)+ 0.8320000000000000d+00*dw(147, 32)
jacc( 37, 32) = jacc( 37, 32)+ 0.3330000000000000d+00*dw(147, 32)
jacc( 41, 32) = jacc( 41, 32)+ 0.9000000000000000d+00*dw(147, 32)
jacc( 45, 32) = jacc( 45, 32)+ 0.6700000000000000d-01*dw(147, 32)
jacc( 49, 32) = jacc( 49, 32)+ 0.7000000000000000d+00*dw(147, 32)
jacc( 50, 32) = jacc( 50, 32)+ 0.1033000000000000d+01*dw(147, 32)
jacc( 51, 32) = jacc( 51, 32)+ 0.9670000000000000d+00*dw(147, 32)
jacc( 26, 26) = jacc( 26, 26) - dw(148, 26)
jacc( 26, 43) = jacc( 26, 43) - dw(148, 43)
jacc( 34, 26) = jacc( 34, 26)+ 0.5120000000000000d+01*dw(148, 26)
jacc( 34, 43) = jacc( 34, 43)+ 0.5120000000000000d+01*dw(148, 43)
jacc( 35, 26) = jacc( 35, 26)+ 0.1500000000000000d+00*dw(148, 26)
jacc( 35, 43) = jacc( 35, 43)+ 0.1500000000000000d+00*dw(148, 43)
jacc( 43, 26) = jacc( 43, 26) - dw(148, 26)
jacc( 43, 43) = jacc( 43, 43) - dw(148, 43)
jacc( 26, 26) = jacc( 26, 26) - dw(149, 26)
jacc( 26, 42) = jacc( 26, 42) - dw(149, 42)
jacc( 34, 26) = jacc( 34, 26)+ 0.1660000000000000d+01*dw(149, 26)
jacc( 34, 42) = jacc( 34, 42)+ 0.1660000000000000d+01*dw(149, 42)
jacc( 35, 26) = jacc( 35, 26)+ 0.4700000000000000d+00*dw(149, 26)
jacc( 35, 42) = jacc( 35, 42)+ 0.4700000000000000d+00*dw(149, 42)
jacc( 39, 26) = jacc( 39, 26)+ 0.2500000000000000d+00*dw(149, 26)
jacc( 39, 42) = jacc( 39, 42)+ 0.2500000000000000d+00*dw(149, 42)
jacc( 41, 26) = jacc( 41, 26)+ 0.2800000000000000d+00*dw(149, 26)
jacc( 41, 42) = jacc( 41, 42)+ 0.2800000000000000d+00*dw(149, 42)
jacc( 42, 26) = jacc( 42, 26) - dw(149, 26)
jacc( 42, 42) = jacc( 42, 42) - dw(149, 42)
jacc( 49, 26) = jacc( 49, 26)+ 0.1250000000000000d+01*dw(149, 26)
jacc( 49, 42) = jacc( 49, 42)+ 0.1250000000000000d+01*dw(149, 42)
jacc( 50, 26) = jacc( 50, 26)+ 0.7500000000000000d+00*dw(149, 26)
jacc( 50, 42) = jacc( 50, 42)+ 0.7500000000000000d+00*dw(149, 42)
jacc( 26, 26) = jacc( 26, 26) - dw(150, 26)
jacc( 26, 47) = jacc( 26, 47) - dw(150, 47)
jacc( 34, 26) = jacc( 34, 26)+ 0.7000000000000000d+01*dw(150, 26)
jacc( 34, 47) = jacc( 34, 47)+ 0.7000000000000000d+01*dw(150, 47)
jacc( 35, 26) = jacc( 35, 26)+ 0.2100000000000000d+00*dw(150, 26)
jacc( 35, 47) = jacc( 35, 47)+ 0.2100000000000000d+00*dw(150, 47)
jacc( 37, 26) = jacc( 37, 26)+ 0.1000000000000000d-02*dw(150, 26)
jacc( 37, 47) = jacc( 37, 47)+ 0.1000000000000000d-02*dw(150, 47)
jacc( 39, 26) = jacc( 39, 26)+ 0.1800000000000000d+00*dw(150, 26)
jacc( 39, 47) = jacc( 39, 47)+ 0.1800000000000000d+00*dw(150, 47)
jacc( 41, 26) = jacc( 41, 26)+ 0.2400000000000000d+00*dw(150, 26)
jacc( 41, 47) = jacc( 41, 47)+ 0.2400000000000000d+00*dw(150, 47)
jacc( 42, 26) = jacc( 42, 26)+ 0.5700000000000000d+00*dw(150, 26)
jacc( 42, 47) = jacc( 42, 47)+ 0.5700000000000000d+00*dw(150, 47)
jacc( 46, 26) = jacc( 46, 26)+ 0.3900000000000000d+00*dw(150, 26)
jacc( 46, 47) = jacc( 46, 47)+ 0.3900000000000000d+00*dw(150, 47)
jacc( 47, 26) = jacc( 47, 26) - dw(150, 26)
jacc( 47, 47) = jacc( 47, 47) - dw(150, 47)
jacc( 49, 26) = jacc( 49, 26)+ 0.7600000000000000d+00*dw(150, 26)
jacc( 49, 47) = jacc( 49, 47)+ 0.7600000000000000d+00*dw(150, 47)
jacc( 50, 26) = jacc( 50, 26)+ 0.7000000000000001d-01*dw(150, 26)
jacc( 50, 47) = jacc( 50, 47)+ 0.7000000000000001d-01*dw(150, 47)
jacc( 26, 26) = jacc( 26, 26) - dw(151, 26)
jacc( 26, 44) = jacc( 26, 44) - dw(151, 44)
jacc( 33, 26) = jacc( 33, 26)+ 0.5300000000000000d+00*dw(151, 26)
jacc( 33, 44) = jacc( 33, 44)+ 0.5300000000000000d+00*dw(151, 44)
jacc( 35, 26) = jacc( 35, 26)+ 0.4700000000000000d+00*dw(151, 26)
jacc( 35, 44) = jacc( 35, 44)+ 0.4700000000000000d+00*dw(151, 44)
jacc( 39, 26) = jacc( 39, 26)+ 0.2500000000000000d+00*dw(151, 26)
jacc( 39, 44) = jacc( 39, 44)+ 0.2500000000000000d+00*dw(151, 44)
jacc( 44, 26) = jacc( 44, 26) - dw(151, 26)
jacc( 44, 44) = jacc( 44, 44) - dw(151, 44)
jacc( 49, 26) = jacc( 49, 26)+ 0.1030000000000000d+01*dw(151, 26)
jacc( 49, 44) = jacc( 49, 44)+ 0.1030000000000000d+01*dw(151, 44)
jacc( 50, 26) = jacc( 50, 26)+ 0.2800000000000000d+00*dw(151, 26)
jacc( 50, 44) = jacc( 50, 44)+ 0.2800000000000000d+00*dw(151, 44)
jacc( 52, 26) = jacc( 52, 26)+ 0.4700000000000000d+00*dw(151, 26)
jacc( 52, 44) = jacc( 52, 44)+ 0.4700000000000000d+00*dw(151, 44)
jacc(  7,  7) = jacc(  7,  7) - dw(152,  7)
jacc(  7, 42) = jacc(  7, 42) - dw(152, 42)
jacc(  8,  7) = jacc(  8,  7) + dw(152,  7)
jacc(  8, 42) = jacc(  8, 42) + dw(152, 42)
jacc( 42,  7) = jacc( 42,  7) - dw(152,  7)
jacc( 42, 42) = jacc( 42, 42) - dw(152, 42)
jacc( 50,  7) = jacc( 50,  7) + dw(152,  7)
jacc( 50, 42) = jacc( 50, 42) + dw(152, 42)
jacc(  2, 42) = jacc(  2, 42) - dw(153, 42)
jacc(  2,  2) = jacc(  2,  2) - dw(153,  2)
jacc( 35, 42) = jacc( 35, 42)+ 0.5000000000000000d-01*dw(153, 42)
jacc( 35,  2) = jacc( 35,  2)+ 0.5000000000000000d-01*dw(153,  2)
jacc( 41, 42) = jacc( 41, 42)+ 0.1000000000000000d+00*dw(153, 42)
jacc( 41,  2) = jacc( 41,  2)+ 0.1000000000000000d+00*dw(153,  2)
jacc( 42, 42) = jacc( 42, 42) - dw(153, 42)
jacc( 42,  2) = jacc( 42,  2) - dw(153,  2)
jacc( 45, 42) = jacc( 45, 42)+ 0.9000000000000000d+00*dw(153, 42)
jacc( 45,  2) = jacc( 45,  2)+ 0.9000000000000000d+00*dw(153,  2)
jacc( 49, 42) = jacc( 49, 42)+ 0.1000000000000000d+00*dw(153, 42)
jacc( 49,  2) = jacc( 49,  2)+ 0.1000000000000000d+00*dw(153,  2)
jacc( 50, 42) = jacc( 50, 42) + dw(153, 42)
jacc( 50,  2) = jacc( 50,  2) + dw(153,  2)
jacc(  4, 42) = jacc(  4, 42) - dw(154, 42)
jacc(  4,  4) = jacc(  4,  4) - dw(154,  4)
jacc( 39, 42) = jacc( 39, 42)+ 0.8999999999999999d-02*dw(154, 42)
jacc( 39,  4) = jacc( 39,  4)+ 0.8999999999999999d-02*dw(154,  4)
jacc( 42, 42) = jacc( 42, 42) - dw(154, 42)
jacc( 42,  4) = jacc( 42,  4) - dw(154,  4)
jacc( 45, 42) = jacc( 45, 42)+ 0.9910000000000000d+00*dw(154, 42)
jacc( 45,  4) = jacc( 45,  4)+ 0.9910000000000000d+00*dw(154,  4)
jacc( 49, 42) = jacc( 49, 42)+ 0.9910000000000000d+00*dw(154, 42)
jacc( 49,  4) = jacc( 49,  4)+ 0.9910000000000000d+00*dw(154,  4)
jacc( 50, 42) = jacc( 50, 42) + dw(154, 42)
jacc( 50,  4) = jacc( 50,  4) + dw(154,  4)
jacc( 31, 52) = jacc( 31, 52) - dw(155, 52)
jacc( 31, 31) = jacc( 31, 31) - dw(155, 31)
jacc( 32, 52) = jacc( 32, 52)+ 0.2000000000000000d+00*dw(155, 52)
jacc( 32, 31) = jacc( 32, 31)+ 0.2000000000000000d+00*dw(155, 31)
jacc( 33, 52) = jacc( 33, 52)+ 0.8000000000000000d+00*dw(155, 52)
jacc( 33, 31) = jacc( 33, 31)+ 0.8000000000000000d+00*dw(155, 31)
jacc( 34, 52) = jacc( 34, 52)+ 0.2400000000000000d+01*dw(155, 52)
jacc( 34, 31) = jacc( 34, 31)+ 0.2400000000000000d+01*dw(155, 31)
jacc( 35, 52) = jacc( 35, 52)+ 0.8000000000000000d+00*dw(155, 52)
jacc( 35, 31) = jacc( 35, 31)+ 0.8000000000000000d+00*dw(155, 31)
jacc( 48, 52) = jacc( 48, 52)+ 0.2000000000000000d+00*dw(155, 52)
jacc( 48, 31) = jacc( 48, 31)+ 0.2000000000000000d+00*dw(155, 31)
jacc( 49, 52) = jacc( 49, 52) + dw(155, 52)
jacc( 49, 31) = jacc( 49, 31) + dw(155, 31)
jacc( 50, 52) = jacc( 50, 52)+ 0.8000000000000000d+00*dw(155, 52)
jacc( 50, 31) = jacc( 50, 31)+ 0.8000000000000000d+00*dw(155, 31)
jacc( 52, 52) = jacc( 52, 52) - dw(155, 52)
jacc( 52, 31) = jacc( 52, 31) - dw(155, 31)

do j = 1, ns
   do i = 1, ns
      jacc(i,j) = jacc(i,j) * convers_factor_jac(i,j)
   enddo
enddo


return
end subroutine jacdchemdc_3


!===============================================================================

!> rates_3
!> \brief Computation of reaction rates
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     ns                 total number of chemical species
!> \param[in]     nr                 total number of chemical reactions
!> \param[in]     rk                 kinetic rates
!> \param[in]     y                  concentrations vector
!> \param[out]    w                  reaction rates
!______________________________________________________________________________

subroutine rates_3(ns,nr,rk,y,w)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns)

! Local variables

double precision w(nr)

w(  1) =  rk(  1) * y( 52)
w(  2) =  rk(  2) * y( 43)
w(  3) =  rk(  3) * y( 47) * y( 48)
w(  4) =  rk(  4) * y( 43) * y( 52)
w(  5) =  rk(  5) * y( 43) * y( 52)
w(  6) =  rk(  6) * y( 43) * y( 48)
w(  7) =  rk(  7) * y( 52) * y( 47)
w(  8) =  rk(  8) * y( 47)
w(  9) =  rk(  9) * y( 47)
w( 10) =  rk( 10) * y(  1)
w( 11) =  rk( 11) * y(  1)
w( 12) =  rk( 12) * y( 47) * y( 42)
w( 13) =  rk( 13) * y( 47) * y( 50)
w( 14) =  rk( 14) * y( 44)
w( 15) =  rk( 15) * y( 44)
w( 16) =  rk( 16) * y( 44) * y( 48)
w( 17) =  rk( 17) * y( 44) * y( 52)
w( 18) =  rk( 18) * y( 44) * y( 52)
w( 19) =  rk( 19) * y(  9)
w( 20) =  rk( 20) * y(  9)
w( 21) =  rk( 21) * y( 48) * y( 48)
w( 22) =  rk( 22) * y( 48) * y( 52)
w( 23) =  rk( 23) * y( 42) * y( 48)
w( 24) =  rk( 24) * y( 15)
w( 25) =  rk( 25) * y( 42) * y( 15)
w( 26) =  rk( 26) * y( 15) * y( 15)
w( 27) =  rk( 27) * y( 52) * y( 42)
w( 28) =  rk( 28) * y( 42) * y( 38)
w( 29) =  rk( 29) * y( 50) * y( 48)
w( 30) =  rk( 30) * y( 50) * y( 52)
w( 31) =  rk( 31) * y( 13)
w( 32) =  rk( 32) * y( 42) * y( 13)
w( 33) =  rk( 33) * y( 50) * y( 50)
w( 34) =  rk( 34) * y( 50) * y( 50)
w( 35) =  rk( 35) * y( 17)
w( 36) =  rk( 36) * y( 42) * y( 17)
w( 37) =  rk( 37) * y(  1)
w( 38) =  rk( 38) * y( 42)
w( 39) =  rk( 39) * y( 42) * y( 43)
w( 40) =  rk( 40) * y( 42) * y( 42)
w( 41) =  rk( 41) * y( 42) * y( 42)
w( 42) =  rk( 42) * y( 42) * y( 50)
w( 43) =  rk( 43) * y( 50) * y( 43)
w( 44) =  rk( 44) * y( 17) * y( 43)
w( 45) =  rk( 45) * y( 44) * y( 43)
w( 46) =  rk( 46) * y( 44) * y( 42)
w( 47) =  rk( 47) * y( 44) * y( 50)
w( 48) =  rk( 48) * y( 44) * y( 47)
w( 49) =  rk( 49) * y( 44) * y( 44)
w( 50) =  rk( 50) * y( 13)
w( 51) =  rk( 51) * y( 38)
w( 52) =  rk( 52) * y(  9)
w( 53) =  rk( 53) * y( 49) * y( 48)
w( 54) =  rk( 54) * y( 39) * y( 48)
w( 55) =  rk( 55) * y( 49) * y( 50)
w( 56) =  rk( 56) * y( 39) * y( 50)
w( 57) =  rk( 57) * y( 49) * y( 49)
w( 58) =  rk( 58) * y( 39) * y( 39)
w( 59) =  rk( 59) * y( 49) * y( 39)
w( 60) =  rk( 60) * y( 33) * y( 42)
w( 61) =  rk( 61) * y( 33)
w( 62) =  rk( 62) * y( 21) * y( 42)
w( 63) =  rk( 63) * y( 21)
w( 64) =  rk( 64) * y( 42) * y( 37)
w( 65) =  rk( 65) * y( 42) * y(  3)
w( 66) =  rk( 66) * y( 40) * y( 48)
w( 67) =  rk( 67) * y( 40) * y( 50)
w( 68) =  rk( 68) * y( 40) * y( 40)
w( 69) =  rk( 69) * y( 19) * y( 42)
w( 70) =  rk( 70) * y( 19)
w( 71) =  rk( 71) * y( 11) * y( 42)
w( 72) =  rk( 72) * y( 41) * y( 42)
w( 73) =  rk( 73) * y( 41)
w( 74) =  rk( 74) * y( 41)
w( 75) =  rk( 75) * y( 41) * y( 43)
w( 76) =  rk( 76) * y( 41) * y( 44)
w( 77) =  rk( 77) * y( 41) * y( 50)
w( 78) =  rk( 78) * y( 18)
w( 79) =  rk( 79) * y( 18) * y( 48)
w( 80) =  rk( 80) * y( 18) * y( 50)
w( 81) =  rk( 81) * y( 20) * y( 42)
w( 82) =  rk( 82) * y( 45) * y( 43)
w( 83) =  rk( 83) * y( 45) * y( 42)
w( 84) =  rk( 84) * y( 45) * y( 44)
w( 85) =  rk( 85) * y( 45)
w( 86) =  rk( 86) * y( 51) * y( 48)
w( 87) =  rk( 87) * y( 51) * y( 52)
w( 88) =  rk( 88) * y( 10)
w( 89) =  rk( 89) * y( 10)
w( 90) =  rk( 90) * y( 51) * y( 50)
w( 91) =  rk( 91) * y( 51) * y( 40)
w( 92) =  rk( 92) * y( 51) * y( 49)
w( 93) =  rk( 93) * y( 51) * y( 51)
w( 94) =  rk( 94) * y( 22) * y( 42)
w( 95) =  rk( 95) * y( 22)
w( 96) =  rk( 96) * y( 36) * y( 42)
w( 97) =  rk( 97) * y( 35) * y( 43)
w( 98) =  rk( 98) * y( 35) * y( 42)
w( 99) =  rk( 99) * y( 35) * y( 44)
w(100) =  rk(100) * y( 35)
w(101) =  rk(101) * y( 46) * y( 48)
w(102) =  rk(102) * y( 46) * y( 52)
w(103) =  rk(103) * y( 14)
w(104) =  rk(104) * y( 14)
w(105) =  rk(105) * y( 14) * y( 42)
w(106) =  rk(106) * y( 46) * y( 50)
w(107) =  rk(107) * y( 51) * y( 40)
w(108) =  rk(108) * y( 51) * y( 49)
w(109) =  rk(109) * y( 46) * y( 46)
w(110) =  rk(110) * y( 46) * y( 51)
w(111) =  rk(111) * y( 34) * y( 42)
w(112) =  rk(112) * y( 16)
w(113) =  rk(113) * y( 16)
w(114) =  rk(114) * y( 16) * y( 52)
w(115) =  rk(115) * y( 43) * y( 24)
w(116) =  rk(116) * y( 42) * y( 24)
w(117) =  rk(117) * y( 47) * y( 24)
w(118) =  rk(118) * y( 44) * y( 24)
w(119) =  rk(119) * y( 43) * y( 23)
w(120) =  rk(120) * y( 42) * y( 23)
w(121) =  rk(121) * y( 47) * y( 23)
w(122) =  rk(122) * y( 44) * y( 23)
w(123) =  rk(123) * y( 25) * y( 43)
w(124) =  rk(124) * y( 25) * y( 42)
w(125) =  rk(125) * y( 25) * y( 47)
w(126) =  rk(126) * y( 25) * y( 44)
w(127) =  rk(127) * y(  5) * y( 42)
w(128) =  rk(128) * y( 12) * y( 48)
w(129) =  rk(129) * y( 12)
w(130) =  rk(130) * y( 42) * y( 27)
w(131) =  rk(131) * y( 27) * y( 44)
w(132) =  rk(132) * y( 30) * y( 52)
w(133) =  rk(133) * y( 30) * y( 50)
w(134) =  rk(134) * y( 28)
w(135) =  rk(135) * y( 28) * y( 42)
w(136) =  rk(136) * y( 28) * y( 47)
w(137) =  rk(137) * y( 42) * y(  6)
w(138) =  rk(138) * y( 42) * y( 29)
w(139) =  rk(139) * y( 29)
w(140) =  rk(140) * y( 43) * y( 31)
w(141) =  rk(141) * y( 42) * y( 31)
w(142) =  rk(142) * y( 47) * y( 31)
w(143) =  rk(143) * y( 44) * y( 31)
w(144) =  rk(144) * y( 42) * y( 32)
w(145) =  rk(145) * y( 47) * y( 32)
w(146) =  rk(146) * y( 44) * y( 32)
w(147) =  rk(147) * y( 32)
w(148) =  rk(148) * y( 26) * y( 43)
w(149) =  rk(149) * y( 26) * y( 42)
w(150) =  rk(150) * y( 26) * y( 47)
w(151) =  rk(151) * y( 26) * y( 44)
w(152) =  rk(152) * y(  7) * y( 42)
w(153) =  rk(153) * y( 42) * y(  2)
w(154) =  rk(154) * y( 42) * y(  4)
w(155) =  rk(155) * y( 52) * y( 31)

return
end subroutine rates_3


!===============================================================================

!> dratedc_3
!> \brief Computation of derivatives of reaction rates
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     ns                 total number of chemical species
!> \param[in]     nr                 total number of chemical reactions
!> \param[in]     rk                 kinetic rates
!> \param[in]     y                  concentrations vector
!> \param[out]    dw                 derivatives of reaction rates
!______________________________________________________________________________

subroutine dratedc_3(ns,nr,rk,y,dw)

implicit none

! Arguments

integer nr,ns
double precision rk(nr),y(ns)

! Local variables

double precision dw(nr,ns)

dw(  1, 52) =  rk(  1)
dw(  2, 43) =  rk(  2)
dw(  3, 47) =  rk(  3) * y( 48)
dw(  3, 48) =  rk(  3) * y( 47)
dw(  4, 43) =  rk(  4) * y( 52)
dw(  4, 52) =  rk(  4) * y( 43)
dw(  5, 43) =  rk(  5) * y( 52)
dw(  5, 52) =  rk(  5) * y( 43)
dw(  6, 43) =  rk(  6) * y( 48)
dw(  6, 48) =  rk(  6) * y( 43)
dw(  7, 52) =  rk(  7) * y( 47)
dw(  7, 47) =  rk(  7) * y( 52)
dw(  8, 47) =  rk(  8)
dw(  9, 47) =  rk(  9)
dw( 10,  1) =  rk( 10)
dw( 11,  1) =  rk( 11)
dw( 12, 47) =  rk( 12) * y( 42)
dw( 12, 42) =  rk( 12) * y( 47)
dw( 13, 47) =  rk( 13) * y( 50)
dw( 13, 50) =  rk( 13) * y( 47)
dw( 14, 44) =  rk( 14)
dw( 15, 44) =  rk( 15)
dw( 16, 44) =  rk( 16) * y( 48)
dw( 16, 48) =  rk( 16) * y( 44)
dw( 17, 44) =  rk( 17) * y( 52)
dw( 17, 52) =  rk( 17) * y( 44)
dw( 18, 44) =  rk( 18) * y( 52)
dw( 18, 52) =  rk( 18) * y( 44)
dw( 19,  9) =  rk( 19)
dw( 20,  9) =  rk( 20)
dw( 21, 48) =  rk( 21) * y( 48)
dw( 21, 48) =  rk( 21) * y( 48)
dw( 22, 48) =  rk( 22) * y( 52)
dw( 22, 52) =  rk( 22) * y( 48)
dw( 23, 42) =  rk( 23) * y( 48)
dw( 23, 48) =  rk( 23) * y( 42)
dw( 24, 15) =  rk( 24)
dw( 25, 42) =  rk( 25) * y( 15)
dw( 25, 15) =  rk( 25) * y( 42)
dw( 26, 15) =  rk( 26) * y( 15)
dw( 26, 15) =  rk( 26) * y( 15)
dw( 27, 52) =  rk( 27) * y( 42)
dw( 27, 42) =  rk( 27) * y( 52)
dw( 28, 42) =  rk( 28) * y( 38)
dw( 28, 38) =  rk( 28) * y( 42)
dw( 29, 50) =  rk( 29) * y( 48)
dw( 29, 48) =  rk( 29) * y( 50)
dw( 30, 50) =  rk( 30) * y( 52)
dw( 30, 52) =  rk( 30) * y( 50)
dw( 31, 13) =  rk( 31)
dw( 32, 42) =  rk( 32) * y( 13)
dw( 32, 13) =  rk( 32) * y( 42)
dw( 33, 50) =  rk( 33) * y( 50)
dw( 33, 50) =  rk( 33) * y( 50)
dw( 34, 50) =  rk( 34) * y( 50)
dw( 34, 50) =  rk( 34) * y( 50)
dw( 35, 17) =  rk( 35)
dw( 36, 42) =  rk( 36) * y( 17)
dw( 36, 17) =  rk( 36) * y( 42)
dw( 37,  1) =  rk( 37)
dw( 38, 42) =  rk( 38)
dw( 39, 42) =  rk( 39) * y( 43)
dw( 39, 43) =  rk( 39) * y( 42)
dw( 40, 42) =  rk( 40) * y( 42)
dw( 40, 42) =  rk( 40) * y( 42)
dw( 41, 42) =  rk( 41) * y( 42)
dw( 41, 42) =  rk( 41) * y( 42)
dw( 42, 42) =  rk( 42) * y( 50)
dw( 42, 50) =  rk( 42) * y( 42)
dw( 43, 50) =  rk( 43) * y( 43)
dw( 43, 43) =  rk( 43) * y( 50)
dw( 44, 17) =  rk( 44) * y( 43)
dw( 44, 43) =  rk( 44) * y( 17)
dw( 45, 44) =  rk( 45) * y( 43)
dw( 45, 43) =  rk( 45) * y( 44)
dw( 46, 44) =  rk( 46) * y( 42)
dw( 46, 42) =  rk( 46) * y( 44)
dw( 47, 44) =  rk( 47) * y( 50)
dw( 47, 50) =  rk( 47) * y( 44)
dw( 48, 44) =  rk( 48) * y( 47)
dw( 48, 47) =  rk( 48) * y( 44)
dw( 49, 44) =  rk( 49) * y( 44)
dw( 49, 44) =  rk( 49) * y( 44)
dw( 50, 13) =  rk( 50)
dw( 51, 38) =  rk( 51)
dw( 52,  9) =  rk( 52)
dw( 53, 49) =  rk( 53) * y( 48)
dw( 53, 48) =  rk( 53) * y( 49)
dw( 54, 39) =  rk( 54) * y( 48)
dw( 54, 48) =  rk( 54) * y( 39)
dw( 55, 49) =  rk( 55) * y( 50)
dw( 55, 50) =  rk( 55) * y( 49)
dw( 56, 39) =  rk( 56) * y( 50)
dw( 56, 50) =  rk( 56) * y( 39)
dw( 57, 49) =  rk( 57) * y( 49)
dw( 57, 49) =  rk( 57) * y( 49)
dw( 58, 39) =  rk( 58) * y( 39)
dw( 58, 39) =  rk( 58) * y( 39)
dw( 59, 49) =  rk( 59) * y( 39)
dw( 59, 39) =  rk( 59) * y( 49)
dw( 60, 33) =  rk( 60) * y( 42)
dw( 60, 42) =  rk( 60) * y( 33)
dw( 61, 33) =  rk( 61)
dw( 62, 21) =  rk( 62) * y( 42)
dw( 62, 42) =  rk( 62) * y( 21)
dw( 63, 21) =  rk( 63)
dw( 64, 42) =  rk( 64) * y( 37)
dw( 64, 37) =  rk( 64) * y( 42)
dw( 65, 42) =  rk( 65) * y(  3)
dw( 65,  3) =  rk( 65) * y( 42)
dw( 66, 40) =  rk( 66) * y( 48)
dw( 66, 48) =  rk( 66) * y( 40)
dw( 67, 40) =  rk( 67) * y( 50)
dw( 67, 50) =  rk( 67) * y( 40)
dw( 68, 40) =  rk( 68) * y( 40)
dw( 68, 40) =  rk( 68) * y( 40)
dw( 69, 19) =  rk( 69) * y( 42)
dw( 69, 42) =  rk( 69) * y( 19)
dw( 70, 19) =  rk( 70)
dw( 71, 11) =  rk( 71) * y( 42)
dw( 71, 42) =  rk( 71) * y( 11)
dw( 72, 41) =  rk( 72) * y( 42)
dw( 72, 42) =  rk( 72) * y( 41)
dw( 73, 41) =  rk( 73)
dw( 74, 41) =  rk( 74)
dw( 75, 41) =  rk( 75) * y( 43)
dw( 75, 43) =  rk( 75) * y( 41)
dw( 76, 41) =  rk( 76) * y( 44)
dw( 76, 44) =  rk( 76) * y( 41)
dw( 77, 41) =  rk( 77) * y( 50)
dw( 77, 50) =  rk( 77) * y( 41)
dw( 78, 18) =  rk( 78)
dw( 79, 18) =  rk( 79) * y( 48)
dw( 79, 48) =  rk( 79) * y( 18)
dw( 80, 18) =  rk( 80) * y( 50)
dw( 80, 50) =  rk( 80) * y( 18)
dw( 81, 20) =  rk( 81) * y( 42)
dw( 81, 42) =  rk( 81) * y( 20)
dw( 82, 45) =  rk( 82) * y( 43)
dw( 82, 43) =  rk( 82) * y( 45)
dw( 83, 45) =  rk( 83) * y( 42)
dw( 83, 42) =  rk( 83) * y( 45)
dw( 84, 45) =  rk( 84) * y( 44)
dw( 84, 44) =  rk( 84) * y( 45)
dw( 85, 45) =  rk( 85)
dw( 86, 51) =  rk( 86) * y( 48)
dw( 86, 48) =  rk( 86) * y( 51)
dw( 87, 51) =  rk( 87) * y( 52)
dw( 87, 52) =  rk( 87) * y( 51)
dw( 88, 10) =  rk( 88)
dw( 89, 10) =  rk( 89)
dw( 90, 51) =  rk( 90) * y( 50)
dw( 90, 50) =  rk( 90) * y( 51)
dw( 91, 51) =  rk( 91) * y( 40)
dw( 91, 40) =  rk( 91) * y( 51)
dw( 92, 51) =  rk( 92) * y( 49)
dw( 92, 49) =  rk( 92) * y( 51)
dw( 93, 51) =  rk( 93) * y( 51)
dw( 93, 51) =  rk( 93) * y( 51)
dw( 94, 22) =  rk( 94) * y( 42)
dw( 94, 42) =  rk( 94) * y( 22)
dw( 95, 22) =  rk( 95)
dw( 96, 36) =  rk( 96) * y( 42)
dw( 96, 42) =  rk( 96) * y( 36)
dw( 97, 35) =  rk( 97) * y( 43)
dw( 97, 43) =  rk( 97) * y( 35)
dw( 98, 35) =  rk( 98) * y( 42)
dw( 98, 42) =  rk( 98) * y( 35)
dw( 99, 35) =  rk( 99) * y( 44)
dw( 99, 44) =  rk( 99) * y( 35)
dw(100, 35) =  rk(100)
dw(101, 46) =  rk(101) * y( 48)
dw(101, 48) =  rk(101) * y( 46)
dw(102, 46) =  rk(102) * y( 52)
dw(102, 52) =  rk(102) * y( 46)
dw(103, 14) =  rk(103)
dw(104, 14) =  rk(104)
dw(105, 14) =  rk(105) * y( 42)
dw(105, 42) =  rk(105) * y( 14)
dw(106, 46) =  rk(106) * y( 50)
dw(106, 50) =  rk(106) * y( 46)
dw(107, 51) =  rk(107) * y( 40)
dw(107, 40) =  rk(107) * y( 51)
dw(108, 51) =  rk(108) * y( 49)
dw(108, 49) =  rk(108) * y( 51)
dw(109, 46) =  rk(109) * y( 46)
dw(109, 46) =  rk(109) * y( 46)
dw(110, 46) =  rk(110) * y( 51)
dw(110, 51) =  rk(110) * y( 46)
dw(111, 34) =  rk(111) * y( 42)
dw(111, 42) =  rk(111) * y( 34)
dw(112, 16) =  rk(112)
dw(113, 16) =  rk(113)
dw(114, 16) =  rk(114) * y( 52)
dw(114, 52) =  rk(114) * y( 16)
dw(115, 43) =  rk(115) * y( 24)
dw(115, 24) =  rk(115) * y( 43)
dw(116, 42) =  rk(116) * y( 24)
dw(116, 24) =  rk(116) * y( 42)
dw(117, 47) =  rk(117) * y( 24)
dw(117, 24) =  rk(117) * y( 47)
dw(118, 44) =  rk(118) * y( 24)
dw(118, 24) =  rk(118) * y( 44)
dw(119, 43) =  rk(119) * y( 23)
dw(119, 23) =  rk(119) * y( 43)
dw(120, 42) =  rk(120) * y( 23)
dw(120, 23) =  rk(120) * y( 42)
dw(121, 47) =  rk(121) * y( 23)
dw(121, 23) =  rk(121) * y( 47)
dw(122, 44) =  rk(122) * y( 23)
dw(122, 23) =  rk(122) * y( 44)
dw(123, 25) =  rk(123) * y( 43)
dw(123, 43) =  rk(123) * y( 25)
dw(124, 25) =  rk(124) * y( 42)
dw(124, 42) =  rk(124) * y( 25)
dw(125, 25) =  rk(125) * y( 47)
dw(125, 47) =  rk(125) * y( 25)
dw(126, 25) =  rk(126) * y( 44)
dw(126, 44) =  rk(126) * y( 25)
dw(127,  5) =  rk(127) * y( 42)
dw(127, 42) =  rk(127) * y(  5)
dw(128, 12) =  rk(128) * y( 48)
dw(128, 48) =  rk(128) * y( 12)
dw(129, 12) =  rk(129)
dw(130, 42) =  rk(130) * y( 27)
dw(130, 27) =  rk(130) * y( 42)
dw(131, 27) =  rk(131) * y( 44)
dw(131, 44) =  rk(131) * y( 27)
dw(132, 30) =  rk(132) * y( 52)
dw(132, 52) =  rk(132) * y( 30)
dw(133, 30) =  rk(133) * y( 50)
dw(133, 50) =  rk(133) * y( 30)
dw(134, 28) =  rk(134)
dw(135, 28) =  rk(135) * y( 42)
dw(135, 42) =  rk(135) * y( 28)
dw(136, 28) =  rk(136) * y( 47)
dw(136, 47) =  rk(136) * y( 28)
dw(137, 42) =  rk(137) * y(  6)
dw(137,  6) =  rk(137) * y( 42)
dw(138, 42) =  rk(138) * y( 29)
dw(138, 29) =  rk(138) * y( 42)
dw(139, 29) =  rk(139)
dw(140, 43) =  rk(140) * y( 31)
dw(140, 31) =  rk(140) * y( 43)
dw(141, 42) =  rk(141) * y( 31)
dw(141, 31) =  rk(141) * y( 42)
dw(142, 47) =  rk(142) * y( 31)
dw(142, 31) =  rk(142) * y( 47)
dw(143, 44) =  rk(143) * y( 31)
dw(143, 31) =  rk(143) * y( 44)
dw(144, 42) =  rk(144) * y( 32)
dw(144, 32) =  rk(144) * y( 42)
dw(145, 47) =  rk(145) * y( 32)
dw(145, 32) =  rk(145) * y( 47)
dw(146, 44) =  rk(146) * y( 32)
dw(146, 32) =  rk(146) * y( 44)
dw(147, 32) =  rk(147)
dw(148, 26) =  rk(148) * y( 43)
dw(148, 43) =  rk(148) * y( 26)
dw(149, 26) =  rk(149) * y( 42)
dw(149, 42) =  rk(149) * y( 26)
dw(150, 26) =  rk(150) * y( 47)
dw(150, 47) =  rk(150) * y( 26)
dw(151, 26) =  rk(151) * y( 44)
dw(151, 44) =  rk(151) * y( 26)
dw(152,  7) =  rk(152) * y( 42)
dw(152, 42) =  rk(152) * y(  7)
dw(153, 42) =  rk(153) * y(  2)
dw(153,  2) =  rk(153) * y( 42)
dw(154, 42) =  rk(154) * y(  4)
dw(154,  4) =  rk(154) * y( 42)
dw(155, 52) =  rk(155) * y( 31)
dw(155, 31) =  rk(155) * y( 52)

return
end subroutine dratedc_3


!===============================================================================

!> lu_decompose_3
!> \brief Computation of LU factorization of matrix m
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     ns                 matrix row number from the chemical species
!>                                   number
!> \param[in,out] m                  on entry, an invertible matrix.
!>                                   On exit, an LU factorization of m
!______________________________________________________________________________

subroutine lu_decompose_3 (ns,m)

implicit none

! Arguments

integer ns
double precision m(ns,ns)

! Local variables

double precision temp

!     Upper part.
m(1, 47) = m(1, 47) / m(1, 1)

!     Upper part.
m(2, 42) = m(2, 42) / m(2, 2)

!     Upper part.
m(3, 42) = m(3, 42) / m(3, 3)

!     Upper part.
m(4, 42) = m(4, 42) / m(4, 4)

!     Upper part.
m(5, 42) = m(5, 42) / m(5, 5)

!     Upper part.
m(6, 42) = m(6, 42) / m(6, 6)

!     Upper part.
m(7, 42) = m(7, 42) / m(7, 7)

!     Upper part.
temp = m(8, 7) * m(7, 42)
m(8, 42) = ( m(8, 42) - temp ) / m(8, 8)

!     Upper part.
m(9, 44) = m(9, 44) / m(9, 9)
!     Upper part.
m(9, 52) = m(9, 52) / m(9, 9)

!     Upper part.
m(10, 51) = m(10, 51) / m(10, 10)
!     Upper part.
m(10, 52) = m(10, 52) / m(10, 10)

!     Upper part.
m(11, 40) = m(11, 40) / m(11, 11)
!     Upper part.
m(11, 42) = m(11, 42) / m(11, 11)

!     Upper part.
temp = m(12, 5) * m(5, 42)
temp = temp + m(12, 6) * m(6, 42)
m(12, 42) = ( m(12, 42) - temp ) / m(12, 12)
!     Upper part.
m(12, 48) = m(12, 48) / m(12, 12)

!     Upper part.
m(13, 42) = m(13, 42) / m(13, 13)
!     Upper part.
m(13, 50) = m(13, 50) / m(13, 13)
!     Upper part.
m(13, 52) = m(13, 52) / m(13, 13)

!     Upper part.
m(14, 42) = m(14, 42) / m(14, 14)
!     Upper part.
m(14, 46) = m(14, 46) / m(14, 14)
!     Upper part.
m(14, 52) = m(14, 52) / m(14, 14)

!     Upper part.
m(15, 42) = m(15, 42) / m(15, 15)
!     Upper part.
m(15, 48) = m(15, 48) / m(15, 15)
!     Upper part.
m(15, 52) = m(15, 52) / m(15, 15)

!     Upper part.
m(16, 34) = m(16, 34) / m(16, 16)
!     Upper part.
m(16, 42) = m(16, 42) / m(16, 16)
!     Upper part.
m(16, 52) = m(16, 52) / m(16, 16)

!     Upper part.
m(17, 42) = m(17, 42) / m(17, 17)
!     Upper part.
m(17, 43) = m(17, 43) / m(17, 17)
!     Upper part.
m(17, 50) = m(17, 50) / m(17, 17)

!     Upper part.
m(18, 41) = m(18, 41) / m(18, 18)
!     Upper part.
m(18, 48) = m(18, 48) / m(18, 18)
!     Upper part.
m(18, 50) = m(18, 50) / m(18, 18)

!     Upper part.
m(19, 40) = m(19, 40) / m(19, 19)
!     Upper part.
temp = m(19, 18) * m(18, 41)
m(19, 41) = ( m(19, 41) - temp ) / m(19, 19)
!     Upper part.
m(19, 42) = m(19, 42) / m(19, 19)
!     Upper part.
temp = m(19, 18) * m(18, 48)
m(19, 48) = ( m(19, 48) - temp ) / m(19, 19)
!     Upper part.
temp = m(19, 18) * m(18, 50)
m(19, 50) = ( m(19, 50) - temp ) / m(19, 19)

!     Upper part.
m(20, 23) = m(20, 23) / m(20, 20)
!     Upper part.
temp = m(20, 18) * m(18, 41)
m(20, 41) = ( m(20, 41) - temp ) / m(20, 20)
!     Upper part.
m(20, 42) = m(20, 42) / m(20, 20)
!     Upper part.
m(20, 47) = m(20, 47) / m(20, 20)
!     Upper part.
temp = m(20, 18) * m(18, 48)
m(20, 48) = ( m(20, 48) - temp ) / m(20, 20)
!     Upper part.
temp = m(20, 18) * m(18, 50)
m(20, 50) = ( m(20, 50) - temp ) / m(20, 20)

!     Upper part.
m(21, 39) = m(21, 39) / m(21, 21)
!     Upper part.
m(21, 42) = m(21, 42) / m(21, 21)
!     Upper part.
m(21, 49) = m(21, 49) / m(21, 21)
!     Upper part.
m(21, 50) = m(21, 50) / m(21, 21)

!     Upper part.
m(22, 42) = m(22, 42) / m(22, 22)
!     Upper part.
m(22, 46) = m(22, 46) / m(22, 22)
!     Upper part.
m(22, 50) = m(22, 50) / m(22, 22)
!     Upper part.
m(22, 51) = m(22, 51) / m(22, 22)

!     Lower part.
temp = m(42, 20) * m(20, 23)
m(42, 23) = m(42, 23) - temp
!     Lower part.
temp = m(50, 20) * m(20, 23)
m(50, 23) = m(50, 23) - temp
!     Upper part.
m(23, 42) = m(23, 42) / m(23, 23)
!     Upper part.
m(23, 43) = m(23, 43) / m(23, 23)
!     Upper part.
m(23, 44) = m(23, 44) / m(23, 23)
!     Upper part.
m(23, 47) = m(23, 47) / m(23, 23)

!     Upper part.
m(24, 42) = m(24, 42) / m(24, 24)
!     Upper part.
m(24, 43) = m(24, 43) / m(24, 24)
!     Upper part.
m(24, 44) = m(24, 44) / m(24, 24)
!     Upper part.
m(24, 47) = m(24, 47) / m(24, 24)

!     Upper part.
m(25, 42) = m(25, 42) / m(25, 25)
!     Upper part.
m(25, 43) = m(25, 43) / m(25, 25)
!     Upper part.
m(25, 44) = m(25, 44) / m(25, 25)
!     Upper part.
m(25, 47) = m(25, 47) / m(25, 25)

!     Upper part.
m(26, 42) = m(26, 42) / m(26, 26)
!     Upper part.
m(26, 43) = m(26, 43) / m(26, 26)
!     Upper part.
m(26, 44) = m(26, 44) / m(26, 26)
!     Upper part.
m(26, 47) = m(26, 47) / m(26, 26)

!     Upper part.
m(27, 30) = m(27, 30) / m(27, 27)
!     Upper part.
temp = m(27, 5) * m(5, 42)
temp = temp + m(27, 6) * m(6, 42)
temp = temp + m(27, 12) * m(12, 42)
m(27, 42) = ( m(27, 42) - temp ) / m(27, 27)
!     Upper part.
m(27, 44) = m(27, 44) / m(27, 27)
!     Upper part.
temp = m(27, 12) * m(12, 48)
m(27, 48) = ( m(27, 48) - temp ) / m(27, 27)
!     Upper part.
m(27, 50) = m(27, 50) / m(27, 27)

!     Upper part.
temp = m(28, 27) * m(27, 30)
m(28, 30) = ( m(28, 30) - temp ) / m(28, 28)
!     Upper part.
temp = m(28, 12) * m(12, 42)
temp = temp + m(28, 27) * m(27, 42)
m(28, 42) = ( m(28, 42) - temp ) / m(28, 28)
!     Upper part.
temp = m(28, 27) * m(27, 44)
m(28, 44) = ( m(28, 44) - temp ) / m(28, 28)
!     Upper part.
m(28, 47) = m(28, 47) / m(28, 28)
!     Upper part.
temp = m(28, 12) * m(12, 48)
temp = temp + m(28, 27) * m(27, 48)
m(28, 48) = ( m(28, 48) - temp ) / m(28, 28)
!     Upper part.
temp = m(28, 27) * m(27, 50)
m(28, 50) = ( m(28, 50) - temp ) / m(28, 28)

!     Upper part.
temp = m(29, 28) * m(28, 30)
m(29, 30) = ( m(29, 30) - temp ) / m(29, 29)
!     Upper part.
m(29, 32) = m(29, 32) / m(29, 29)
!     Upper part.
temp = m(29, 6) * m(6, 42)
temp = temp + m(29, 28) * m(28, 42)
m(29, 42) = ( m(29, 42) - temp ) / m(29, 29)
!     Upper part.
temp = m(29, 28) * m(28, 44)
m(29, 44) = ( m(29, 44) - temp ) / m(29, 29)
!     Upper part.
temp = m(29, 28) * m(28, 47)
m(29, 47) = ( m(29, 47) - temp ) / m(29, 29)
!     Upper part.
temp = m(29, 28) * m(28, 48)
m(29, 48) = ( m(29, 48) - temp ) / m(29, 29)
!     Upper part.
temp = m(29, 28) * m(28, 50)
m(29, 50) = ( m(29, 50) - temp ) / m(29, 29)

!     Lower part.
temp = m(30, 27) * m(27, 30)
m(30, 30) = m(30, 30) - temp
!     Lower part.
temp = m(35, 28) * m(28, 30)
m(35, 30) = m(35, 30) - temp
!     Lower part.
temp = m(37, 28) * m(28, 30)
temp = temp + m(37, 29) * m(29, 30)
m(37, 30) = m(37, 30) - temp
!     Lower part.
temp = m(38, 27) * m(27, 30)
m(38, 30) = m(38, 30) - temp
!     Lower part.
temp = m(41, 28) * m(28, 30)
m(41, 30) = m(41, 30) - temp
!     Lower part.
temp = m(42, 27) * m(27, 30)
temp = temp + m(42, 28) * m(28, 30)
temp = temp + m(42, 29) * m(29, 30)
m(42, 30) = m(42, 30) - temp
!     Lower part.
temp = m(44, 27) * m(27, 30)
m(44, 30) = m(44, 30) - temp
!     Lower part.
temp = m(47, 28) * m(28, 30)
m(47, 30) = m(47, 30) - temp
!     Lower part.
temp = m(49, 27) * m(27, 30)
temp = temp + m(49, 28) * m(28, 30)
temp = temp + m(49, 29) * m(29, 30)
m(49, 30) = m(49, 30) - temp
!     Lower part.
temp = m(50, 27) * m(27, 30)
temp = temp + m(50, 28) * m(28, 30)
temp = temp + m(50, 29) * m(29, 30)
m(50, 30) = m(50, 30) - temp
!     Lower part.
temp = m(51, 28) * m(28, 30)
temp = temp + m(51, 29) * m(29, 30)
m(51, 30) = m(51, 30) - temp
!     Upper part.
temp = m(30, 27) * m(27, 42)
m(30, 42) = ( m(30, 42) - temp ) / m(30, 30)
!     Upper part.
temp = m(30, 27) * m(27, 44)
m(30, 44) = ( m(30, 44) - temp ) / m(30, 30)
!     Upper part.
temp = m(30, 27) * m(27, 48)
m(30, 48) = ( m(30, 48) - temp ) / m(30, 30)
!     Upper part.
temp = m(30, 27) * m(27, 50)
m(30, 50) = ( m(30, 50) - temp ) / m(30, 30)
!     Upper part.
m(30, 52) = m(30, 52) / m(30, 30)

!     Upper part.
m(31, 42) = m(31, 42) / m(31, 31)
!     Upper part.
m(31, 43) = m(31, 43) / m(31, 31)
!     Upper part.
m(31, 44) = m(31, 44) / m(31, 31)
!     Upper part.
m(31, 47) = m(31, 47) / m(31, 31)
!     Upper part.
m(31, 52) = m(31, 52) / m(31, 31)

!     Lower part.
temp = m(37, 29) * m(29, 32)
m(37, 32) = m(37, 32) - temp
!     Lower part.
temp = m(42, 29) * m(29, 32)
m(42, 32) = m(42, 32) - temp
!     Lower part.
temp = m(49, 29) * m(29, 32)
m(49, 32) = m(49, 32) - temp
!     Lower part.
temp = m(50, 29) * m(29, 32)
m(50, 32) = m(50, 32) - temp
!     Lower part.
temp = m(51, 29) * m(29, 32)
m(51, 32) = m(51, 32) - temp
!     Upper part.
temp = m(32, 31) * m(31, 42)
m(32, 42) = ( m(32, 42) - temp ) / m(32, 32)
!     Upper part.
temp = m(32, 31) * m(31, 43)
m(32, 43) = ( m(32, 43) - temp ) / m(32, 32)
!     Upper part.
temp = m(32, 31) * m(31, 44)
m(32, 44) = ( m(32, 44) - temp ) / m(32, 32)
!     Upper part.
temp = m(32, 31) * m(31, 47)
m(32, 47) = ( m(32, 47) - temp ) / m(32, 32)
!     Upper part.
temp = m(32, 31) * m(31, 52)
m(32, 52) = ( m(32, 52) - temp ) / m(32, 32)

!     Upper part.
temp = m(33, 16) * m(16, 34)
m(33, 34) = ( m(33, 34) - temp ) / m(33, 33)
!     Upper part.
m(33, 39) = m(33, 39) / m(33, 33)
!     Upper part.
temp = m(33, 12) * m(12, 42)
temp = temp + m(33, 16) * m(16, 42)
temp = temp + m(33, 26) * m(26, 42)
temp = temp + m(33, 30) * m(30, 42)
temp = temp + m(33, 31) * m(31, 42)
temp = temp + m(33, 32) * m(32, 42)
m(33, 42) = ( m(33, 42) - temp ) / m(33, 33)
!     Upper part.
temp = m(33, 26) * m(26, 43)
temp = temp + m(33, 31) * m(31, 43)
temp = temp + m(33, 32) * m(32, 43)
m(33, 43) = ( m(33, 43) - temp ) / m(33, 33)
!     Upper part.
temp = m(33, 26) * m(26, 44)
temp = temp + m(33, 30) * m(30, 44)
temp = temp + m(33, 31) * m(31, 44)
temp = temp + m(33, 32) * m(32, 44)
m(33, 44) = ( m(33, 44) - temp ) / m(33, 33)
!     Upper part.
temp = m(33, 26) * m(26, 47)
temp = temp + m(33, 31) * m(31, 47)
temp = temp + m(33, 32) * m(32, 47)
m(33, 47) = ( m(33, 47) - temp ) / m(33, 33)
!     Upper part.
temp = m(33, 12) * m(12, 48)
temp = temp + m(33, 30) * m(30, 48)
m(33, 48) = ( m(33, 48) - temp ) / m(33, 33)
!     Upper part.
temp = m(33, 30) * m(30, 50)
m(33, 50) = ( m(33, 50) - temp ) / m(33, 33)
!     Upper part.
temp = m(33, 16) * m(16, 52)
temp = temp + m(33, 30) * m(30, 52)
temp = temp + m(33, 31) * m(31, 52)
temp = temp + m(33, 32) * m(32, 52)
m(33, 52) = ( m(33, 52) - temp ) / m(33, 33)

!     Lower part.
temp = m(34, 16) * m(16, 34)
temp = temp + m(34, 33) * m(33, 34)
m(34, 34) = m(34, 34) - temp
!     Lower part.
temp = m(35, 16) * m(16, 34)
temp = temp + m(35, 33) * m(33, 34)
m(35, 34) = m(35, 34) - temp
!     Lower part.
temp = m(38, 33) * m(33, 34)
m(38, 34) = m(38, 34) - temp
!     Lower part.
temp = m(39, 16) * m(16, 34)
m(39, 34) = m(39, 34) - temp
!     Lower part.
temp = m(41, 33) * m(33, 34)
m(41, 34) = m(41, 34) - temp
!     Lower part.
temp = m(42, 33) * m(33, 34)
m(42, 34) = m(42, 34) - temp
!     Lower part.
temp = m(45, 16) * m(16, 34)
temp = temp + m(45, 33) * m(33, 34)
m(45, 34) = m(45, 34) - temp
!     Lower part.
temp = m(49, 16) * m(16, 34)
m(49, 34) = m(49, 34) - temp
!     Lower part.
temp = m(50, 16) * m(16, 34)
temp = temp + m(50, 33) * m(33, 34)
m(50, 34) = m(50, 34) - temp
!     Lower part.
temp = m(52, 16) * m(16, 34)
temp = temp + m(52, 33) * m(33, 34)
m(52, 34) = m(52, 34) - temp
!     Upper part.
temp = m(34, 33) * m(33, 39)
m(34, 39) = ( m(34, 39) - temp ) / m(34, 34)
!     Upper part.
temp = m(34, 6) * m(6, 42)
temp = temp + m(34, 16) * m(16, 42)
temp = temp + m(34, 24) * m(24, 42)
temp = temp + m(34, 25) * m(25, 42)
temp = temp + m(34, 26) * m(26, 42)
temp = temp + m(34, 31) * m(31, 42)
temp = temp + m(34, 32) * m(32, 42)
temp = temp + m(34, 33) * m(33, 42)
m(34, 42) = ( m(34, 42) - temp ) / m(34, 34)
!     Upper part.
temp = m(34, 24) * m(24, 43)
temp = temp + m(34, 25) * m(25, 43)
temp = temp + m(34, 26) * m(26, 43)
temp = temp + m(34, 31) * m(31, 43)
temp = temp + m(34, 32) * m(32, 43)
temp = temp + m(34, 33) * m(33, 43)
m(34, 43) = ( m(34, 43) - temp ) / m(34, 34)
!     Upper part.
temp = m(34, 24) * m(24, 44)
temp = temp + m(34, 25) * m(25, 44)
temp = temp + m(34, 26) * m(26, 44)
temp = temp + m(34, 31) * m(31, 44)
temp = temp + m(34, 32) * m(32, 44)
temp = temp + m(34, 33) * m(33, 44)
m(34, 44) = ( m(34, 44) - temp ) / m(34, 34)
!     Upper part.
temp = m(34, 24) * m(24, 47)
temp = temp + m(34, 25) * m(25, 47)
temp = temp + m(34, 26) * m(26, 47)
temp = temp + m(34, 31) * m(31, 47)
temp = temp + m(34, 32) * m(32, 47)
temp = temp + m(34, 33) * m(33, 47)
m(34, 47) = ( m(34, 47) - temp ) / m(34, 34)
!     Upper part.
temp = m(34, 33) * m(33, 48)
m(34, 48) = ( m(34, 48) - temp ) / m(34, 34)
!     Upper part.
temp = m(34, 33) * m(33, 50)
m(34, 50) = ( m(34, 50) - temp ) / m(34, 34)
!     Upper part.
temp = m(34, 16) * m(16, 52)
temp = temp + m(34, 31) * m(31, 52)
temp = temp + m(34, 32) * m(32, 52)
temp = temp + m(34, 33) * m(33, 52)
m(34, 52) = ( m(34, 52) - temp ) / m(34, 34)

!     Upper part.
temp = m(35, 21) * m(21, 39)
temp = temp + m(35, 33) * m(33, 39)
temp = temp + m(35, 34) * m(34, 39)
m(35, 39) = ( m(35, 39) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 2) * m(2, 42)
temp = temp + m(35, 16) * m(16, 42)
temp = temp + m(35, 21) * m(21, 42)
temp = temp + m(35, 23) * m(23, 42)
temp = temp + m(35, 24) * m(24, 42)
temp = temp + m(35, 25) * m(25, 42)
temp = temp + m(35, 26) * m(26, 42)
temp = temp + m(35, 28) * m(28, 42)
temp = temp + m(35, 30) * m(30, 42)
temp = temp + m(35, 31) * m(31, 42)
temp = temp + m(35, 32) * m(32, 42)
temp = temp + m(35, 33) * m(33, 42)
temp = temp + m(35, 34) * m(34, 42)
m(35, 42) = ( m(35, 42) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 23) * m(23, 43)
temp = temp + m(35, 24) * m(24, 43)
temp = temp + m(35, 25) * m(25, 43)
temp = temp + m(35, 26) * m(26, 43)
temp = temp + m(35, 31) * m(31, 43)
temp = temp + m(35, 32) * m(32, 43)
temp = temp + m(35, 33) * m(33, 43)
temp = temp + m(35, 34) * m(34, 43)
m(35, 43) = ( m(35, 43) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 23) * m(23, 44)
temp = temp + m(35, 24) * m(24, 44)
temp = temp + m(35, 25) * m(25, 44)
temp = temp + m(35, 26) * m(26, 44)
temp = temp + m(35, 28) * m(28, 44)
temp = temp + m(35, 30) * m(30, 44)
temp = temp + m(35, 31) * m(31, 44)
temp = temp + m(35, 32) * m(32, 44)
temp = temp + m(35, 33) * m(33, 44)
temp = temp + m(35, 34) * m(34, 44)
m(35, 44) = ( m(35, 44) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 23) * m(23, 47)
temp = temp + m(35, 24) * m(24, 47)
temp = temp + m(35, 25) * m(25, 47)
temp = temp + m(35, 26) * m(26, 47)
temp = temp + m(35, 28) * m(28, 47)
temp = temp + m(35, 31) * m(31, 47)
temp = temp + m(35, 32) * m(32, 47)
temp = temp + m(35, 33) * m(33, 47)
temp = temp + m(35, 34) * m(34, 47)
m(35, 47) = ( m(35, 47) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 28) * m(28, 48)
temp = temp + m(35, 30) * m(30, 48)
temp = temp + m(35, 33) * m(33, 48)
temp = temp + m(35, 34) * m(34, 48)
m(35, 48) = ( m(35, 48) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 21) * m(21, 49)
m(35, 49) = ( m(35, 49) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 21) * m(21, 50)
temp = temp + m(35, 28) * m(28, 50)
temp = temp + m(35, 30) * m(30, 50)
temp = temp + m(35, 33) * m(33, 50)
temp = temp + m(35, 34) * m(34, 50)
m(35, 50) = ( m(35, 50) - temp ) / m(35, 35)
!     Upper part.
temp = m(35, 16) * m(16, 52)
temp = temp + m(35, 30) * m(30, 52)
temp = temp + m(35, 31) * m(31, 52)
temp = temp + m(35, 32) * m(32, 52)
temp = temp + m(35, 33) * m(33, 52)
temp = temp + m(35, 34) * m(34, 52)
m(35, 52) = ( m(35, 52) - temp ) / m(35, 35)

!     Upper part.
m(36, 40) = m(36, 40) / m(36, 36)
!     Upper part.
m(36, 42) = m(36, 42) / m(36, 36)
!     Upper part.
m(36, 46) = m(36, 46) / m(36, 36)
!     Upper part.
m(36, 49) = m(36, 49) / m(36, 36)
!     Upper part.
m(36, 50) = m(36, 50) / m(36, 36)
!     Upper part.
m(36, 51) = m(36, 51) / m(36, 36)

!     Upper part.
temp = m(37, 35) * m(35, 39)
m(37, 39) = ( m(37, 39) - temp ) / m(37, 37)
!     Upper part.
m(37, 41) = m(37, 41) / m(37, 37)
!     Upper part.
temp = m(37, 23) * m(23, 42)
temp = temp + m(37, 24) * m(24, 42)
temp = temp + m(37, 25) * m(25, 42)
temp = temp + m(37, 26) * m(26, 42)
temp = temp + m(37, 28) * m(28, 42)
temp = temp + m(37, 29) * m(29, 42)
temp = temp + m(37, 30) * m(30, 42)
temp = temp + m(37, 31) * m(31, 42)
temp = temp + m(37, 32) * m(32, 42)
temp = temp + m(37, 35) * m(35, 42)
m(37, 42) = ( m(37, 42) - temp ) / m(37, 37)
!     Upper part.
temp = m(37, 23) * m(23, 43)
temp = temp + m(37, 24) * m(24, 43)
temp = temp + m(37, 25) * m(25, 43)
temp = temp + m(37, 26) * m(26, 43)
temp = temp + m(37, 31) * m(31, 43)
temp = temp + m(37, 32) * m(32, 43)
temp = temp + m(37, 35) * m(35, 43)
m(37, 43) = ( m(37, 43) - temp ) / m(37, 37)
!     Upper part.
temp = m(37, 23) * m(23, 44)
temp = temp + m(37, 24) * m(24, 44)
temp = temp + m(37, 25) * m(25, 44)
temp = temp + m(37, 26) * m(26, 44)
temp = temp + m(37, 28) * m(28, 44)
temp = temp + m(37, 29) * m(29, 44)
temp = temp + m(37, 30) * m(30, 44)
temp = temp + m(37, 31) * m(31, 44)
temp = temp + m(37, 32) * m(32, 44)
temp = temp + m(37, 35) * m(35, 44)
m(37, 44) = ( m(37, 44) - temp ) / m(37, 37)
!     Upper part.
m(37, 45) = m(37, 45) / m(37, 37)
!     Upper part.
temp = m(37, 23) * m(23, 47)
temp = temp + m(37, 24) * m(24, 47)
temp = temp + m(37, 25) * m(25, 47)
temp = temp + m(37, 26) * m(26, 47)
temp = temp + m(37, 28) * m(28, 47)
temp = temp + m(37, 29) * m(29, 47)
temp = temp + m(37, 31) * m(31, 47)
temp = temp + m(37, 32) * m(32, 47)
temp = temp + m(37, 35) * m(35, 47)
m(37, 47) = ( m(37, 47) - temp ) / m(37, 37)
!     Upper part.
temp = m(37, 28) * m(28, 48)
temp = temp + m(37, 29) * m(29, 48)
temp = temp + m(37, 30) * m(30, 48)
temp = temp + m(37, 35) * m(35, 48)
m(37, 48) = ( m(37, 48) - temp ) / m(37, 37)
!     Upper part.
temp = m(37, 35) * m(35, 49)
m(37, 49) = ( m(37, 49) - temp ) / m(37, 37)
!     Upper part.
temp = m(37, 28) * m(28, 50)
temp = temp + m(37, 29) * m(29, 50)
temp = temp + m(37, 30) * m(30, 50)
temp = temp + m(37, 35) * m(35, 50)
m(37, 50) = ( m(37, 50) - temp ) / m(37, 37)
!     Upper part.
temp = m(37, 30) * m(30, 52)
temp = temp + m(37, 31) * m(31, 52)
temp = temp + m(37, 32) * m(32, 52)
temp = temp + m(37, 35) * m(35, 52)
m(37, 52) = ( m(37, 52) - temp ) / m(37, 37)

!     Upper part.
temp = m(38, 33) * m(33, 39)
temp = temp + m(38, 34) * m(34, 39)
temp = temp + m(38, 35) * m(35, 39)
m(38, 39) = ( m(38, 39) - temp ) / m(38, 38)
!     Upper part.
m(38, 41) = m(38, 41) / m(38, 38)
!     Upper part.
temp = m(38, 27) * m(27, 42)
temp = temp + m(38, 30) * m(30, 42)
temp = temp + m(38, 32) * m(32, 42)
temp = temp + m(38, 33) * m(33, 42)
temp = temp + m(38, 34) * m(34, 42)
temp = temp + m(38, 35) * m(35, 42)
m(38, 42) = ( m(38, 42) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 32) * m(32, 43)
temp = temp + m(38, 33) * m(33, 43)
temp = temp + m(38, 34) * m(34, 43)
temp = temp + m(38, 35) * m(35, 43)
m(38, 43) = ( m(38, 43) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 9) * m(9, 44)
temp = temp + m(38, 27) * m(27, 44)
temp = temp + m(38, 30) * m(30, 44)
temp = temp + m(38, 32) * m(32, 44)
temp = temp + m(38, 33) * m(33, 44)
temp = temp + m(38, 34) * m(34, 44)
temp = temp + m(38, 35) * m(35, 44)
m(38, 44) = ( m(38, 44) - temp ) / m(38, 38)
!     Upper part.
m(38, 45) = m(38, 45) / m(38, 38)
!     Upper part.
temp = m(38, 32) * m(32, 47)
temp = temp + m(38, 33) * m(33, 47)
temp = temp + m(38, 34) * m(34, 47)
temp = temp + m(38, 35) * m(35, 47)
m(38, 47) = ( m(38, 47) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 27) * m(27, 48)
temp = temp + m(38, 30) * m(30, 48)
temp = temp + m(38, 33) * m(33, 48)
temp = temp + m(38, 34) * m(34, 48)
temp = temp + m(38, 35) * m(35, 48)
m(38, 48) = ( m(38, 48) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 35) * m(35, 49)
m(38, 49) = ( m(38, 49) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 27) * m(27, 50)
temp = temp + m(38, 30) * m(30, 50)
temp = temp + m(38, 33) * m(33, 50)
temp = temp + m(38, 34) * m(34, 50)
temp = temp + m(38, 35) * m(35, 50)
m(38, 50) = ( m(38, 50) - temp ) / m(38, 38)
!     Upper part.
temp = m(38, 9) * m(9, 52)
temp = temp + m(38, 30) * m(30, 52)
temp = temp + m(38, 32) * m(32, 52)
temp = temp + m(38, 33) * m(33, 52)
temp = temp + m(38, 34) * m(34, 52)
temp = temp + m(38, 35) * m(35, 52)
m(38, 52) = ( m(38, 52) - temp ) / m(38, 38)

!     Lower part.
temp = m(39, 34) * m(34, 39)
m(39, 39) = m(39, 39) - temp
!     Lower part.
temp = m(40, 35) * m(35, 39)
m(40, 39) = m(40, 39) - temp
!     Lower part.
temp = m(41, 33) * m(33, 39)
temp = temp + m(41, 34) * m(34, 39)
m(41, 39) = m(41, 39) - temp
!     Lower part.
temp = m(42, 21) * m(21, 39)
temp = temp + m(42, 33) * m(33, 39)
temp = temp + m(42, 34) * m(34, 39)
temp = temp + m(42, 35) * m(35, 39)
temp = temp + m(42, 37) * m(37, 39)
temp = temp + m(42, 38) * m(38, 39)
m(42, 39) = m(42, 39) - temp
!     Lower part.
temp = m(43, 35) * m(35, 39)
m(43, 39) = m(43, 39) - temp
!     Lower part.
temp = m(44, 35) * m(35, 39)
temp = temp + m(44, 38) * m(38, 39)
m(44, 39) = m(44, 39) - temp
!     Lower part.
temp = m(45, 21) * m(21, 39)
temp = temp + m(45, 33) * m(33, 39)
temp = temp + m(45, 34) * m(34, 39)
m(45, 39) = m(45, 39) - temp
!     Lower part.
temp = m(46, 35) * m(35, 39)
m(46, 39) = m(46, 39) - temp
!     Lower part.
temp = m(49, 21) * m(21, 39)
temp = temp + m(49, 34) * m(34, 39)
m(49, 39) = m(49, 39) - temp
!     Lower part.
temp = m(50, 21) * m(21, 39)
temp = temp + m(50, 33) * m(33, 39)
temp = temp + m(50, 34) * m(34, 39)
temp = temp + m(50, 35) * m(35, 39)
temp = temp + m(50, 37) * m(37, 39)
m(50, 39) = m(50, 39) - temp
!     Lower part.
temp = m(52, 33) * m(33, 39)
temp = temp + m(52, 34) * m(34, 39)
temp = temp + m(52, 38) * m(38, 39)
m(52, 39) = m(52, 39) - temp
!     Upper part.
temp = m(39, 4) * m(4, 42)
temp = temp + m(39, 16) * m(16, 42)
temp = temp + m(39, 24) * m(24, 42)
temp = temp + m(39, 26) * m(26, 42)
temp = temp + m(39, 31) * m(31, 42)
temp = temp + m(39, 34) * m(34, 42)
m(39, 42) = ( m(39, 42) - temp ) / m(39, 39)
!     Upper part.
temp = m(39, 24) * m(24, 43)
temp = temp + m(39, 26) * m(26, 43)
temp = temp + m(39, 31) * m(31, 43)
temp = temp + m(39, 34) * m(34, 43)
m(39, 43) = ( m(39, 43) - temp ) / m(39, 39)
!     Upper part.
temp = m(39, 24) * m(24, 44)
temp = temp + m(39, 26) * m(26, 44)
temp = temp + m(39, 31) * m(31, 44)
temp = temp + m(39, 34) * m(34, 44)
m(39, 44) = ( m(39, 44) - temp ) / m(39, 39)
!     Upper part.
temp = m(39, 24) * m(24, 47)
temp = temp + m(39, 26) * m(26, 47)
temp = temp + m(39, 31) * m(31, 47)
temp = temp + m(39, 34) * m(34, 47)
m(39, 47) = ( m(39, 47) - temp ) / m(39, 39)
!     Upper part.
temp = m(39, 34) * m(34, 48)
m(39, 48) = ( m(39, 48) - temp ) / m(39, 39)
!     Upper part.
m(39, 49) = m(39, 49) / m(39, 39)
!     Upper part.
temp = m(39, 34) * m(34, 50)
m(39, 50) = ( m(39, 50) - temp ) / m(39, 39)
!     Upper part.
temp = m(39, 16) * m(16, 52)
temp = temp + m(39, 31) * m(31, 52)
temp = temp + m(39, 34) * m(34, 52)
m(39, 52) = ( m(39, 52) - temp ) / m(39, 39)

!     Lower part.
temp = m(40, 19) * m(19, 40)
temp = temp + m(40, 36) * m(36, 40)
m(40, 40) = m(40, 40) - temp
!     Lower part.
temp = m(41, 11) * m(11, 40)
temp = temp + m(41, 19) * m(19, 40)
m(41, 40) = m(41, 40) - temp
!     Lower part.
temp = m(42, 11) * m(11, 40)
temp = temp + m(42, 19) * m(19, 40)
temp = temp + m(42, 36) * m(36, 40)
m(42, 40) = m(42, 40) - temp
!     Lower part.
temp = m(49, 19) * m(19, 40)
m(49, 40) = m(49, 40) - temp
!     Lower part.
temp = m(50, 11) * m(11, 40)
temp = temp + m(50, 19) * m(19, 40)
m(50, 40) = m(50, 40) - temp
!     Upper part.
temp = m(40, 19) * m(19, 41)
m(40, 41) = ( m(40, 41) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 3) * m(3, 42)
temp = temp + m(40, 19) * m(19, 42)
temp = temp + m(40, 22) * m(22, 42)
temp = temp + m(40, 35) * m(35, 42)
temp = temp + m(40, 36) * m(36, 42)
temp = temp + m(40, 39) * m(39, 42)
m(40, 42) = ( m(40, 42) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 35) * m(35, 43)
temp = temp + m(40, 39) * m(39, 43)
m(40, 43) = ( m(40, 43) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 35) * m(35, 44)
temp = temp + m(40, 39) * m(39, 44)
m(40, 44) = ( m(40, 44) - temp ) / m(40, 40)
!     Upper part.
m(40, 45) = m(40, 45) / m(40, 40)
!     Upper part.
temp = m(40, 22) * m(22, 46)
temp = temp + m(40, 36) * m(36, 46)
m(40, 46) = ( m(40, 46) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 35) * m(35, 47)
temp = temp + m(40, 39) * m(39, 47)
m(40, 47) = ( m(40, 47) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 19) * m(19, 48)
temp = temp + m(40, 35) * m(35, 48)
temp = temp + m(40, 39) * m(39, 48)
m(40, 48) = ( m(40, 48) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 35) * m(35, 49)
temp = temp + m(40, 36) * m(36, 49)
temp = temp + m(40, 39) * m(39, 49)
m(40, 49) = ( m(40, 49) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 19) * m(19, 50)
temp = temp + m(40, 22) * m(22, 50)
temp = temp + m(40, 35) * m(35, 50)
temp = temp + m(40, 36) * m(36, 50)
temp = temp + m(40, 39) * m(39, 50)
m(40, 50) = ( m(40, 50) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 22) * m(22, 51)
temp = temp + m(40, 36) * m(36, 51)
m(40, 51) = ( m(40, 51) - temp ) / m(40, 40)
!     Upper part.
temp = m(40, 35) * m(35, 52)
temp = temp + m(40, 39) * m(39, 52)
m(40, 52) = ( m(40, 52) - temp ) / m(40, 40)

!     Lower part.
temp = m(41, 18) * m(18, 41)
temp = temp + m(41, 19) * m(19, 41)
temp = temp + m(41, 40) * m(40, 41)
m(41, 41) = m(41, 41) - temp
!     Lower part.
temp = m(42, 19) * m(19, 41)
temp = temp + m(42, 20) * m(20, 41)
temp = temp + m(42, 37) * m(37, 41)
temp = temp + m(42, 38) * m(38, 41)
temp = temp + m(42, 40) * m(40, 41)
m(42, 41) = m(42, 41) - temp
!     Lower part.
temp = m(44, 38) * m(38, 41)
m(44, 41) = m(44, 41) - temp
!     Lower part.
temp = m(45, 40) * m(40, 41)
m(45, 41) = m(45, 41) - temp
!     Lower part.
temp = m(48, 18) * m(18, 41)
temp = temp + m(48, 40) * m(40, 41)
m(48, 41) = m(48, 41) - temp
!     Lower part.
temp = m(49, 19) * m(19, 41)
temp = temp + m(49, 40) * m(40, 41)
m(49, 41) = m(49, 41) - temp
!     Lower part.
temp = m(50, 18) * m(18, 41)
temp = temp + m(50, 19) * m(19, 41)
temp = temp + m(50, 20) * m(20, 41)
temp = temp + m(50, 37) * m(37, 41)
temp = temp + m(50, 40) * m(40, 41)
m(50, 41) = m(50, 41) - temp
!     Lower part.
temp = m(51, 40) * m(40, 41)
m(51, 41) = m(51, 41) - temp
!     Lower part.
temp = m(52, 18) * m(18, 41)
temp = temp + m(52, 38) * m(38, 41)
temp = temp + m(52, 40) * m(40, 41)
m(52, 41) = m(52, 41) - temp
!     Upper part.
temp = m(41, 2) * m(2, 42)
temp = temp + m(41, 11) * m(11, 42)
temp = temp + m(41, 19) * m(19, 42)
temp = temp + m(41, 23) * m(23, 42)
temp = temp + m(41, 24) * m(24, 42)
temp = temp + m(41, 25) * m(25, 42)
temp = temp + m(41, 26) * m(26, 42)
temp = temp + m(41, 28) * m(28, 42)
temp = temp + m(41, 30) * m(30, 42)
temp = temp + m(41, 31) * m(31, 42)
temp = temp + m(41, 32) * m(32, 42)
temp = temp + m(41, 33) * m(33, 42)
temp = temp + m(41, 34) * m(34, 42)
temp = temp + m(41, 39) * m(39, 42)
temp = temp + m(41, 40) * m(40, 42)
m(41, 42) = ( m(41, 42) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 23) * m(23, 43)
temp = temp + m(41, 24) * m(24, 43)
temp = temp + m(41, 25) * m(25, 43)
temp = temp + m(41, 26) * m(26, 43)
temp = temp + m(41, 31) * m(31, 43)
temp = temp + m(41, 32) * m(32, 43)
temp = temp + m(41, 33) * m(33, 43)
temp = temp + m(41, 34) * m(34, 43)
temp = temp + m(41, 39) * m(39, 43)
temp = temp + m(41, 40) * m(40, 43)
m(41, 43) = ( m(41, 43) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 23) * m(23, 44)
temp = temp + m(41, 24) * m(24, 44)
temp = temp + m(41, 25) * m(25, 44)
temp = temp + m(41, 26) * m(26, 44)
temp = temp + m(41, 28) * m(28, 44)
temp = temp + m(41, 30) * m(30, 44)
temp = temp + m(41, 31) * m(31, 44)
temp = temp + m(41, 32) * m(32, 44)
temp = temp + m(41, 33) * m(33, 44)
temp = temp + m(41, 34) * m(34, 44)
temp = temp + m(41, 39) * m(39, 44)
temp = temp + m(41, 40) * m(40, 44)
m(41, 44) = ( m(41, 44) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 40) * m(40, 45)
m(41, 45) = ( m(41, 45) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 40) * m(40, 46)
m(41, 46) = ( m(41, 46) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 23) * m(23, 47)
temp = temp + m(41, 24) * m(24, 47)
temp = temp + m(41, 25) * m(25, 47)
temp = temp + m(41, 26) * m(26, 47)
temp = temp + m(41, 28) * m(28, 47)
temp = temp + m(41, 31) * m(31, 47)
temp = temp + m(41, 32) * m(32, 47)
temp = temp + m(41, 33) * m(33, 47)
temp = temp + m(41, 34) * m(34, 47)
temp = temp + m(41, 39) * m(39, 47)
temp = temp + m(41, 40) * m(40, 47)
m(41, 47) = ( m(41, 47) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 18) * m(18, 48)
temp = temp + m(41, 19) * m(19, 48)
temp = temp + m(41, 28) * m(28, 48)
temp = temp + m(41, 30) * m(30, 48)
temp = temp + m(41, 33) * m(33, 48)
temp = temp + m(41, 34) * m(34, 48)
temp = temp + m(41, 39) * m(39, 48)
temp = temp + m(41, 40) * m(40, 48)
m(41, 48) = ( m(41, 48) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 39) * m(39, 49)
temp = temp + m(41, 40) * m(40, 49)
m(41, 49) = ( m(41, 49) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 18) * m(18, 50)
temp = temp + m(41, 19) * m(19, 50)
temp = temp + m(41, 28) * m(28, 50)
temp = temp + m(41, 30) * m(30, 50)
temp = temp + m(41, 33) * m(33, 50)
temp = temp + m(41, 34) * m(34, 50)
temp = temp + m(41, 39) * m(39, 50)
temp = temp + m(41, 40) * m(40, 50)
m(41, 50) = ( m(41, 50) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 40) * m(40, 51)
m(41, 51) = ( m(41, 51) - temp ) / m(41, 41)
!     Upper part.
temp = m(41, 30) * m(30, 52)
temp = temp + m(41, 31) * m(31, 52)
temp = temp + m(41, 32) * m(32, 52)
temp = temp + m(41, 33) * m(33, 52)
temp = temp + m(41, 34) * m(34, 52)
temp = temp + m(41, 39) * m(39, 52)
temp = temp + m(41, 40) * m(40, 52)
m(41, 52) = ( m(41, 52) - temp ) / m(41, 41)

!     Lower part.
temp = m(42, 2) * m(2, 42)
temp = temp + m(42, 3) * m(3, 42)
temp = temp + m(42, 4) * m(4, 42)
temp = temp + m(42, 5) * m(5, 42)
temp = temp + m(42, 6) * m(6, 42)
temp = temp + m(42, 7) * m(7, 42)
temp = temp + m(42, 11) * m(11, 42)
temp = temp + m(42, 13) * m(13, 42)
temp = temp + m(42, 14) * m(14, 42)
temp = temp + m(42, 15) * m(15, 42)
temp = temp + m(42, 17) * m(17, 42)
temp = temp + m(42, 19) * m(19, 42)
temp = temp + m(42, 20) * m(20, 42)
temp = temp + m(42, 21) * m(21, 42)
temp = temp + m(42, 22) * m(22, 42)
temp = temp + m(42, 23) * m(23, 42)
temp = temp + m(42, 24) * m(24, 42)
temp = temp + m(42, 25) * m(25, 42)
temp = temp + m(42, 26) * m(26, 42)
temp = temp + m(42, 27) * m(27, 42)
temp = temp + m(42, 28) * m(28, 42)
temp = temp + m(42, 29) * m(29, 42)
temp = temp + m(42, 30) * m(30, 42)
temp = temp + m(42, 31) * m(31, 42)
temp = temp + m(42, 32) * m(32, 42)
temp = temp + m(42, 33) * m(33, 42)
temp = temp + m(42, 34) * m(34, 42)
temp = temp + m(42, 35) * m(35, 42)
temp = temp + m(42, 36) * m(36, 42)
temp = temp + m(42, 37) * m(37, 42)
temp = temp + m(42, 38) * m(38, 42)
temp = temp + m(42, 39) * m(39, 42)
temp = temp + m(42, 40) * m(40, 42)
temp = temp + m(42, 41) * m(41, 42)
m(42, 42) = m(42, 42) - temp
!     Lower part.
temp = m(43, 17) * m(17, 42)
temp = temp + m(43, 23) * m(23, 42)
temp = temp + m(43, 24) * m(24, 42)
temp = temp + m(43, 25) * m(25, 42)
temp = temp + m(43, 26) * m(26, 42)
temp = temp + m(43, 31) * m(31, 42)
temp = temp + m(43, 35) * m(35, 42)
temp = temp + m(43, 39) * m(39, 42)
temp = temp + m(43, 41) * m(41, 42)
m(43, 42) = m(43, 42) - temp
!     Lower part.
temp = m(44, 13) * m(13, 42)
temp = temp + m(44, 23) * m(23, 42)
temp = temp + m(44, 24) * m(24, 42)
temp = temp + m(44, 25) * m(25, 42)
temp = temp + m(44, 26) * m(26, 42)
temp = temp + m(44, 27) * m(27, 42)
temp = temp + m(44, 30) * m(30, 42)
temp = temp + m(44, 31) * m(31, 42)
temp = temp + m(44, 32) * m(32, 42)
temp = temp + m(44, 35) * m(35, 42)
temp = temp + m(44, 38) * m(38, 42)
temp = temp + m(44, 39) * m(39, 42)
temp = temp + m(44, 41) * m(41, 42)
m(44, 42) = m(44, 42) - temp
!     Lower part.
temp = m(45, 2) * m(2, 42)
temp = temp + m(45, 4) * m(4, 42)
temp = temp + m(45, 14) * m(14, 42)
temp = temp + m(45, 16) * m(16, 42)
temp = temp + m(45, 21) * m(21, 42)
temp = temp + m(45, 24) * m(24, 42)
temp = temp + m(45, 25) * m(25, 42)
temp = temp + m(45, 32) * m(32, 42)
temp = temp + m(45, 33) * m(33, 42)
temp = temp + m(45, 34) * m(34, 42)
temp = temp + m(45, 39) * m(39, 42)
temp = temp + m(45, 40) * m(40, 42)
temp = temp + m(45, 41) * m(41, 42)
m(45, 42) = m(45, 42) - temp
!     Lower part.
temp = m(46, 14) * m(14, 42)
temp = temp + m(46, 26) * m(26, 42)
temp = temp + m(46, 31) * m(31, 42)
temp = temp + m(46, 32) * m(32, 42)
temp = temp + m(46, 35) * m(35, 42)
temp = temp + m(46, 39) * m(39, 42)
m(46, 42) = m(46, 42) - temp
!     Lower part.
temp = m(47, 23) * m(23, 42)
temp = temp + m(47, 24) * m(24, 42)
temp = temp + m(47, 25) * m(25, 42)
temp = temp + m(47, 26) * m(26, 42)
temp = temp + m(47, 28) * m(28, 42)
temp = temp + m(47, 30) * m(30, 42)
temp = temp + m(47, 31) * m(31, 42)
temp = temp + m(47, 32) * m(32, 42)
m(47, 42) = m(47, 42) - temp
!     Lower part.
temp = m(48, 12) * m(12, 42)
temp = temp + m(48, 15) * m(15, 42)
temp = temp + m(48, 31) * m(31, 42)
temp = temp + m(48, 39) * m(39, 42)
temp = temp + m(48, 40) * m(40, 42)
temp = temp + m(48, 41) * m(41, 42)
m(48, 42) = m(48, 42) - temp
!     Lower part.
temp = m(49, 2) * m(2, 42)
temp = temp + m(49, 4) * m(4, 42)
temp = temp + m(49, 5) * m(5, 42)
temp = temp + m(49, 6) * m(6, 42)
temp = temp + m(49, 16) * m(16, 42)
temp = temp + m(49, 19) * m(19, 42)
temp = temp + m(49, 21) * m(21, 42)
temp = temp + m(49, 23) * m(23, 42)
temp = temp + m(49, 24) * m(24, 42)
temp = temp + m(49, 25) * m(25, 42)
temp = temp + m(49, 26) * m(26, 42)
temp = temp + m(49, 27) * m(27, 42)
temp = temp + m(49, 28) * m(28, 42)
temp = temp + m(49, 29) * m(29, 42)
temp = temp + m(49, 30) * m(30, 42)
temp = temp + m(49, 31) * m(31, 42)
temp = temp + m(49, 32) * m(32, 42)
temp = temp + m(49, 34) * m(34, 42)
temp = temp + m(49, 39) * m(39, 42)
temp = temp + m(49, 40) * m(40, 42)
temp = temp + m(49, 41) * m(41, 42)
m(49, 42) = m(49, 42) - temp
!     Lower part.
temp = m(50, 2) * m(2, 42)
temp = temp + m(50, 4) * m(4, 42)
temp = temp + m(50, 5) * m(5, 42)
temp = temp + m(50, 6) * m(6, 42)
temp = temp + m(50, 7) * m(7, 42)
temp = temp + m(50, 11) * m(11, 42)
temp = temp + m(50, 12) * m(12, 42)
temp = temp + m(50, 13) * m(13, 42)
temp = temp + m(50, 16) * m(16, 42)
temp = temp + m(50, 17) * m(17, 42)
temp = temp + m(50, 19) * m(19, 42)
temp = temp + m(50, 20) * m(20, 42)
temp = temp + m(50, 21) * m(21, 42)
temp = temp + m(50, 23) * m(23, 42)
temp = temp + m(50, 24) * m(24, 42)
temp = temp + m(50, 25) * m(25, 42)
temp = temp + m(50, 26) * m(26, 42)
temp = temp + m(50, 27) * m(27, 42)
temp = temp + m(50, 28) * m(28, 42)
temp = temp + m(50, 29) * m(29, 42)
temp = temp + m(50, 30) * m(30, 42)
temp = temp + m(50, 31) * m(31, 42)
temp = temp + m(50, 32) * m(32, 42)
temp = temp + m(50, 33) * m(33, 42)
temp = temp + m(50, 34) * m(34, 42)
temp = temp + m(50, 35) * m(35, 42)
temp = temp + m(50, 37) * m(37, 42)
temp = temp + m(50, 39) * m(39, 42)
temp = temp + m(50, 40) * m(40, 42)
temp = temp + m(50, 41) * m(41, 42)
m(50, 42) = m(50, 42) - temp
!     Lower part.
temp = m(51, 22) * m(22, 42)
temp = temp + m(51, 28) * m(28, 42)
temp = temp + m(51, 29) * m(29, 42)
temp = temp + m(51, 30) * m(30, 42)
temp = temp + m(51, 32) * m(32, 42)
temp = temp + m(51, 40) * m(40, 42)
temp = temp + m(51, 41) * m(41, 42)
m(51, 42) = m(51, 42) - temp
!     Lower part.
temp = m(52, 12) * m(12, 42)
temp = temp + m(52, 13) * m(13, 42)
temp = temp + m(52, 14) * m(14, 42)
temp = temp + m(52, 15) * m(15, 42)
temp = temp + m(52, 16) * m(16, 42)
temp = temp + m(52, 23) * m(23, 42)
temp = temp + m(52, 24) * m(24, 42)
temp = temp + m(52, 25) * m(25, 42)
temp = temp + m(52, 26) * m(26, 42)
temp = temp + m(52, 30) * m(30, 42)
temp = temp + m(52, 31) * m(31, 42)
temp = temp + m(52, 33) * m(33, 42)
temp = temp + m(52, 34) * m(34, 42)
temp = temp + m(52, 38) * m(38, 42)
temp = temp + m(52, 39) * m(39, 42)
temp = temp + m(52, 40) * m(40, 42)
temp = temp + m(52, 41) * m(41, 42)
m(52, 42) = m(52, 42) - temp
!     Upper part.
temp = m(42, 17) * m(17, 43)
temp = temp + m(42, 23) * m(23, 43)
temp = temp + m(42, 24) * m(24, 43)
temp = temp + m(42, 25) * m(25, 43)
temp = temp + m(42, 26) * m(26, 43)
temp = temp + m(42, 31) * m(31, 43)
temp = temp + m(42, 32) * m(32, 43)
temp = temp + m(42, 33) * m(33, 43)
temp = temp + m(42, 34) * m(34, 43)
temp = temp + m(42, 35) * m(35, 43)
temp = temp + m(42, 37) * m(37, 43)
temp = temp + m(42, 38) * m(38, 43)
temp = temp + m(42, 39) * m(39, 43)
temp = temp + m(42, 40) * m(40, 43)
temp = temp + m(42, 41) * m(41, 43)
m(42, 43) = ( m(42, 43) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 23) * m(23, 44)
temp = temp + m(42, 24) * m(24, 44)
temp = temp + m(42, 25) * m(25, 44)
temp = temp + m(42, 26) * m(26, 44)
temp = temp + m(42, 27) * m(27, 44)
temp = temp + m(42, 28) * m(28, 44)
temp = temp + m(42, 29) * m(29, 44)
temp = temp + m(42, 30) * m(30, 44)
temp = temp + m(42, 31) * m(31, 44)
temp = temp + m(42, 32) * m(32, 44)
temp = temp + m(42, 33) * m(33, 44)
temp = temp + m(42, 34) * m(34, 44)
temp = temp + m(42, 35) * m(35, 44)
temp = temp + m(42, 37) * m(37, 44)
temp = temp + m(42, 38) * m(38, 44)
temp = temp + m(42, 39) * m(39, 44)
temp = temp + m(42, 40) * m(40, 44)
temp = temp + m(42, 41) * m(41, 44)
m(42, 44) = ( m(42, 44) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 37) * m(37, 45)
temp = temp + m(42, 38) * m(38, 45)
temp = temp + m(42, 40) * m(40, 45)
temp = temp + m(42, 41) * m(41, 45)
m(42, 45) = ( m(42, 45) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 14) * m(14, 46)
temp = temp + m(42, 22) * m(22, 46)
temp = temp + m(42, 36) * m(36, 46)
temp = temp + m(42, 40) * m(40, 46)
temp = temp + m(42, 41) * m(41, 46)
m(42, 46) = ( m(42, 46) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 1) * m(1, 47)
temp = temp + m(42, 20) * m(20, 47)
temp = temp + m(42, 23) * m(23, 47)
temp = temp + m(42, 24) * m(24, 47)
temp = temp + m(42, 25) * m(25, 47)
temp = temp + m(42, 26) * m(26, 47)
temp = temp + m(42, 28) * m(28, 47)
temp = temp + m(42, 29) * m(29, 47)
temp = temp + m(42, 31) * m(31, 47)
temp = temp + m(42, 32) * m(32, 47)
temp = temp + m(42, 33) * m(33, 47)
temp = temp + m(42, 34) * m(34, 47)
temp = temp + m(42, 35) * m(35, 47)
temp = temp + m(42, 37) * m(37, 47)
temp = temp + m(42, 38) * m(38, 47)
temp = temp + m(42, 39) * m(39, 47)
temp = temp + m(42, 40) * m(40, 47)
temp = temp + m(42, 41) * m(41, 47)
m(42, 47) = ( m(42, 47) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 15) * m(15, 48)
temp = temp + m(42, 19) * m(19, 48)
temp = temp + m(42, 20) * m(20, 48)
temp = temp + m(42, 27) * m(27, 48)
temp = temp + m(42, 28) * m(28, 48)
temp = temp + m(42, 29) * m(29, 48)
temp = temp + m(42, 30) * m(30, 48)
temp = temp + m(42, 33) * m(33, 48)
temp = temp + m(42, 34) * m(34, 48)
temp = temp + m(42, 35) * m(35, 48)
temp = temp + m(42, 37) * m(37, 48)
temp = temp + m(42, 38) * m(38, 48)
temp = temp + m(42, 39) * m(39, 48)
temp = temp + m(42, 40) * m(40, 48)
temp = temp + m(42, 41) * m(41, 48)
m(42, 48) = ( m(42, 48) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 21) * m(21, 49)
temp = temp + m(42, 35) * m(35, 49)
temp = temp + m(42, 36) * m(36, 49)
temp = temp + m(42, 37) * m(37, 49)
temp = temp + m(42, 38) * m(38, 49)
temp = temp + m(42, 39) * m(39, 49)
temp = temp + m(42, 40) * m(40, 49)
temp = temp + m(42, 41) * m(41, 49)
m(42, 49) = ( m(42, 49) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 13) * m(13, 50)
temp = temp + m(42, 17) * m(17, 50)
temp = temp + m(42, 19) * m(19, 50)
temp = temp + m(42, 20) * m(20, 50)
temp = temp + m(42, 21) * m(21, 50)
temp = temp + m(42, 22) * m(22, 50)
temp = temp + m(42, 27) * m(27, 50)
temp = temp + m(42, 28) * m(28, 50)
temp = temp + m(42, 29) * m(29, 50)
temp = temp + m(42, 30) * m(30, 50)
temp = temp + m(42, 33) * m(33, 50)
temp = temp + m(42, 34) * m(34, 50)
temp = temp + m(42, 35) * m(35, 50)
temp = temp + m(42, 36) * m(36, 50)
temp = temp + m(42, 37) * m(37, 50)
temp = temp + m(42, 38) * m(38, 50)
temp = temp + m(42, 39) * m(39, 50)
temp = temp + m(42, 40) * m(40, 50)
temp = temp + m(42, 41) * m(41, 50)
m(42, 50) = ( m(42, 50) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 22) * m(22, 51)
temp = temp + m(42, 36) * m(36, 51)
temp = temp + m(42, 40) * m(40, 51)
temp = temp + m(42, 41) * m(41, 51)
m(42, 51) = ( m(42, 51) - temp ) / m(42, 42)
!     Upper part.
temp = m(42, 13) * m(13, 52)
temp = temp + m(42, 14) * m(14, 52)
temp = temp + m(42, 15) * m(15, 52)
temp = temp + m(42, 30) * m(30, 52)
temp = temp + m(42, 31) * m(31, 52)
temp = temp + m(42, 32) * m(32, 52)
temp = temp + m(42, 33) * m(33, 52)
temp = temp + m(42, 34) * m(34, 52)
temp = temp + m(42, 35) * m(35, 52)
temp = temp + m(42, 37) * m(37, 52)
temp = temp + m(42, 38) * m(38, 52)
temp = temp + m(42, 39) * m(39, 52)
temp = temp + m(42, 40) * m(40, 52)
temp = temp + m(42, 41) * m(41, 52)
m(42, 52) = ( m(42, 52) - temp ) / m(42, 42)

!     Lower part.
temp = m(43, 17) * m(17, 43)
temp = temp + m(43, 23) * m(23, 43)
temp = temp + m(43, 24) * m(24, 43)
temp = temp + m(43, 25) * m(25, 43)
temp = temp + m(43, 26) * m(26, 43)
temp = temp + m(43, 31) * m(31, 43)
temp = temp + m(43, 35) * m(35, 43)
temp = temp + m(43, 39) * m(39, 43)
temp = temp + m(43, 41) * m(41, 43)
temp = temp + m(43, 42) * m(42, 43)
m(43, 43) = m(43, 43) - temp
!     Lower part.
temp = m(44, 23) * m(23, 43)
temp = temp + m(44, 24) * m(24, 43)
temp = temp + m(44, 25) * m(25, 43)
temp = temp + m(44, 26) * m(26, 43)
temp = temp + m(44, 31) * m(31, 43)
temp = temp + m(44, 32) * m(32, 43)
temp = temp + m(44, 35) * m(35, 43)
temp = temp + m(44, 38) * m(38, 43)
temp = temp + m(44, 39) * m(39, 43)
temp = temp + m(44, 41) * m(41, 43)
temp = temp + m(44, 42) * m(42, 43)
m(44, 43) = m(44, 43) - temp
!     Lower part.
temp = m(45, 24) * m(24, 43)
temp = temp + m(45, 25) * m(25, 43)
temp = temp + m(45, 32) * m(32, 43)
temp = temp + m(45, 33) * m(33, 43)
temp = temp + m(45, 34) * m(34, 43)
temp = temp + m(45, 39) * m(39, 43)
temp = temp + m(45, 40) * m(40, 43)
temp = temp + m(45, 41) * m(41, 43)
temp = temp + m(45, 42) * m(42, 43)
m(45, 43) = m(45, 43) - temp
!     Lower part.
temp = m(46, 26) * m(26, 43)
temp = temp + m(46, 31) * m(31, 43)
temp = temp + m(46, 32) * m(32, 43)
temp = temp + m(46, 35) * m(35, 43)
temp = temp + m(46, 39) * m(39, 43)
temp = temp + m(46, 42) * m(42, 43)
m(46, 43) = m(46, 43) - temp
!     Lower part.
temp = m(47, 23) * m(23, 43)
temp = temp + m(47, 24) * m(24, 43)
temp = temp + m(47, 25) * m(25, 43)
temp = temp + m(47, 26) * m(26, 43)
temp = temp + m(47, 31) * m(31, 43)
temp = temp + m(47, 32) * m(32, 43)
temp = temp + m(47, 42) * m(42, 43)
m(47, 43) = m(47, 43) - temp
!     Lower part.
temp = m(48, 31) * m(31, 43)
temp = temp + m(48, 39) * m(39, 43)
temp = temp + m(48, 40) * m(40, 43)
temp = temp + m(48, 41) * m(41, 43)
temp = temp + m(48, 42) * m(42, 43)
m(48, 43) = m(48, 43) - temp
!     Lower part.
temp = m(49, 23) * m(23, 43)
temp = temp + m(49, 24) * m(24, 43)
temp = temp + m(49, 25) * m(25, 43)
temp = temp + m(49, 26) * m(26, 43)
temp = temp + m(49, 31) * m(31, 43)
temp = temp + m(49, 32) * m(32, 43)
temp = temp + m(49, 34) * m(34, 43)
temp = temp + m(49, 39) * m(39, 43)
temp = temp + m(49, 40) * m(40, 43)
temp = temp + m(49, 41) * m(41, 43)
temp = temp + m(49, 42) * m(42, 43)
m(49, 43) = m(49, 43) - temp
!     Lower part.
temp = m(50, 17) * m(17, 43)
temp = temp + m(50, 23) * m(23, 43)
temp = temp + m(50, 24) * m(24, 43)
temp = temp + m(50, 25) * m(25, 43)
temp = temp + m(50, 26) * m(26, 43)
temp = temp + m(50, 31) * m(31, 43)
temp = temp + m(50, 32) * m(32, 43)
temp = temp + m(50, 33) * m(33, 43)
temp = temp + m(50, 34) * m(34, 43)
temp = temp + m(50, 35) * m(35, 43)
temp = temp + m(50, 37) * m(37, 43)
temp = temp + m(50, 39) * m(39, 43)
temp = temp + m(50, 40) * m(40, 43)
temp = temp + m(50, 41) * m(41, 43)
temp = temp + m(50, 42) * m(42, 43)
m(50, 43) = m(50, 43) - temp
!     Lower part.
temp = m(51, 32) * m(32, 43)
temp = temp + m(51, 40) * m(40, 43)
temp = temp + m(51, 41) * m(41, 43)
temp = temp + m(51, 42) * m(42, 43)
m(51, 43) = m(51, 43) - temp
!     Lower part.
temp = m(52, 23) * m(23, 43)
temp = temp + m(52, 24) * m(24, 43)
temp = temp + m(52, 25) * m(25, 43)
temp = temp + m(52, 26) * m(26, 43)
temp = temp + m(52, 31) * m(31, 43)
temp = temp + m(52, 33) * m(33, 43)
temp = temp + m(52, 34) * m(34, 43)
temp = temp + m(52, 38) * m(38, 43)
temp = temp + m(52, 39) * m(39, 43)
temp = temp + m(52, 40) * m(40, 43)
temp = temp + m(52, 41) * m(41, 43)
temp = temp + m(52, 42) * m(42, 43)
m(52, 43) = m(52, 43) - temp
!     Upper part.
temp = m(43, 23) * m(23, 44)
temp = temp + m(43, 24) * m(24, 44)
temp = temp + m(43, 25) * m(25, 44)
temp = temp + m(43, 26) * m(26, 44)
temp = temp + m(43, 31) * m(31, 44)
temp = temp + m(43, 35) * m(35, 44)
temp = temp + m(43, 39) * m(39, 44)
temp = temp + m(43, 41) * m(41, 44)
temp = temp + m(43, 42) * m(42, 44)
m(43, 44) = ( m(43, 44) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 41) * m(41, 45)
temp = temp + m(43, 42) * m(42, 45)
m(43, 45) = ( m(43, 45) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 41) * m(41, 46)
temp = temp + m(43, 42) * m(42, 46)
m(43, 46) = ( m(43, 46) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 1) * m(1, 47)
temp = temp + m(43, 23) * m(23, 47)
temp = temp + m(43, 24) * m(24, 47)
temp = temp + m(43, 25) * m(25, 47)
temp = temp + m(43, 26) * m(26, 47)
temp = temp + m(43, 31) * m(31, 47)
temp = temp + m(43, 35) * m(35, 47)
temp = temp + m(43, 39) * m(39, 47)
temp = temp + m(43, 41) * m(41, 47)
temp = temp + m(43, 42) * m(42, 47)
m(43, 47) = ( m(43, 47) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 35) * m(35, 48)
temp = temp + m(43, 39) * m(39, 48)
temp = temp + m(43, 41) * m(41, 48)
temp = temp + m(43, 42) * m(42, 48)
m(43, 48) = ( m(43, 48) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 35) * m(35, 49)
temp = temp + m(43, 39) * m(39, 49)
temp = temp + m(43, 41) * m(41, 49)
temp = temp + m(43, 42) * m(42, 49)
m(43, 49) = ( m(43, 49) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 17) * m(17, 50)
temp = temp + m(43, 35) * m(35, 50)
temp = temp + m(43, 39) * m(39, 50)
temp = temp + m(43, 41) * m(41, 50)
temp = temp + m(43, 42) * m(42, 50)
m(43, 50) = ( m(43, 50) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 41) * m(41, 51)
temp = temp + m(43, 42) * m(42, 51)
m(43, 51) = ( m(43, 51) - temp ) / m(43, 43)
!     Upper part.
temp = m(43, 31) * m(31, 52)
temp = temp + m(43, 35) * m(35, 52)
temp = temp + m(43, 39) * m(39, 52)
temp = temp + m(43, 41) * m(41, 52)
temp = temp + m(43, 42) * m(42, 52)
m(43, 52) = ( m(43, 52) - temp ) / m(43, 43)

!     Lower part.
temp = m(44, 9) * m(9, 44)
temp = temp + m(44, 23) * m(23, 44)
temp = temp + m(44, 24) * m(24, 44)
temp = temp + m(44, 25) * m(25, 44)
temp = temp + m(44, 26) * m(26, 44)
temp = temp + m(44, 27) * m(27, 44)
temp = temp + m(44, 30) * m(30, 44)
temp = temp + m(44, 31) * m(31, 44)
temp = temp + m(44, 32) * m(32, 44)
temp = temp + m(44, 35) * m(35, 44)
temp = temp + m(44, 38) * m(38, 44)
temp = temp + m(44, 39) * m(39, 44)
temp = temp + m(44, 41) * m(41, 44)
temp = temp + m(44, 42) * m(42, 44)
temp = temp + m(44, 43) * m(43, 44)
m(44, 44) = m(44, 44) - temp
!     Lower part.
temp = m(45, 24) * m(24, 44)
temp = temp + m(45, 25) * m(25, 44)
temp = temp + m(45, 32) * m(32, 44)
temp = temp + m(45, 33) * m(33, 44)
temp = temp + m(45, 34) * m(34, 44)
temp = temp + m(45, 39) * m(39, 44)
temp = temp + m(45, 40) * m(40, 44)
temp = temp + m(45, 41) * m(41, 44)
temp = temp + m(45, 42) * m(42, 44)
temp = temp + m(45, 43) * m(43, 44)
m(45, 44) = m(45, 44) - temp
!     Lower part.
temp = m(46, 26) * m(26, 44)
temp = temp + m(46, 31) * m(31, 44)
temp = temp + m(46, 32) * m(32, 44)
temp = temp + m(46, 35) * m(35, 44)
temp = temp + m(46, 39) * m(39, 44)
temp = temp + m(46, 42) * m(42, 44)
temp = temp + m(46, 43) * m(43, 44)
m(46, 44) = m(46, 44) - temp
!     Lower part.
temp = m(47, 23) * m(23, 44)
temp = temp + m(47, 24) * m(24, 44)
temp = temp + m(47, 25) * m(25, 44)
temp = temp + m(47, 26) * m(26, 44)
temp = temp + m(47, 28) * m(28, 44)
temp = temp + m(47, 30) * m(30, 44)
temp = temp + m(47, 31) * m(31, 44)
temp = temp + m(47, 32) * m(32, 44)
temp = temp + m(47, 42) * m(42, 44)
temp = temp + m(47, 43) * m(43, 44)
m(47, 44) = m(47, 44) - temp
!     Lower part.
temp = m(48, 31) * m(31, 44)
temp = temp + m(48, 39) * m(39, 44)
temp = temp + m(48, 40) * m(40, 44)
temp = temp + m(48, 41) * m(41, 44)
temp = temp + m(48, 42) * m(42, 44)
temp = temp + m(48, 43) * m(43, 44)
m(48, 44) = m(48, 44) - temp
!     Lower part.
temp = m(49, 23) * m(23, 44)
temp = temp + m(49, 24) * m(24, 44)
temp = temp + m(49, 25) * m(25, 44)
temp = temp + m(49, 26) * m(26, 44)
temp = temp + m(49, 27) * m(27, 44)
temp = temp + m(49, 28) * m(28, 44)
temp = temp + m(49, 29) * m(29, 44)
temp = temp + m(49, 30) * m(30, 44)
temp = temp + m(49, 31) * m(31, 44)
temp = temp + m(49, 32) * m(32, 44)
temp = temp + m(49, 34) * m(34, 44)
temp = temp + m(49, 39) * m(39, 44)
temp = temp + m(49, 40) * m(40, 44)
temp = temp + m(49, 41) * m(41, 44)
temp = temp + m(49, 42) * m(42, 44)
temp = temp + m(49, 43) * m(43, 44)
m(49, 44) = m(49, 44) - temp
!     Lower part.
temp = m(50, 23) * m(23, 44)
temp = temp + m(50, 24) * m(24, 44)
temp = temp + m(50, 25) * m(25, 44)
temp = temp + m(50, 26) * m(26, 44)
temp = temp + m(50, 27) * m(27, 44)
temp = temp + m(50, 28) * m(28, 44)
temp = temp + m(50, 29) * m(29, 44)
temp = temp + m(50, 30) * m(30, 44)
temp = temp + m(50, 31) * m(31, 44)
temp = temp + m(50, 32) * m(32, 44)
temp = temp + m(50, 33) * m(33, 44)
temp = temp + m(50, 34) * m(34, 44)
temp = temp + m(50, 35) * m(35, 44)
temp = temp + m(50, 37) * m(37, 44)
temp = temp + m(50, 39) * m(39, 44)
temp = temp + m(50, 40) * m(40, 44)
temp = temp + m(50, 41) * m(41, 44)
temp = temp + m(50, 42) * m(42, 44)
temp = temp + m(50, 43) * m(43, 44)
m(50, 44) = m(50, 44) - temp
!     Lower part.
temp = m(51, 28) * m(28, 44)
temp = temp + m(51, 29) * m(29, 44)
temp = temp + m(51, 30) * m(30, 44)
temp = temp + m(51, 32) * m(32, 44)
temp = temp + m(51, 40) * m(40, 44)
temp = temp + m(51, 41) * m(41, 44)
temp = temp + m(51, 42) * m(42, 44)
temp = temp + m(51, 43) * m(43, 44)
m(51, 44) = m(51, 44) - temp
!     Lower part.
temp = m(52, 9) * m(9, 44)
temp = temp + m(52, 23) * m(23, 44)
temp = temp + m(52, 24) * m(24, 44)
temp = temp + m(52, 25) * m(25, 44)
temp = temp + m(52, 26) * m(26, 44)
temp = temp + m(52, 30) * m(30, 44)
temp = temp + m(52, 31) * m(31, 44)
temp = temp + m(52, 33) * m(33, 44)
temp = temp + m(52, 34) * m(34, 44)
temp = temp + m(52, 38) * m(38, 44)
temp = temp + m(52, 39) * m(39, 44)
temp = temp + m(52, 40) * m(40, 44)
temp = temp + m(52, 41) * m(41, 44)
temp = temp + m(52, 42) * m(42, 44)
temp = temp + m(52, 43) * m(43, 44)
m(52, 44) = m(52, 44) - temp
!     Upper part.
temp = m(44, 38) * m(38, 45)
temp = temp + m(44, 41) * m(41, 45)
temp = temp + m(44, 42) * m(42, 45)
temp = temp + m(44, 43) * m(43, 45)
m(44, 45) = ( m(44, 45) - temp ) / m(44, 44)
!     Upper part.
temp = m(44, 41) * m(41, 46)
temp = temp + m(44, 42) * m(42, 46)
temp = temp + m(44, 43) * m(43, 46)
m(44, 46) = ( m(44, 46) - temp ) / m(44, 44)
!     Upper part.
temp = m(44, 23) * m(23, 47)
temp = temp + m(44, 24) * m(24, 47)
temp = temp + m(44, 25) * m(25, 47)
temp = temp + m(44, 26) * m(26, 47)
temp = temp + m(44, 31) * m(31, 47)
temp = temp + m(44, 32) * m(32, 47)
temp = temp + m(44, 35) * m(35, 47)
temp = temp + m(44, 38) * m(38, 47)
temp = temp + m(44, 39) * m(39, 47)
temp = temp + m(44, 41) * m(41, 47)
temp = temp + m(44, 42) * m(42, 47)
temp = temp + m(44, 43) * m(43, 47)
m(44, 47) = ( m(44, 47) - temp ) / m(44, 44)
!     Upper part.
temp = m(44, 27) * m(27, 48)
temp = temp + m(44, 30) * m(30, 48)
temp = temp + m(44, 35) * m(35, 48)
temp = temp + m(44, 38) * m(38, 48)
temp = temp + m(44, 39) * m(39, 48)
temp = temp + m(44, 41) * m(41, 48)
temp = temp + m(44, 42) * m(42, 48)
temp = temp + m(44, 43) * m(43, 48)
m(44, 48) = ( m(44, 48) - temp ) / m(44, 44)
!     Upper part.
temp = m(44, 35) * m(35, 49)
temp = temp + m(44, 38) * m(38, 49)
temp = temp + m(44, 39) * m(39, 49)
temp = temp + m(44, 41) * m(41, 49)
temp = temp + m(44, 42) * m(42, 49)
temp = temp + m(44, 43) * m(43, 49)
m(44, 49) = ( m(44, 49) - temp ) / m(44, 44)
!     Upper part.
temp = m(44, 13) * m(13, 50)
temp = temp + m(44, 27) * m(27, 50)
temp = temp + m(44, 30) * m(30, 50)
temp = temp + m(44, 35) * m(35, 50)
temp = temp + m(44, 38) * m(38, 50)
temp = temp + m(44, 39) * m(39, 50)
temp = temp + m(44, 41) * m(41, 50)
temp = temp + m(44, 42) * m(42, 50)
temp = temp + m(44, 43) * m(43, 50)
m(44, 50) = ( m(44, 50) - temp ) / m(44, 44)
!     Upper part.
temp = m(44, 41) * m(41, 51)
temp = temp + m(44, 42) * m(42, 51)
temp = temp + m(44, 43) * m(43, 51)
m(44, 51) = ( m(44, 51) - temp ) / m(44, 44)
!     Upper part.
temp = m(44, 9) * m(9, 52)
temp = temp + m(44, 13) * m(13, 52)
temp = temp + m(44, 30) * m(30, 52)
temp = temp + m(44, 31) * m(31, 52)
temp = temp + m(44, 32) * m(32, 52)
temp = temp + m(44, 35) * m(35, 52)
temp = temp + m(44, 38) * m(38, 52)
temp = temp + m(44, 39) * m(39, 52)
temp = temp + m(44, 41) * m(41, 52)
temp = temp + m(44, 42) * m(42, 52)
temp = temp + m(44, 43) * m(43, 52)
m(44, 52) = ( m(44, 52) - temp ) / m(44, 44)

!     Lower part.
temp = m(45, 40) * m(40, 45)
temp = temp + m(45, 41) * m(41, 45)
temp = temp + m(45, 42) * m(42, 45)
temp = temp + m(45, 43) * m(43, 45)
temp = temp + m(45, 44) * m(44, 45)
m(45, 45) = m(45, 45) - temp
!     Lower part.
temp = m(46, 42) * m(42, 45)
temp = temp + m(46, 43) * m(43, 45)
temp = temp + m(46, 44) * m(44, 45)
m(46, 45) = m(46, 45) - temp
!     Lower part.
temp = m(47, 42) * m(42, 45)
temp = temp + m(47, 43) * m(43, 45)
temp = temp + m(47, 44) * m(44, 45)
m(47, 45) = m(47, 45) - temp
!     Lower part.
temp = m(48, 40) * m(40, 45)
temp = temp + m(48, 41) * m(41, 45)
temp = temp + m(48, 42) * m(42, 45)
temp = temp + m(48, 43) * m(43, 45)
temp = temp + m(48, 44) * m(44, 45)
m(48, 45) = m(48, 45) - temp
!     Lower part.
temp = m(49, 40) * m(40, 45)
temp = temp + m(49, 41) * m(41, 45)
temp = temp + m(49, 42) * m(42, 45)
temp = temp + m(49, 43) * m(43, 45)
temp = temp + m(49, 44) * m(44, 45)
m(49, 45) = m(49, 45) - temp
!     Lower part.
temp = m(50, 37) * m(37, 45)
temp = temp + m(50, 40) * m(40, 45)
temp = temp + m(50, 41) * m(41, 45)
temp = temp + m(50, 42) * m(42, 45)
temp = temp + m(50, 43) * m(43, 45)
temp = temp + m(50, 44) * m(44, 45)
m(50, 45) = m(50, 45) - temp
!     Lower part.
temp = m(51, 40) * m(40, 45)
temp = temp + m(51, 41) * m(41, 45)
temp = temp + m(51, 42) * m(42, 45)
temp = temp + m(51, 43) * m(43, 45)
temp = temp + m(51, 44) * m(44, 45)
m(51, 45) = m(51, 45) - temp
!     Lower part.
temp = m(52, 38) * m(38, 45)
temp = temp + m(52, 40) * m(40, 45)
temp = temp + m(52, 41) * m(41, 45)
temp = temp + m(52, 42) * m(42, 45)
temp = temp + m(52, 43) * m(43, 45)
temp = temp + m(52, 44) * m(44, 45)
m(52, 45) = m(52, 45) - temp
!     Upper part.
temp = m(45, 14) * m(14, 46)
temp = temp + m(45, 40) * m(40, 46)
temp = temp + m(45, 41) * m(41, 46)
temp = temp + m(45, 42) * m(42, 46)
temp = temp + m(45, 43) * m(43, 46)
temp = temp + m(45, 44) * m(44, 46)
m(45, 46) = ( m(45, 46) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 24) * m(24, 47)
temp = temp + m(45, 25) * m(25, 47)
temp = temp + m(45, 32) * m(32, 47)
temp = temp + m(45, 33) * m(33, 47)
temp = temp + m(45, 34) * m(34, 47)
temp = temp + m(45, 39) * m(39, 47)
temp = temp + m(45, 40) * m(40, 47)
temp = temp + m(45, 41) * m(41, 47)
temp = temp + m(45, 42) * m(42, 47)
temp = temp + m(45, 43) * m(43, 47)
temp = temp + m(45, 44) * m(44, 47)
m(45, 47) = ( m(45, 47) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 33) * m(33, 48)
temp = temp + m(45, 34) * m(34, 48)
temp = temp + m(45, 39) * m(39, 48)
temp = temp + m(45, 40) * m(40, 48)
temp = temp + m(45, 41) * m(41, 48)
temp = temp + m(45, 42) * m(42, 48)
temp = temp + m(45, 43) * m(43, 48)
temp = temp + m(45, 44) * m(44, 48)
m(45, 48) = ( m(45, 48) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 21) * m(21, 49)
temp = temp + m(45, 39) * m(39, 49)
temp = temp + m(45, 40) * m(40, 49)
temp = temp + m(45, 41) * m(41, 49)
temp = temp + m(45, 42) * m(42, 49)
temp = temp + m(45, 43) * m(43, 49)
temp = temp + m(45, 44) * m(44, 49)
m(45, 49) = ( m(45, 49) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 21) * m(21, 50)
temp = temp + m(45, 33) * m(33, 50)
temp = temp + m(45, 34) * m(34, 50)
temp = temp + m(45, 39) * m(39, 50)
temp = temp + m(45, 40) * m(40, 50)
temp = temp + m(45, 41) * m(41, 50)
temp = temp + m(45, 42) * m(42, 50)
temp = temp + m(45, 43) * m(43, 50)
temp = temp + m(45, 44) * m(44, 50)
m(45, 50) = ( m(45, 50) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 40) * m(40, 51)
temp = temp + m(45, 41) * m(41, 51)
temp = temp + m(45, 42) * m(42, 51)
temp = temp + m(45, 43) * m(43, 51)
temp = temp + m(45, 44) * m(44, 51)
m(45, 51) = ( m(45, 51) - temp ) / m(45, 45)
!     Upper part.
temp = m(45, 14) * m(14, 52)
temp = temp + m(45, 16) * m(16, 52)
temp = temp + m(45, 32) * m(32, 52)
temp = temp + m(45, 33) * m(33, 52)
temp = temp + m(45, 34) * m(34, 52)
temp = temp + m(45, 39) * m(39, 52)
temp = temp + m(45, 40) * m(40, 52)
temp = temp + m(45, 41) * m(41, 52)
temp = temp + m(45, 42) * m(42, 52)
temp = temp + m(45, 43) * m(43, 52)
temp = temp + m(45, 44) * m(44, 52)
m(45, 52) = ( m(45, 52) - temp ) / m(45, 45)

!     Lower part.
temp = m(46, 14) * m(14, 46)
temp = temp + m(46, 42) * m(42, 46)
temp = temp + m(46, 43) * m(43, 46)
temp = temp + m(46, 44) * m(44, 46)
temp = temp + m(46, 45) * m(45, 46)
m(46, 46) = m(46, 46) - temp
!     Lower part.
temp = m(47, 42) * m(42, 46)
temp = temp + m(47, 43) * m(43, 46)
temp = temp + m(47, 44) * m(44, 46)
temp = temp + m(47, 45) * m(45, 46)
m(47, 46) = m(47, 46) - temp
!     Lower part.
temp = m(48, 40) * m(40, 46)
temp = temp + m(48, 41) * m(41, 46)
temp = temp + m(48, 42) * m(42, 46)
temp = temp + m(48, 43) * m(43, 46)
temp = temp + m(48, 44) * m(44, 46)
temp = temp + m(48, 45) * m(45, 46)
m(48, 46) = m(48, 46) - temp
!     Lower part.
temp = m(49, 40) * m(40, 46)
temp = temp + m(49, 41) * m(41, 46)
temp = temp + m(49, 42) * m(42, 46)
temp = temp + m(49, 43) * m(43, 46)
temp = temp + m(49, 44) * m(44, 46)
temp = temp + m(49, 45) * m(45, 46)
m(49, 46) = m(49, 46) - temp
!     Lower part.
temp = m(50, 40) * m(40, 46)
temp = temp + m(50, 41) * m(41, 46)
temp = temp + m(50, 42) * m(42, 46)
temp = temp + m(50, 43) * m(43, 46)
temp = temp + m(50, 44) * m(44, 46)
temp = temp + m(50, 45) * m(45, 46)
m(50, 46) = m(50, 46) - temp
!     Lower part.
temp = m(51, 22) * m(22, 46)
temp = temp + m(51, 40) * m(40, 46)
temp = temp + m(51, 41) * m(41, 46)
temp = temp + m(51, 42) * m(42, 46)
temp = temp + m(51, 43) * m(43, 46)
temp = temp + m(51, 44) * m(44, 46)
temp = temp + m(51, 45) * m(45, 46)
m(51, 46) = m(51, 46) - temp
!     Lower part.
temp = m(52, 14) * m(14, 46)
temp = temp + m(52, 40) * m(40, 46)
temp = temp + m(52, 41) * m(41, 46)
temp = temp + m(52, 42) * m(42, 46)
temp = temp + m(52, 43) * m(43, 46)
temp = temp + m(52, 44) * m(44, 46)
temp = temp + m(52, 45) * m(45, 46)
m(52, 46) = m(52, 46) - temp
!     Upper part.
temp = m(46, 26) * m(26, 47)
temp = temp + m(46, 31) * m(31, 47)
temp = temp + m(46, 32) * m(32, 47)
temp = temp + m(46, 35) * m(35, 47)
temp = temp + m(46, 39) * m(39, 47)
temp = temp + m(46, 42) * m(42, 47)
temp = temp + m(46, 43) * m(43, 47)
temp = temp + m(46, 44) * m(44, 47)
temp = temp + m(46, 45) * m(45, 47)
m(46, 47) = ( m(46, 47) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 35) * m(35, 48)
temp = temp + m(46, 39) * m(39, 48)
temp = temp + m(46, 42) * m(42, 48)
temp = temp + m(46, 43) * m(43, 48)
temp = temp + m(46, 44) * m(44, 48)
temp = temp + m(46, 45) * m(45, 48)
m(46, 48) = ( m(46, 48) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 35) * m(35, 49)
temp = temp + m(46, 39) * m(39, 49)
temp = temp + m(46, 42) * m(42, 49)
temp = temp + m(46, 43) * m(43, 49)
temp = temp + m(46, 44) * m(44, 49)
temp = temp + m(46, 45) * m(45, 49)
m(46, 49) = ( m(46, 49) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 35) * m(35, 50)
temp = temp + m(46, 39) * m(39, 50)
temp = temp + m(46, 42) * m(42, 50)
temp = temp + m(46, 43) * m(43, 50)
temp = temp + m(46, 44) * m(44, 50)
temp = temp + m(46, 45) * m(45, 50)
m(46, 50) = ( m(46, 50) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 42) * m(42, 51)
temp = temp + m(46, 43) * m(43, 51)
temp = temp + m(46, 44) * m(44, 51)
temp = temp + m(46, 45) * m(45, 51)
m(46, 51) = ( m(46, 51) - temp ) / m(46, 46)
!     Upper part.
temp = m(46, 14) * m(14, 52)
temp = temp + m(46, 31) * m(31, 52)
temp = temp + m(46, 32) * m(32, 52)
temp = temp + m(46, 35) * m(35, 52)
temp = temp + m(46, 39) * m(39, 52)
temp = temp + m(46, 42) * m(42, 52)
temp = temp + m(46, 43) * m(43, 52)
temp = temp + m(46, 44) * m(44, 52)
temp = temp + m(46, 45) * m(45, 52)
m(46, 52) = ( m(46, 52) - temp ) / m(46, 46)

!     Lower part.
temp = m(47, 23) * m(23, 47)
temp = temp + m(47, 24) * m(24, 47)
temp = temp + m(47, 25) * m(25, 47)
temp = temp + m(47, 26) * m(26, 47)
temp = temp + m(47, 28) * m(28, 47)
temp = temp + m(47, 31) * m(31, 47)
temp = temp + m(47, 32) * m(32, 47)
temp = temp + m(47, 42) * m(42, 47)
temp = temp + m(47, 43) * m(43, 47)
temp = temp + m(47, 44) * m(44, 47)
temp = temp + m(47, 45) * m(45, 47)
temp = temp + m(47, 46) * m(46, 47)
m(47, 47) = m(47, 47) - temp
!     Lower part.
temp = m(48, 31) * m(31, 47)
temp = temp + m(48, 39) * m(39, 47)
temp = temp + m(48, 40) * m(40, 47)
temp = temp + m(48, 41) * m(41, 47)
temp = temp + m(48, 42) * m(42, 47)
temp = temp + m(48, 43) * m(43, 47)
temp = temp + m(48, 44) * m(44, 47)
temp = temp + m(48, 45) * m(45, 47)
temp = temp + m(48, 46) * m(46, 47)
m(48, 47) = m(48, 47) - temp
!     Lower part.
temp = m(49, 23) * m(23, 47)
temp = temp + m(49, 24) * m(24, 47)
temp = temp + m(49, 25) * m(25, 47)
temp = temp + m(49, 26) * m(26, 47)
temp = temp + m(49, 28) * m(28, 47)
temp = temp + m(49, 29) * m(29, 47)
temp = temp + m(49, 31) * m(31, 47)
temp = temp + m(49, 32) * m(32, 47)
temp = temp + m(49, 34) * m(34, 47)
temp = temp + m(49, 39) * m(39, 47)
temp = temp + m(49, 40) * m(40, 47)
temp = temp + m(49, 41) * m(41, 47)
temp = temp + m(49, 42) * m(42, 47)
temp = temp + m(49, 43) * m(43, 47)
temp = temp + m(49, 44) * m(44, 47)
temp = temp + m(49, 45) * m(45, 47)
temp = temp + m(49, 46) * m(46, 47)
m(49, 47) = m(49, 47) - temp
!     Lower part.
temp = m(50, 1) * m(1, 47)
temp = temp + m(50, 20) * m(20, 47)
temp = temp + m(50, 23) * m(23, 47)
temp = temp + m(50, 24) * m(24, 47)
temp = temp + m(50, 25) * m(25, 47)
temp = temp + m(50, 26) * m(26, 47)
temp = temp + m(50, 28) * m(28, 47)
temp = temp + m(50, 29) * m(29, 47)
temp = temp + m(50, 31) * m(31, 47)
temp = temp + m(50, 32) * m(32, 47)
temp = temp + m(50, 33) * m(33, 47)
temp = temp + m(50, 34) * m(34, 47)
temp = temp + m(50, 35) * m(35, 47)
temp = temp + m(50, 37) * m(37, 47)
temp = temp + m(50, 39) * m(39, 47)
temp = temp + m(50, 40) * m(40, 47)
temp = temp + m(50, 41) * m(41, 47)
temp = temp + m(50, 42) * m(42, 47)
temp = temp + m(50, 43) * m(43, 47)
temp = temp + m(50, 44) * m(44, 47)
temp = temp + m(50, 45) * m(45, 47)
temp = temp + m(50, 46) * m(46, 47)
m(50, 47) = m(50, 47) - temp
!     Lower part.
temp = m(51, 28) * m(28, 47)
temp = temp + m(51, 29) * m(29, 47)
temp = temp + m(51, 32) * m(32, 47)
temp = temp + m(51, 40) * m(40, 47)
temp = temp + m(51, 41) * m(41, 47)
temp = temp + m(51, 42) * m(42, 47)
temp = temp + m(51, 43) * m(43, 47)
temp = temp + m(51, 44) * m(44, 47)
temp = temp + m(51, 45) * m(45, 47)
temp = temp + m(51, 46) * m(46, 47)
m(51, 47) = m(51, 47) - temp
!     Lower part.
temp = m(52, 23) * m(23, 47)
temp = temp + m(52, 24) * m(24, 47)
temp = temp + m(52, 25) * m(25, 47)
temp = temp + m(52, 26) * m(26, 47)
temp = temp + m(52, 31) * m(31, 47)
temp = temp + m(52, 33) * m(33, 47)
temp = temp + m(52, 34) * m(34, 47)
temp = temp + m(52, 38) * m(38, 47)
temp = temp + m(52, 39) * m(39, 47)
temp = temp + m(52, 40) * m(40, 47)
temp = temp + m(52, 41) * m(41, 47)
temp = temp + m(52, 42) * m(42, 47)
temp = temp + m(52, 43) * m(43, 47)
temp = temp + m(52, 44) * m(44, 47)
temp = temp + m(52, 45) * m(45, 47)
temp = temp + m(52, 46) * m(46, 47)
m(52, 47) = m(52, 47) - temp
!     Upper part.
temp = m(47, 28) * m(28, 48)
temp = temp + m(47, 30) * m(30, 48)
temp = temp + m(47, 42) * m(42, 48)
temp = temp + m(47, 43) * m(43, 48)
temp = temp + m(47, 44) * m(44, 48)
temp = temp + m(47, 45) * m(45, 48)
temp = temp + m(47, 46) * m(46, 48)
m(47, 48) = ( m(47, 48) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 42) * m(42, 49)
temp = temp + m(47, 43) * m(43, 49)
temp = temp + m(47, 44) * m(44, 49)
temp = temp + m(47, 45) * m(45, 49)
temp = temp + m(47, 46) * m(46, 49)
m(47, 49) = ( m(47, 49) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 28) * m(28, 50)
temp = temp + m(47, 30) * m(30, 50)
temp = temp + m(47, 42) * m(42, 50)
temp = temp + m(47, 43) * m(43, 50)
temp = temp + m(47, 44) * m(44, 50)
temp = temp + m(47, 45) * m(45, 50)
temp = temp + m(47, 46) * m(46, 50)
m(47, 50) = ( m(47, 50) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 42) * m(42, 51)
temp = temp + m(47, 43) * m(43, 51)
temp = temp + m(47, 44) * m(44, 51)
temp = temp + m(47, 45) * m(45, 51)
temp = temp + m(47, 46) * m(46, 51)
m(47, 51) = ( m(47, 51) - temp ) / m(47, 47)
!     Upper part.
temp = m(47, 30) * m(30, 52)
temp = temp + m(47, 31) * m(31, 52)
temp = temp + m(47, 32) * m(32, 52)
temp = temp + m(47, 42) * m(42, 52)
temp = temp + m(47, 43) * m(43, 52)
temp = temp + m(47, 44) * m(44, 52)
temp = temp + m(47, 45) * m(45, 52)
temp = temp + m(47, 46) * m(46, 52)
m(47, 52) = ( m(47, 52) - temp ) / m(47, 47)

!     Lower part.
temp = m(48, 12) * m(12, 48)
temp = temp + m(48, 15) * m(15, 48)
temp = temp + m(48, 18) * m(18, 48)
temp = temp + m(48, 39) * m(39, 48)
temp = temp + m(48, 40) * m(40, 48)
temp = temp + m(48, 41) * m(41, 48)
temp = temp + m(48, 42) * m(42, 48)
temp = temp + m(48, 43) * m(43, 48)
temp = temp + m(48, 44) * m(44, 48)
temp = temp + m(48, 45) * m(45, 48)
temp = temp + m(48, 46) * m(46, 48)
temp = temp + m(48, 47) * m(47, 48)
m(48, 48) = m(48, 48) - temp
!     Lower part.
temp = m(49, 19) * m(19, 48)
temp = temp + m(49, 27) * m(27, 48)
temp = temp + m(49, 28) * m(28, 48)
temp = temp + m(49, 29) * m(29, 48)
temp = temp + m(49, 30) * m(30, 48)
temp = temp + m(49, 34) * m(34, 48)
temp = temp + m(49, 39) * m(39, 48)
temp = temp + m(49, 40) * m(40, 48)
temp = temp + m(49, 41) * m(41, 48)
temp = temp + m(49, 42) * m(42, 48)
temp = temp + m(49, 43) * m(43, 48)
temp = temp + m(49, 44) * m(44, 48)
temp = temp + m(49, 45) * m(45, 48)
temp = temp + m(49, 46) * m(46, 48)
temp = temp + m(49, 47) * m(47, 48)
m(49, 48) = m(49, 48) - temp
!     Lower part.
temp = m(50, 12) * m(12, 48)
temp = temp + m(50, 18) * m(18, 48)
temp = temp + m(50, 19) * m(19, 48)
temp = temp + m(50, 20) * m(20, 48)
temp = temp + m(50, 27) * m(27, 48)
temp = temp + m(50, 28) * m(28, 48)
temp = temp + m(50, 29) * m(29, 48)
temp = temp + m(50, 30) * m(30, 48)
temp = temp + m(50, 33) * m(33, 48)
temp = temp + m(50, 34) * m(34, 48)
temp = temp + m(50, 35) * m(35, 48)
temp = temp + m(50, 37) * m(37, 48)
temp = temp + m(50, 39) * m(39, 48)
temp = temp + m(50, 40) * m(40, 48)
temp = temp + m(50, 41) * m(41, 48)
temp = temp + m(50, 42) * m(42, 48)
temp = temp + m(50, 43) * m(43, 48)
temp = temp + m(50, 44) * m(44, 48)
temp = temp + m(50, 45) * m(45, 48)
temp = temp + m(50, 46) * m(46, 48)
temp = temp + m(50, 47) * m(47, 48)
m(50, 48) = m(50, 48) - temp
!     Lower part.
temp = m(51, 28) * m(28, 48)
temp = temp + m(51, 29) * m(29, 48)
temp = temp + m(51, 30) * m(30, 48)
temp = temp + m(51, 40) * m(40, 48)
temp = temp + m(51, 41) * m(41, 48)
temp = temp + m(51, 42) * m(42, 48)
temp = temp + m(51, 43) * m(43, 48)
temp = temp + m(51, 44) * m(44, 48)
temp = temp + m(51, 45) * m(45, 48)
temp = temp + m(51, 46) * m(46, 48)
temp = temp + m(51, 47) * m(47, 48)
m(51, 48) = m(51, 48) - temp
!     Lower part.
temp = m(52, 12) * m(12, 48)
temp = temp + m(52, 15) * m(15, 48)
temp = temp + m(52, 18) * m(18, 48)
temp = temp + m(52, 30) * m(30, 48)
temp = temp + m(52, 33) * m(33, 48)
temp = temp + m(52, 34) * m(34, 48)
temp = temp + m(52, 38) * m(38, 48)
temp = temp + m(52, 39) * m(39, 48)
temp = temp + m(52, 40) * m(40, 48)
temp = temp + m(52, 41) * m(41, 48)
temp = temp + m(52, 42) * m(42, 48)
temp = temp + m(52, 43) * m(43, 48)
temp = temp + m(52, 44) * m(44, 48)
temp = temp + m(52, 45) * m(45, 48)
temp = temp + m(52, 46) * m(46, 48)
temp = temp + m(52, 47) * m(47, 48)
m(52, 48) = m(52, 48) - temp
!     Upper part.
temp = m(48, 39) * m(39, 49)
temp = temp + m(48, 40) * m(40, 49)
temp = temp + m(48, 41) * m(41, 49)
temp = temp + m(48, 42) * m(42, 49)
temp = temp + m(48, 43) * m(43, 49)
temp = temp + m(48, 44) * m(44, 49)
temp = temp + m(48, 45) * m(45, 49)
temp = temp + m(48, 46) * m(46, 49)
temp = temp + m(48, 47) * m(47, 49)
m(48, 49) = ( m(48, 49) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 18) * m(18, 50)
temp = temp + m(48, 39) * m(39, 50)
temp = temp + m(48, 40) * m(40, 50)
temp = temp + m(48, 41) * m(41, 50)
temp = temp + m(48, 42) * m(42, 50)
temp = temp + m(48, 43) * m(43, 50)
temp = temp + m(48, 44) * m(44, 50)
temp = temp + m(48, 45) * m(45, 50)
temp = temp + m(48, 46) * m(46, 50)
temp = temp + m(48, 47) * m(47, 50)
m(48, 50) = ( m(48, 50) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 40) * m(40, 51)
temp = temp + m(48, 41) * m(41, 51)
temp = temp + m(48, 42) * m(42, 51)
temp = temp + m(48, 43) * m(43, 51)
temp = temp + m(48, 44) * m(44, 51)
temp = temp + m(48, 45) * m(45, 51)
temp = temp + m(48, 46) * m(46, 51)
temp = temp + m(48, 47) * m(47, 51)
m(48, 51) = ( m(48, 51) - temp ) / m(48, 48)
!     Upper part.
temp = m(48, 15) * m(15, 52)
temp = temp + m(48, 31) * m(31, 52)
temp = temp + m(48, 39) * m(39, 52)
temp = temp + m(48, 40) * m(40, 52)
temp = temp + m(48, 41) * m(41, 52)
temp = temp + m(48, 42) * m(42, 52)
temp = temp + m(48, 43) * m(43, 52)
temp = temp + m(48, 44) * m(44, 52)
temp = temp + m(48, 45) * m(45, 52)
temp = temp + m(48, 46) * m(46, 52)
temp = temp + m(48, 47) * m(47, 52)
m(48, 52) = ( m(48, 52) - temp ) / m(48, 48)

!     Lower part.
temp = m(49, 21) * m(21, 49)
temp = temp + m(49, 39) * m(39, 49)
temp = temp + m(49, 40) * m(40, 49)
temp = temp + m(49, 41) * m(41, 49)
temp = temp + m(49, 42) * m(42, 49)
temp = temp + m(49, 43) * m(43, 49)
temp = temp + m(49, 44) * m(44, 49)
temp = temp + m(49, 45) * m(45, 49)
temp = temp + m(49, 46) * m(46, 49)
temp = temp + m(49, 47) * m(47, 49)
temp = temp + m(49, 48) * m(48, 49)
m(49, 49) = m(49, 49) - temp
!     Lower part.
temp = m(50, 21) * m(21, 49)
temp = temp + m(50, 35) * m(35, 49)
temp = temp + m(50, 37) * m(37, 49)
temp = temp + m(50, 39) * m(39, 49)
temp = temp + m(50, 40) * m(40, 49)
temp = temp + m(50, 41) * m(41, 49)
temp = temp + m(50, 42) * m(42, 49)
temp = temp + m(50, 43) * m(43, 49)
temp = temp + m(50, 44) * m(44, 49)
temp = temp + m(50, 45) * m(45, 49)
temp = temp + m(50, 46) * m(46, 49)
temp = temp + m(50, 47) * m(47, 49)
temp = temp + m(50, 48) * m(48, 49)
m(50, 49) = m(50, 49) - temp
!     Lower part.
temp = m(51, 40) * m(40, 49)
temp = temp + m(51, 41) * m(41, 49)
temp = temp + m(51, 42) * m(42, 49)
temp = temp + m(51, 43) * m(43, 49)
temp = temp + m(51, 44) * m(44, 49)
temp = temp + m(51, 45) * m(45, 49)
temp = temp + m(51, 46) * m(46, 49)
temp = temp + m(51, 47) * m(47, 49)
temp = temp + m(51, 48) * m(48, 49)
m(51, 49) = m(51, 49) - temp
!     Lower part.
temp = m(52, 38) * m(38, 49)
temp = temp + m(52, 39) * m(39, 49)
temp = temp + m(52, 40) * m(40, 49)
temp = temp + m(52, 41) * m(41, 49)
temp = temp + m(52, 42) * m(42, 49)
temp = temp + m(52, 43) * m(43, 49)
temp = temp + m(52, 44) * m(44, 49)
temp = temp + m(52, 45) * m(45, 49)
temp = temp + m(52, 46) * m(46, 49)
temp = temp + m(52, 47) * m(47, 49)
temp = temp + m(52, 48) * m(48, 49)
m(52, 49) = m(52, 49) - temp
!     Upper part.
temp = m(49, 19) * m(19, 50)
temp = temp + m(49, 21) * m(21, 50)
temp = temp + m(49, 27) * m(27, 50)
temp = temp + m(49, 28) * m(28, 50)
temp = temp + m(49, 29) * m(29, 50)
temp = temp + m(49, 30) * m(30, 50)
temp = temp + m(49, 34) * m(34, 50)
temp = temp + m(49, 39) * m(39, 50)
temp = temp + m(49, 40) * m(40, 50)
temp = temp + m(49, 41) * m(41, 50)
temp = temp + m(49, 42) * m(42, 50)
temp = temp + m(49, 43) * m(43, 50)
temp = temp + m(49, 44) * m(44, 50)
temp = temp + m(49, 45) * m(45, 50)
temp = temp + m(49, 46) * m(46, 50)
temp = temp + m(49, 47) * m(47, 50)
temp = temp + m(49, 48) * m(48, 50)
m(49, 50) = ( m(49, 50) - temp ) / m(49, 49)
!     Upper part.
temp = m(49, 40) * m(40, 51)
temp = temp + m(49, 41) * m(41, 51)
temp = temp + m(49, 42) * m(42, 51)
temp = temp + m(49, 43) * m(43, 51)
temp = temp + m(49, 44) * m(44, 51)
temp = temp + m(49, 45) * m(45, 51)
temp = temp + m(49, 46) * m(46, 51)
temp = temp + m(49, 47) * m(47, 51)
temp = temp + m(49, 48) * m(48, 51)
m(49, 51) = ( m(49, 51) - temp ) / m(49, 49)
!     Upper part.
temp = m(49, 16) * m(16, 52)
temp = temp + m(49, 30) * m(30, 52)
temp = temp + m(49, 31) * m(31, 52)
temp = temp + m(49, 32) * m(32, 52)
temp = temp + m(49, 34) * m(34, 52)
temp = temp + m(49, 39) * m(39, 52)
temp = temp + m(49, 40) * m(40, 52)
temp = temp + m(49, 41) * m(41, 52)
temp = temp + m(49, 42) * m(42, 52)
temp = temp + m(49, 43) * m(43, 52)
temp = temp + m(49, 44) * m(44, 52)
temp = temp + m(49, 45) * m(45, 52)
temp = temp + m(49, 46) * m(46, 52)
temp = temp + m(49, 47) * m(47, 52)
temp = temp + m(49, 48) * m(48, 52)
m(49, 52) = ( m(49, 52) - temp ) / m(49, 49)

!     Lower part.
temp = m(50, 13) * m(13, 50)
temp = temp + m(50, 17) * m(17, 50)
temp = temp + m(50, 18) * m(18, 50)
temp = temp + m(50, 19) * m(19, 50)
temp = temp + m(50, 20) * m(20, 50)
temp = temp + m(50, 21) * m(21, 50)
temp = temp + m(50, 27) * m(27, 50)
temp = temp + m(50, 28) * m(28, 50)
temp = temp + m(50, 29) * m(29, 50)
temp = temp + m(50, 30) * m(30, 50)
temp = temp + m(50, 33) * m(33, 50)
temp = temp + m(50, 34) * m(34, 50)
temp = temp + m(50, 35) * m(35, 50)
temp = temp + m(50, 37) * m(37, 50)
temp = temp + m(50, 39) * m(39, 50)
temp = temp + m(50, 40) * m(40, 50)
temp = temp + m(50, 41) * m(41, 50)
temp = temp + m(50, 42) * m(42, 50)
temp = temp + m(50, 43) * m(43, 50)
temp = temp + m(50, 44) * m(44, 50)
temp = temp + m(50, 45) * m(45, 50)
temp = temp + m(50, 46) * m(46, 50)
temp = temp + m(50, 47) * m(47, 50)
temp = temp + m(50, 48) * m(48, 50)
temp = temp + m(50, 49) * m(49, 50)
m(50, 50) = m(50, 50) - temp
!     Lower part.
temp = m(51, 22) * m(22, 50)
temp = temp + m(51, 28) * m(28, 50)
temp = temp + m(51, 29) * m(29, 50)
temp = temp + m(51, 30) * m(30, 50)
temp = temp + m(51, 40) * m(40, 50)
temp = temp + m(51, 41) * m(41, 50)
temp = temp + m(51, 42) * m(42, 50)
temp = temp + m(51, 43) * m(43, 50)
temp = temp + m(51, 44) * m(44, 50)
temp = temp + m(51, 45) * m(45, 50)
temp = temp + m(51, 46) * m(46, 50)
temp = temp + m(51, 47) * m(47, 50)
temp = temp + m(51, 48) * m(48, 50)
temp = temp + m(51, 49) * m(49, 50)
m(51, 50) = m(51, 50) - temp
!     Lower part.
temp = m(52, 13) * m(13, 50)
temp = temp + m(52, 18) * m(18, 50)
temp = temp + m(52, 30) * m(30, 50)
temp = temp + m(52, 33) * m(33, 50)
temp = temp + m(52, 34) * m(34, 50)
temp = temp + m(52, 38) * m(38, 50)
temp = temp + m(52, 39) * m(39, 50)
temp = temp + m(52, 40) * m(40, 50)
temp = temp + m(52, 41) * m(41, 50)
temp = temp + m(52, 42) * m(42, 50)
temp = temp + m(52, 43) * m(43, 50)
temp = temp + m(52, 44) * m(44, 50)
temp = temp + m(52, 45) * m(45, 50)
temp = temp + m(52, 46) * m(46, 50)
temp = temp + m(52, 47) * m(47, 50)
temp = temp + m(52, 48) * m(48, 50)
temp = temp + m(52, 49) * m(49, 50)
m(52, 50) = m(52, 50) - temp
!     Upper part.
temp = m(50, 40) * m(40, 51)
temp = temp + m(50, 41) * m(41, 51)
temp = temp + m(50, 42) * m(42, 51)
temp = temp + m(50, 43) * m(43, 51)
temp = temp + m(50, 44) * m(44, 51)
temp = temp + m(50, 45) * m(45, 51)
temp = temp + m(50, 46) * m(46, 51)
temp = temp + m(50, 47) * m(47, 51)
temp = temp + m(50, 48) * m(48, 51)
temp = temp + m(50, 49) * m(49, 51)
m(50, 51) = ( m(50, 51) - temp ) / m(50, 50)
!     Upper part.
temp = m(50, 13) * m(13, 52)
temp = temp + m(50, 16) * m(16, 52)
temp = temp + m(50, 30) * m(30, 52)
temp = temp + m(50, 31) * m(31, 52)
temp = temp + m(50, 32) * m(32, 52)
temp = temp + m(50, 33) * m(33, 52)
temp = temp + m(50, 34) * m(34, 52)
temp = temp + m(50, 35) * m(35, 52)
temp = temp + m(50, 37) * m(37, 52)
temp = temp + m(50, 39) * m(39, 52)
temp = temp + m(50, 40) * m(40, 52)
temp = temp + m(50, 41) * m(41, 52)
temp = temp + m(50, 42) * m(42, 52)
temp = temp + m(50, 43) * m(43, 52)
temp = temp + m(50, 44) * m(44, 52)
temp = temp + m(50, 45) * m(45, 52)
temp = temp + m(50, 46) * m(46, 52)
temp = temp + m(50, 47) * m(47, 52)
temp = temp + m(50, 48) * m(48, 52)
temp = temp + m(50, 49) * m(49, 52)
m(50, 52) = ( m(50, 52) - temp ) / m(50, 50)

!     Lower part.
temp = m(51, 10) * m(10, 51)
temp = temp + m(51, 22) * m(22, 51)
temp = temp + m(51, 40) * m(40, 51)
temp = temp + m(51, 41) * m(41, 51)
temp = temp + m(51, 42) * m(42, 51)
temp = temp + m(51, 43) * m(43, 51)
temp = temp + m(51, 44) * m(44, 51)
temp = temp + m(51, 45) * m(45, 51)
temp = temp + m(51, 46) * m(46, 51)
temp = temp + m(51, 47) * m(47, 51)
temp = temp + m(51, 48) * m(48, 51)
temp = temp + m(51, 49) * m(49, 51)
temp = temp + m(51, 50) * m(50, 51)
m(51, 51) = m(51, 51) - temp
!     Lower part.
temp = m(52, 10) * m(10, 51)
temp = temp + m(52, 40) * m(40, 51)
temp = temp + m(52, 41) * m(41, 51)
temp = temp + m(52, 42) * m(42, 51)
temp = temp + m(52, 43) * m(43, 51)
temp = temp + m(52, 44) * m(44, 51)
temp = temp + m(52, 45) * m(45, 51)
temp = temp + m(52, 46) * m(46, 51)
temp = temp + m(52, 47) * m(47, 51)
temp = temp + m(52, 48) * m(48, 51)
temp = temp + m(52, 49) * m(49, 51)
temp = temp + m(52, 50) * m(50, 51)
m(52, 51) = m(52, 51) - temp
!     Upper part.
temp = m(51, 10) * m(10, 52)
temp = temp + m(51, 30) * m(30, 52)
temp = temp + m(51, 32) * m(32, 52)
temp = temp + m(51, 40) * m(40, 52)
temp = temp + m(51, 41) * m(41, 52)
temp = temp + m(51, 42) * m(42, 52)
temp = temp + m(51, 43) * m(43, 52)
temp = temp + m(51, 44) * m(44, 52)
temp = temp + m(51, 45) * m(45, 52)
temp = temp + m(51, 46) * m(46, 52)
temp = temp + m(51, 47) * m(47, 52)
temp = temp + m(51, 48) * m(48, 52)
temp = temp + m(51, 49) * m(49, 52)
temp = temp + m(51, 50) * m(50, 52)
m(51, 52) = ( m(51, 52) - temp ) / m(51, 51)

!     Lower part.
temp = m(52, 9) * m(9, 52)
temp = temp + m(52, 10) * m(10, 52)
temp = temp + m(52, 13) * m(13, 52)
temp = temp + m(52, 14) * m(14, 52)
temp = temp + m(52, 15) * m(15, 52)
temp = temp + m(52, 16) * m(16, 52)
temp = temp + m(52, 30) * m(30, 52)
temp = temp + m(52, 31) * m(31, 52)
temp = temp + m(52, 33) * m(33, 52)
temp = temp + m(52, 34) * m(34, 52)
temp = temp + m(52, 38) * m(38, 52)
temp = temp + m(52, 39) * m(39, 52)
temp = temp + m(52, 40) * m(40, 52)
temp = temp + m(52, 41) * m(41, 52)
temp = temp + m(52, 42) * m(42, 52)
temp = temp + m(52, 43) * m(43, 52)
temp = temp + m(52, 44) * m(44, 52)
temp = temp + m(52, 45) * m(45, 52)
temp = temp + m(52, 46) * m(46, 52)
temp = temp + m(52, 47) * m(47, 52)
temp = temp + m(52, 48) * m(48, 52)
temp = temp + m(52, 49) * m(49, 52)
temp = temp + m(52, 50) * m(50, 52)
temp = temp + m(52, 51) * m(51, 52)
m(52, 52) = m(52, 52) - temp

return
end subroutine lu_decompose_3


!===============================================================================

!> lu_solve_3
!> \brief Resolution of MY=X where M is an LU factorization computed by lu_decompose_3
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Arguments
!------------------------------------------------------------------------------
!   mode          name               role
!------------------------------------------------------------------------------
!> \param[in]     ns                 matrix row number from the chemical species
!>                                   number
!> \param[in]     m                  an LU factorization computed
!>                                   by lu_decompose_3
!> \param[in,out] x                  on entry, the right-hand side of
!>                                   the equation.
!>                                   On exit, the solution of the equation
!
!______________________________________________________________________________

subroutine lu_solve_3 (ns, m, x)

implicit none

! Arguments

integer ns
double precision m(ns,ns)
double precision x(ns)

! Local variables

double precision temp

!     Forward substitution.

x(1) = x(1) / m(1, 1)

x(2) = x(2) / m(2, 2)

x(3) = x(3) / m(3, 3)

x(4) = x(4) / m(4, 4)

x(5) = x(5) / m(5, 5)

x(6) = x(6) / m(6, 6)

x(7) = x(7) / m(7, 7)

temp = m(8, 7) * x(7)
x(8) = ( x(8) - temp ) / m(8, 8)

x(9) = x(9) / m(9, 9)

x(10) = x(10) / m(10, 10)

x(11) = x(11) / m(11, 11)

temp = m(12, 5) * x(5)
temp = temp + m(12, 6) * x(6)
x(12) = ( x(12) - temp ) / m(12, 12)

x(13) = x(13) / m(13, 13)

x(14) = x(14) / m(14, 14)

x(15) = x(15) / m(15, 15)

x(16) = x(16) / m(16, 16)

x(17) = x(17) / m(17, 17)

x(18) = x(18) / m(18, 18)

temp = m(19, 18) * x(18)
x(19) = ( x(19) - temp ) / m(19, 19)

temp = m(20, 18) * x(18)
x(20) = ( x(20) - temp ) / m(20, 20)

x(21) = x(21) / m(21, 21)

x(22) = x(22) / m(22, 22)

x(23) = x(23) / m(23, 23)

x(24) = x(24) / m(24, 24)

x(25) = x(25) / m(25, 25)

x(26) = x(26) / m(26, 26)

temp = m(27, 5) * x(5)
temp = temp + m(27, 6) * x(6)
temp = temp + m(27, 12) * x(12)
x(27) = ( x(27) - temp ) / m(27, 27)

temp = m(28, 12) * x(12)
temp = temp + m(28, 27) * x(27)
x(28) = ( x(28) - temp ) / m(28, 28)

temp = m(29, 6) * x(6)
temp = temp + m(29, 28) * x(28)
x(29) = ( x(29) - temp ) / m(29, 29)

temp = m(30, 27) * x(27)
x(30) = ( x(30) - temp ) / m(30, 30)

x(31) = x(31) / m(31, 31)

temp = m(32, 31) * x(31)
x(32) = ( x(32) - temp ) / m(32, 32)

temp = m(33, 12) * x(12)
temp = temp + m(33, 16) * x(16)
temp = temp + m(33, 26) * x(26)
temp = temp + m(33, 30) * x(30)
temp = temp + m(33, 31) * x(31)
temp = temp + m(33, 32) * x(32)
x(33) = ( x(33) - temp ) / m(33, 33)

temp = m(34, 6) * x(6)
temp = temp + m(34, 16) * x(16)
temp = temp + m(34, 24) * x(24)
temp = temp + m(34, 25) * x(25)
temp = temp + m(34, 26) * x(26)
temp = temp + m(34, 31) * x(31)
temp = temp + m(34, 32) * x(32)
temp = temp + m(34, 33) * x(33)
x(34) = ( x(34) - temp ) / m(34, 34)

temp = m(35, 2) * x(2)
temp = temp + m(35, 16) * x(16)
temp = temp + m(35, 21) * x(21)
temp = temp + m(35, 23) * x(23)
temp = temp + m(35, 24) * x(24)
temp = temp + m(35, 25) * x(25)
temp = temp + m(35, 26) * x(26)
temp = temp + m(35, 28) * x(28)
temp = temp + m(35, 30) * x(30)
temp = temp + m(35, 31) * x(31)
temp = temp + m(35, 32) * x(32)
temp = temp + m(35, 33) * x(33)
temp = temp + m(35, 34) * x(34)
x(35) = ( x(35) - temp ) / m(35, 35)

x(36) = x(36) / m(36, 36)

temp = m(37, 23) * x(23)
temp = temp + m(37, 24) * x(24)
temp = temp + m(37, 25) * x(25)
temp = temp + m(37, 26) * x(26)
temp = temp + m(37, 28) * x(28)
temp = temp + m(37, 29) * x(29)
temp = temp + m(37, 30) * x(30)
temp = temp + m(37, 31) * x(31)
temp = temp + m(37, 32) * x(32)
temp = temp + m(37, 35) * x(35)
x(37) = ( x(37) - temp ) / m(37, 37)

temp = m(38, 9) * x(9)
temp = temp + m(38, 27) * x(27)
temp = temp + m(38, 30) * x(30)
temp = temp + m(38, 32) * x(32)
temp = temp + m(38, 33) * x(33)
temp = temp + m(38, 34) * x(34)
temp = temp + m(38, 35) * x(35)
x(38) = ( x(38) - temp ) / m(38, 38)

temp = m(39, 4) * x(4)
temp = temp + m(39, 16) * x(16)
temp = temp + m(39, 24) * x(24)
temp = temp + m(39, 26) * x(26)
temp = temp + m(39, 31) * x(31)
temp = temp + m(39, 34) * x(34)
x(39) = ( x(39) - temp ) / m(39, 39)

temp = m(40, 3) * x(3)
temp = temp + m(40, 19) * x(19)
temp = temp + m(40, 22) * x(22)
temp = temp + m(40, 35) * x(35)
temp = temp + m(40, 36) * x(36)
temp = temp + m(40, 39) * x(39)
x(40) = ( x(40) - temp ) / m(40, 40)

temp = m(41, 2) * x(2)
temp = temp + m(41, 11) * x(11)
temp = temp + m(41, 18) * x(18)
temp = temp + m(41, 19) * x(19)
temp = temp + m(41, 23) * x(23)
temp = temp + m(41, 24) * x(24)
temp = temp + m(41, 25) * x(25)
temp = temp + m(41, 26) * x(26)
temp = temp + m(41, 28) * x(28)
temp = temp + m(41, 30) * x(30)
temp = temp + m(41, 31) * x(31)
temp = temp + m(41, 32) * x(32)
temp = temp + m(41, 33) * x(33)
temp = temp + m(41, 34) * x(34)
temp = temp + m(41, 39) * x(39)
temp = temp + m(41, 40) * x(40)
x(41) = ( x(41) - temp ) / m(41, 41)

temp = m(42, 1) * x(1)
temp = temp + m(42, 2) * x(2)
temp = temp + m(42, 3) * x(3)
temp = temp + m(42, 4) * x(4)
temp = temp + m(42, 5) * x(5)
temp = temp + m(42, 6) * x(6)
temp = temp + m(42, 7) * x(7)
temp = temp + m(42, 11) * x(11)
temp = temp + m(42, 13) * x(13)
temp = temp + m(42, 14) * x(14)
temp = temp + m(42, 15) * x(15)
temp = temp + m(42, 17) * x(17)
temp = temp + m(42, 19) * x(19)
temp = temp + m(42, 20) * x(20)
temp = temp + m(42, 21) * x(21)
temp = temp + m(42, 22) * x(22)
temp = temp + m(42, 23) * x(23)
temp = temp + m(42, 24) * x(24)
temp = temp + m(42, 25) * x(25)
temp = temp + m(42, 26) * x(26)
temp = temp + m(42, 27) * x(27)
temp = temp + m(42, 28) * x(28)
temp = temp + m(42, 29) * x(29)
temp = temp + m(42, 30) * x(30)
temp = temp + m(42, 31) * x(31)
temp = temp + m(42, 32) * x(32)
temp = temp + m(42, 33) * x(33)
temp = temp + m(42, 34) * x(34)
temp = temp + m(42, 35) * x(35)
temp = temp + m(42, 36) * x(36)
temp = temp + m(42, 37) * x(37)
temp = temp + m(42, 38) * x(38)
temp = temp + m(42, 39) * x(39)
temp = temp + m(42, 40) * x(40)
temp = temp + m(42, 41) * x(41)
x(42) = ( x(42) - temp ) / m(42, 42)

temp = m(43, 1) * x(1)
temp = temp + m(43, 17) * x(17)
temp = temp + m(43, 23) * x(23)
temp = temp + m(43, 24) * x(24)
temp = temp + m(43, 25) * x(25)
temp = temp + m(43, 26) * x(26)
temp = temp + m(43, 31) * x(31)
temp = temp + m(43, 35) * x(35)
temp = temp + m(43, 39) * x(39)
temp = temp + m(43, 41) * x(41)
temp = temp + m(43, 42) * x(42)
x(43) = ( x(43) - temp ) / m(43, 43)

temp = m(44, 9) * x(9)
temp = temp + m(44, 13) * x(13)
temp = temp + m(44, 23) * x(23)
temp = temp + m(44, 24) * x(24)
temp = temp + m(44, 25) * x(25)
temp = temp + m(44, 26) * x(26)
temp = temp + m(44, 27) * x(27)
temp = temp + m(44, 30) * x(30)
temp = temp + m(44, 31) * x(31)
temp = temp + m(44, 32) * x(32)
temp = temp + m(44, 35) * x(35)
temp = temp + m(44, 38) * x(38)
temp = temp + m(44, 39) * x(39)
temp = temp + m(44, 41) * x(41)
temp = temp + m(44, 42) * x(42)
temp = temp + m(44, 43) * x(43)
x(44) = ( x(44) - temp ) / m(44, 44)

temp = m(45, 2) * x(2)
temp = temp + m(45, 4) * x(4)
temp = temp + m(45, 14) * x(14)
temp = temp + m(45, 16) * x(16)
temp = temp + m(45, 21) * x(21)
temp = temp + m(45, 24) * x(24)
temp = temp + m(45, 25) * x(25)
temp = temp + m(45, 32) * x(32)
temp = temp + m(45, 33) * x(33)
temp = temp + m(45, 34) * x(34)
temp = temp + m(45, 39) * x(39)
temp = temp + m(45, 40) * x(40)
temp = temp + m(45, 41) * x(41)
temp = temp + m(45, 42) * x(42)
temp = temp + m(45, 43) * x(43)
temp = temp + m(45, 44) * x(44)
x(45) = ( x(45) - temp ) / m(45, 45)

temp = m(46, 14) * x(14)
temp = temp + m(46, 26) * x(26)
temp = temp + m(46, 31) * x(31)
temp = temp + m(46, 32) * x(32)
temp = temp + m(46, 35) * x(35)
temp = temp + m(46, 39) * x(39)
temp = temp + m(46, 42) * x(42)
temp = temp + m(46, 43) * x(43)
temp = temp + m(46, 44) * x(44)
temp = temp + m(46, 45) * x(45)
x(46) = ( x(46) - temp ) / m(46, 46)

temp = m(47, 23) * x(23)
temp = temp + m(47, 24) * x(24)
temp = temp + m(47, 25) * x(25)
temp = temp + m(47, 26) * x(26)
temp = temp + m(47, 28) * x(28)
temp = temp + m(47, 30) * x(30)
temp = temp + m(47, 31) * x(31)
temp = temp + m(47, 32) * x(32)
temp = temp + m(47, 42) * x(42)
temp = temp + m(47, 43) * x(43)
temp = temp + m(47, 44) * x(44)
temp = temp + m(47, 45) * x(45)
temp = temp + m(47, 46) * x(46)
x(47) = ( x(47) - temp ) / m(47, 47)

temp = m(48, 12) * x(12)
temp = temp + m(48, 15) * x(15)
temp = temp + m(48, 18) * x(18)
temp = temp + m(48, 31) * x(31)
temp = temp + m(48, 39) * x(39)
temp = temp + m(48, 40) * x(40)
temp = temp + m(48, 41) * x(41)
temp = temp + m(48, 42) * x(42)
temp = temp + m(48, 43) * x(43)
temp = temp + m(48, 44) * x(44)
temp = temp + m(48, 45) * x(45)
temp = temp + m(48, 46) * x(46)
temp = temp + m(48, 47) * x(47)
x(48) = ( x(48) - temp ) / m(48, 48)

temp = m(49, 2) * x(2)
temp = temp + m(49, 4) * x(4)
temp = temp + m(49, 5) * x(5)
temp = temp + m(49, 6) * x(6)
temp = temp + m(49, 16) * x(16)
temp = temp + m(49, 19) * x(19)
temp = temp + m(49, 21) * x(21)
temp = temp + m(49, 23) * x(23)
temp = temp + m(49, 24) * x(24)
temp = temp + m(49, 25) * x(25)
temp = temp + m(49, 26) * x(26)
temp = temp + m(49, 27) * x(27)
temp = temp + m(49, 28) * x(28)
temp = temp + m(49, 29) * x(29)
temp = temp + m(49, 30) * x(30)
temp = temp + m(49, 31) * x(31)
temp = temp + m(49, 32) * x(32)
temp = temp + m(49, 34) * x(34)
temp = temp + m(49, 39) * x(39)
temp = temp + m(49, 40) * x(40)
temp = temp + m(49, 41) * x(41)
temp = temp + m(49, 42) * x(42)
temp = temp + m(49, 43) * x(43)
temp = temp + m(49, 44) * x(44)
temp = temp + m(49, 45) * x(45)
temp = temp + m(49, 46) * x(46)
temp = temp + m(49, 47) * x(47)
temp = temp + m(49, 48) * x(48)
x(49) = ( x(49) - temp ) / m(49, 49)

temp = m(50, 1) * x(1)
temp = temp + m(50, 2) * x(2)
temp = temp + m(50, 4) * x(4)
temp = temp + m(50, 5) * x(5)
temp = temp + m(50, 6) * x(6)
temp = temp + m(50, 7) * x(7)
temp = temp + m(50, 11) * x(11)
temp = temp + m(50, 12) * x(12)
temp = temp + m(50, 13) * x(13)
temp = temp + m(50, 16) * x(16)
temp = temp + m(50, 17) * x(17)
temp = temp + m(50, 18) * x(18)
temp = temp + m(50, 19) * x(19)
temp = temp + m(50, 20) * x(20)
temp = temp + m(50, 21) * x(21)
temp = temp + m(50, 23) * x(23)
temp = temp + m(50, 24) * x(24)
temp = temp + m(50, 25) * x(25)
temp = temp + m(50, 26) * x(26)
temp = temp + m(50, 27) * x(27)
temp = temp + m(50, 28) * x(28)
temp = temp + m(50, 29) * x(29)
temp = temp + m(50, 30) * x(30)
temp = temp + m(50, 31) * x(31)
temp = temp + m(50, 32) * x(32)
temp = temp + m(50, 33) * x(33)
temp = temp + m(50, 34) * x(34)
temp = temp + m(50, 35) * x(35)
temp = temp + m(50, 37) * x(37)
temp = temp + m(50, 39) * x(39)
temp = temp + m(50, 40) * x(40)
temp = temp + m(50, 41) * x(41)
temp = temp + m(50, 42) * x(42)
temp = temp + m(50, 43) * x(43)
temp = temp + m(50, 44) * x(44)
temp = temp + m(50, 45) * x(45)
temp = temp + m(50, 46) * x(46)
temp = temp + m(50, 47) * x(47)
temp = temp + m(50, 48) * x(48)
temp = temp + m(50, 49) * x(49)
x(50) = ( x(50) - temp ) / m(50, 50)

temp = m(51, 10) * x(10)
temp = temp + m(51, 22) * x(22)
temp = temp + m(51, 28) * x(28)
temp = temp + m(51, 29) * x(29)
temp = temp + m(51, 30) * x(30)
temp = temp + m(51, 32) * x(32)
temp = temp + m(51, 40) * x(40)
temp = temp + m(51, 41) * x(41)
temp = temp + m(51, 42) * x(42)
temp = temp + m(51, 43) * x(43)
temp = temp + m(51, 44) * x(44)
temp = temp + m(51, 45) * x(45)
temp = temp + m(51, 46) * x(46)
temp = temp + m(51, 47) * x(47)
temp = temp + m(51, 48) * x(48)
temp = temp + m(51, 49) * x(49)
temp = temp + m(51, 50) * x(50)
x(51) = ( x(51) - temp ) / m(51, 51)

temp = m(52, 9) * x(9)
temp = temp + m(52, 10) * x(10)
temp = temp + m(52, 12) * x(12)
temp = temp + m(52, 13) * x(13)
temp = temp + m(52, 14) * x(14)
temp = temp + m(52, 15) * x(15)
temp = temp + m(52, 16) * x(16)
temp = temp + m(52, 18) * x(18)
temp = temp + m(52, 23) * x(23)
temp = temp + m(52, 24) * x(24)
temp = temp + m(52, 25) * x(25)
temp = temp + m(52, 26) * x(26)
temp = temp + m(52, 30) * x(30)
temp = temp + m(52, 31) * x(31)
temp = temp + m(52, 33) * x(33)
temp = temp + m(52, 34) * x(34)
temp = temp + m(52, 38) * x(38)
temp = temp + m(52, 39) * x(39)
temp = temp + m(52, 40) * x(40)
temp = temp + m(52, 41) * x(41)
temp = temp + m(52, 42) * x(42)
temp = temp + m(52, 43) * x(43)
temp = temp + m(52, 44) * x(44)
temp = temp + m(52, 45) * x(45)
temp = temp + m(52, 46) * x(46)
temp = temp + m(52, 47) * x(47)
temp = temp + m(52, 48) * x(48)
temp = temp + m(52, 49) * x(49)
temp = temp + m(52, 50) * x(50)
temp = temp + m(52, 51) * x(51)
x(52) = ( x(52) - temp ) / m(52, 52)


!     Backward substitution.

temp = m(51, 52) * x(52)
x(51) = x(51) - temp

temp = m(50, 51) * x(51)
temp = temp + m(50, 52) * x(52)
x(50) = x(50) - temp

temp = m(49, 50) * x(50)
temp = temp + m(49, 51) * x(51)
temp = temp + m(49, 52) * x(52)
x(49) = x(49) - temp

temp = m(48, 49) * x(49)
temp = temp + m(48, 50) * x(50)
temp = temp + m(48, 51) * x(51)
temp = temp + m(48, 52) * x(52)
x(48) = x(48) - temp

temp = m(47, 48) * x(48)
temp = temp + m(47, 49) * x(49)
temp = temp + m(47, 50) * x(50)
temp = temp + m(47, 51) * x(51)
temp = temp + m(47, 52) * x(52)
x(47) = x(47) - temp

temp = m(46, 47) * x(47)
temp = temp + m(46, 48) * x(48)
temp = temp + m(46, 49) * x(49)
temp = temp + m(46, 50) * x(50)
temp = temp + m(46, 51) * x(51)
temp = temp + m(46, 52) * x(52)
x(46) = x(46) - temp

temp = m(45, 46) * x(46)
temp = temp + m(45, 47) * x(47)
temp = temp + m(45, 48) * x(48)
temp = temp + m(45, 49) * x(49)
temp = temp + m(45, 50) * x(50)
temp = temp + m(45, 51) * x(51)
temp = temp + m(45, 52) * x(52)
x(45) = x(45) - temp

temp = m(44, 45) * x(45)
temp = temp + m(44, 46) * x(46)
temp = temp + m(44, 47) * x(47)
temp = temp + m(44, 48) * x(48)
temp = temp + m(44, 49) * x(49)
temp = temp + m(44, 50) * x(50)
temp = temp + m(44, 51) * x(51)
temp = temp + m(44, 52) * x(52)
x(44) = x(44) - temp

temp = m(43, 44) * x(44)
temp = temp + m(43, 45) * x(45)
temp = temp + m(43, 46) * x(46)
temp = temp + m(43, 47) * x(47)
temp = temp + m(43, 48) * x(48)
temp = temp + m(43, 49) * x(49)
temp = temp + m(43, 50) * x(50)
temp = temp + m(43, 51) * x(51)
temp = temp + m(43, 52) * x(52)
x(43) = x(43) - temp

temp = m(42, 43) * x(43)
temp = temp + m(42, 44) * x(44)
temp = temp + m(42, 45) * x(45)
temp = temp + m(42, 46) * x(46)
temp = temp + m(42, 47) * x(47)
temp = temp + m(42, 48) * x(48)
temp = temp + m(42, 49) * x(49)
temp = temp + m(42, 50) * x(50)
temp = temp + m(42, 51) * x(51)
temp = temp + m(42, 52) * x(52)
x(42) = x(42) - temp

temp = m(41, 42) * x(42)
temp = temp + m(41, 43) * x(43)
temp = temp + m(41, 44) * x(44)
temp = temp + m(41, 45) * x(45)
temp = temp + m(41, 46) * x(46)
temp = temp + m(41, 47) * x(47)
temp = temp + m(41, 48) * x(48)
temp = temp + m(41, 49) * x(49)
temp = temp + m(41, 50) * x(50)
temp = temp + m(41, 51) * x(51)
temp = temp + m(41, 52) * x(52)
x(41) = x(41) - temp

temp = m(40, 41) * x(41)
temp = temp + m(40, 42) * x(42)
temp = temp + m(40, 43) * x(43)
temp = temp + m(40, 44) * x(44)
temp = temp + m(40, 45) * x(45)
temp = temp + m(40, 46) * x(46)
temp = temp + m(40, 47) * x(47)
temp = temp + m(40, 48) * x(48)
temp = temp + m(40, 49) * x(49)
temp = temp + m(40, 50) * x(50)
temp = temp + m(40, 51) * x(51)
temp = temp + m(40, 52) * x(52)
x(40) = x(40) - temp

temp = m(39, 42) * x(42)
temp = temp + m(39, 43) * x(43)
temp = temp + m(39, 44) * x(44)
temp = temp + m(39, 47) * x(47)
temp = temp + m(39, 48) * x(48)
temp = temp + m(39, 49) * x(49)
temp = temp + m(39, 50) * x(50)
temp = temp + m(39, 52) * x(52)
x(39) = x(39) - temp

temp = m(38, 39) * x(39)
temp = temp + m(38, 41) * x(41)
temp = temp + m(38, 42) * x(42)
temp = temp + m(38, 43) * x(43)
temp = temp + m(38, 44) * x(44)
temp = temp + m(38, 45) * x(45)
temp = temp + m(38, 47) * x(47)
temp = temp + m(38, 48) * x(48)
temp = temp + m(38, 49) * x(49)
temp = temp + m(38, 50) * x(50)
temp = temp + m(38, 52) * x(52)
x(38) = x(38) - temp

temp = m(37, 39) * x(39)
temp = temp + m(37, 41) * x(41)
temp = temp + m(37, 42) * x(42)
temp = temp + m(37, 43) * x(43)
temp = temp + m(37, 44) * x(44)
temp = temp + m(37, 45) * x(45)
temp = temp + m(37, 47) * x(47)
temp = temp + m(37, 48) * x(48)
temp = temp + m(37, 49) * x(49)
temp = temp + m(37, 50) * x(50)
temp = temp + m(37, 52) * x(52)
x(37) = x(37) - temp

temp = m(36, 40) * x(40)
temp = temp + m(36, 42) * x(42)
temp = temp + m(36, 46) * x(46)
temp = temp + m(36, 49) * x(49)
temp = temp + m(36, 50) * x(50)
temp = temp + m(36, 51) * x(51)
x(36) = x(36) - temp

temp = m(35, 39) * x(39)
temp = temp + m(35, 42) * x(42)
temp = temp + m(35, 43) * x(43)
temp = temp + m(35, 44) * x(44)
temp = temp + m(35, 47) * x(47)
temp = temp + m(35, 48) * x(48)
temp = temp + m(35, 49) * x(49)
temp = temp + m(35, 50) * x(50)
temp = temp + m(35, 52) * x(52)
x(35) = x(35) - temp

temp = m(34, 39) * x(39)
temp = temp + m(34, 42) * x(42)
temp = temp + m(34, 43) * x(43)
temp = temp + m(34, 44) * x(44)
temp = temp + m(34, 47) * x(47)
temp = temp + m(34, 48) * x(48)
temp = temp + m(34, 50) * x(50)
temp = temp + m(34, 52) * x(52)
x(34) = x(34) - temp

temp = m(33, 34) * x(34)
temp = temp + m(33, 39) * x(39)
temp = temp + m(33, 42) * x(42)
temp = temp + m(33, 43) * x(43)
temp = temp + m(33, 44) * x(44)
temp = temp + m(33, 47) * x(47)
temp = temp + m(33, 48) * x(48)
temp = temp + m(33, 50) * x(50)
temp = temp + m(33, 52) * x(52)
x(33) = x(33) - temp

temp = m(32, 42) * x(42)
temp = temp + m(32, 43) * x(43)
temp = temp + m(32, 44) * x(44)
temp = temp + m(32, 47) * x(47)
temp = temp + m(32, 52) * x(52)
x(32) = x(32) - temp

temp = m(31, 42) * x(42)
temp = temp + m(31, 43) * x(43)
temp = temp + m(31, 44) * x(44)
temp = temp + m(31, 47) * x(47)
temp = temp + m(31, 52) * x(52)
x(31) = x(31) - temp

temp = m(30, 42) * x(42)
temp = temp + m(30, 44) * x(44)
temp = temp + m(30, 48) * x(48)
temp = temp + m(30, 50) * x(50)
temp = temp + m(30, 52) * x(52)
x(30) = x(30) - temp

temp = m(29, 30) * x(30)
temp = temp + m(29, 32) * x(32)
temp = temp + m(29, 42) * x(42)
temp = temp + m(29, 44) * x(44)
temp = temp + m(29, 47) * x(47)
temp = temp + m(29, 48) * x(48)
temp = temp + m(29, 50) * x(50)
x(29) = x(29) - temp

temp = m(28, 30) * x(30)
temp = temp + m(28, 42) * x(42)
temp = temp + m(28, 44) * x(44)
temp = temp + m(28, 47) * x(47)
temp = temp + m(28, 48) * x(48)
temp = temp + m(28, 50) * x(50)
x(28) = x(28) - temp

temp = m(27, 30) * x(30)
temp = temp + m(27, 42) * x(42)
temp = temp + m(27, 44) * x(44)
temp = temp + m(27, 48) * x(48)
temp = temp + m(27, 50) * x(50)
x(27) = x(27) - temp

temp = m(26, 42) * x(42)
temp = temp + m(26, 43) * x(43)
temp = temp + m(26, 44) * x(44)
temp = temp + m(26, 47) * x(47)
x(26) = x(26) - temp

temp = m(25, 42) * x(42)
temp = temp + m(25, 43) * x(43)
temp = temp + m(25, 44) * x(44)
temp = temp + m(25, 47) * x(47)
x(25) = x(25) - temp

temp = m(24, 42) * x(42)
temp = temp + m(24, 43) * x(43)
temp = temp + m(24, 44) * x(44)
temp = temp + m(24, 47) * x(47)
x(24) = x(24) - temp

temp = m(23, 42) * x(42)
temp = temp + m(23, 43) * x(43)
temp = temp + m(23, 44) * x(44)
temp = temp + m(23, 47) * x(47)
x(23) = x(23) - temp

temp = m(22, 42) * x(42)
temp = temp + m(22, 46) * x(46)
temp = temp + m(22, 50) * x(50)
temp = temp + m(22, 51) * x(51)
x(22) = x(22) - temp

temp = m(21, 39) * x(39)
temp = temp + m(21, 42) * x(42)
temp = temp + m(21, 49) * x(49)
temp = temp + m(21, 50) * x(50)
x(21) = x(21) - temp

temp = m(20, 23) * x(23)
temp = temp + m(20, 41) * x(41)
temp = temp + m(20, 42) * x(42)
temp = temp + m(20, 47) * x(47)
temp = temp + m(20, 48) * x(48)
temp = temp + m(20, 50) * x(50)
x(20) = x(20) - temp

temp = m(19, 40) * x(40)
temp = temp + m(19, 41) * x(41)
temp = temp + m(19, 42) * x(42)
temp = temp + m(19, 48) * x(48)
temp = temp + m(19, 50) * x(50)
x(19) = x(19) - temp

temp = m(18, 41) * x(41)
temp = temp + m(18, 48) * x(48)
temp = temp + m(18, 50) * x(50)
x(18) = x(18) - temp

temp = m(17, 42) * x(42)
temp = temp + m(17, 43) * x(43)
temp = temp + m(17, 50) * x(50)
x(17) = x(17) - temp

temp = m(16, 34) * x(34)
temp = temp + m(16, 42) * x(42)
temp = temp + m(16, 52) * x(52)
x(16) = x(16) - temp

temp = m(15, 42) * x(42)
temp = temp + m(15, 48) * x(48)
temp = temp + m(15, 52) * x(52)
x(15) = x(15) - temp

temp = m(14, 42) * x(42)
temp = temp + m(14, 46) * x(46)
temp = temp + m(14, 52) * x(52)
x(14) = x(14) - temp

temp = m(13, 42) * x(42)
temp = temp + m(13, 50) * x(50)
temp = temp + m(13, 52) * x(52)
x(13) = x(13) - temp

temp = m(12, 42) * x(42)
temp = temp + m(12, 48) * x(48)
x(12) = x(12) - temp

temp = m(11, 40) * x(40)
temp = temp + m(11, 42) * x(42)
x(11) = x(11) - temp

temp = m(10, 51) * x(51)
temp = temp + m(10, 52) * x(52)
x(10) = x(10) - temp

temp = m(9, 44) * x(44)
temp = temp + m(9, 52) * x(52)
x(9) = x(9) - temp

temp = m(8, 42) * x(42)
x(8) = x(8) - temp

temp = m(7, 42) * x(42)
x(7) = x(7) - temp

temp = m(6, 42) * x(42)
x(6) = x(6) - temp

temp = m(5, 42) * x(42)
x(5) = x(5) - temp

temp = m(4, 42) * x(42)
x(4) = x(4) - temp

temp = m(3, 42) * x(42)
x(3) = x(3) - temp

temp = m(2, 42) * x(42)
x(2) = x(2) - temp

temp = m(1, 47) * x(47)
x(1) = x(1) - temp

return
end subroutine lu_solve_3

