/*-------------------------------------------------------------------------
Compiler Generator Coco/R,
Copyright (c) 1990, 2004 Hanspeter Moessenboeck, University of Linz
extended by M. Loeberbauer & A. Woess, Univ. of Linz
ported to C++ by Csaba Balazs, University of Szeged
with improvements by Pat Terry, Rhodes University

This program is free software; you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation; either version 2, or (at your option) any 
later version.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
for more details.

You should have received a copy of the GNU General Public License along 
with this program; if not, write to the Free Software Foundation, Inc., 
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

As an exception, it is allowed to write an extension of Coco/R that is
used as a plugin in non-free software.

If not otherwise stated, any source code generated by Coco/R (other than 
Coco/R itself) does not fall under the GNU General Public License.
-------------------------------------------------------------------------*/

#include <memory.h>
#include <stdio.h>
#include "BitArray.h"

namespace Coco {

BitArray::BitArray(const int length, const bool defaultValue)
{
	Count = length;
	Data = new unsigned char[ (length+7)>>3 ];
	if (defaultValue)
		memset(Data, 0xFF, (length+7)>>3);
	else
		memset(Data, 0x00, (length+7)>>3);
}

BitArray::BitArray(const BitArray &copy) {
	Count  = copy.Count;
	Data = new unsigned char[ (copy.Count+7)>>3 ];
	memcpy(Data, copy.Data, (copy.Count+7)>>3);
}

BitArray::~BitArray()
{
	delete [] Data;
	Data = NULL;
}

int BitArray::getCount() {
	return Count;
}

bool BitArray::Get(const int index) const
{
	return (Data[(index>>3)] & (1<<(index&7))) != 0;
}

void BitArray::Set(const int index, const bool value)
{
	if (value){
		Data[(index>>3)] |= (1 << (index&7));
	} else {
		unsigned char mask = 0xFF;
		mask ^= (1 << (index&7));
		Data[(index>>3)] &= mask;
	}
}

void BitArray::SetAll(const bool value)
{
	if (value)
		memset(Data, 0xFF, (Count+7)>>3);
	else
		memset(Data, 0x00, (Count+7)>>3);
}


void BitArray::Not()
{
	for (int i=0; i<(Count+7)>>3; i++) {
		Data[i] ^= 0xFF;
	}
}

void BitArray::And(const BitArray *value)
{
	for (int i=0; (i<(Count+7)>>3) && (i<(value->Count+7)>>3); i++) {
		Data[i] = (Data[i] & value->Data[i]);
	}
}

void BitArray::Or(const BitArray *value)
{
	for (int i=0; (i<(Count+7)>>3) && (i<(value->Count+7)>>3); i++) {
		Data[i] = (Data[i] | value->Data[i]);
	}
}

void BitArray::Xor(const BitArray *value)
{
	for (int i=0; (i<(Count+7)>>3) && (i<(value->Count+7)>>3); i++) {
		Data[i] = (Data[i] ^ value->Data[i]);
	}
}

BitArray* BitArray::Clone() const
{
	BitArray *newBitArray = new BitArray(Count);
	newBitArray->Count = Count;
	memcpy(newBitArray->Data, Data, (Count+7)>>3);
	return newBitArray;
}

bool BitArray::Equal(const BitArray *right) const
{
	if (Count != right->Count) {
		return false;
	}
	for(int index = 0; index < Count; index++) {
		if (Get(index) != right->Get(index)) {
			return false;
		}
	}
	return true;
}

bool BitArray::Overlaps(const BitArray *right) const
{
	for (int index = 0; index < Count; ++index) {
		if (Get(index) && right->Get(index)) {
			return true;
		}
	}
	return false;
}

const BitArray &BitArray::operator=(const BitArray &right)
{
	if ( &right != this ) {         // avoid self assignment
		delete [] Data;              // prevents memory leak
		Count  = right.Count;
		Data = new unsigned char[ (Count+7)>>3 ];
		memcpy(Data, right.Data, (Count+7)>>3);
	}
	return *this;   // enables cascaded assignments
}

} // namespace
