//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Fit/SliderSettingsWidget.cpp
//! @brief     Implements class SliderSettingsWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Fit/SliderSettingsWidget.h"
#include "GUI/Model/Data/IntensityDataItem.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include <QLabel>
#include <QVBoxLayout>

SliderSettingsWidget::SliderSettingsWidget(QWidget* parent)
    : DataAccessWidget(parent)
    , m_radio1(nullptr)
    , m_radio2(nullptr)
    , m_radio3(nullptr)
    , m_lockzCheckBox(nullptr)
{
    // tuning selectors
    QString tooltip("Allows to tune sample parameters within +/- of given range \nwith the help of "
                    "the slider.");

    auto* label = new QLabel("Tuning:");
    label->setToolTip(tooltip);

    m_radio1 = new QRadioButton("10%");
    m_radio1->setAutoExclusive(true);
    m_radio1->setToolTip(tooltip);
    connect(m_radio1, &QRadioButton::clicked, this, &SliderSettingsWidget::rangeChanged);

    m_radio2 = new QRadioButton("100%");
    m_radio2->setChecked(true);
    m_radio2->setAutoExclusive(true);
    m_radio2->setToolTip(tooltip);
    connect(m_radio2, &QRadioButton::clicked, this, &SliderSettingsWidget::rangeChanged);

    m_radio3 = new QRadioButton("1000%");
    m_radio3->setAutoExclusive(true);
    m_radio3->setToolTip(tooltip);
    connect(m_radio3, &QRadioButton::clicked, this, &SliderSettingsWidget::rangeChanged);

    // Fix z-axis
    m_lockzCheckBox = new QCheckBox("Lock-Z");
    m_lockzCheckBox->setToolTip(
        "Preserve (min, max) range of intensity axis during parameter tuning.");
    connect(m_lockzCheckBox, &QCheckBox::stateChanged, this, &SliderSettingsWidget::onLockZChanged);

    auto* hbox = new QHBoxLayout;

    hbox->addWidget(label);
    hbox->addWidget(m_radio1);
    hbox->addWidget(m_radio2);
    hbox->addWidget(m_radio3);
    hbox->addStretch(1);
    hbox->addWidget(m_lockzCheckBox);

    setLayout(hbox);
}

void SliderSettingsWidget::setJobOrRealItem(QObject* job_item)
{
    DataAccessWidget::setJobOrRealItem(job_item);

    initSlider();
    initZlock();
}

void SliderSettingsWidget::rangeChanged()
{
    if (!jobItem())
        return;

    if (m_radio1->isChecked())
        jobItem()->setSliderRange(10);
    else if (m_radio2->isChecked())
        jobItem()->setSliderRange(100);
    else if (m_radio3->isChecked())
        jobItem()->setSliderRange(1000);

    emit sliderRangeFactorChanged(jobItem()->sliderRange());
    gProjectDocument.value()->setModified();
}

void SliderSettingsWidget::onLockZChanged(int state)
{
    if (state == Qt::Unchecked)
        emit lockzChanged(false);
    else if (state == Qt::Checked)
        emit lockzChanged(true);
    gProjectDocument.value()->setModified();
}

void SliderSettingsWidget::initSlider()
{
    if (!jobItem())
        return;

    if (jobItem()->sliderRange() == 10)
        m_radio1->setChecked(true);
    else if (jobItem()->sliderRange() == 1000)
        m_radio3->setChecked(true);
    else {
        if (jobItem()->sliderRange() != 100)
            jobItem()->setSliderRange(100);
        m_radio2->setChecked(true);
    }
    emit sliderRangeFactorChanged(jobItem()->sliderRange());
}

void SliderSettingsWidget::initZlock()
{
    if (!jobItem())
        return;

    if (auto* intensityItem = jobItem()->intensityDataItem())
        m_lockzCheckBox->setChecked(intensityItem->isZaxisLocked());
}
