; Ethereum Semaphore Library
;
; Copyright (C) 2021 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ZKSEMAPHORE")

(include-book "baby-jubjub-prime")

(include-book "kestrel/prime-fields/prime-fields" :dir :system)
(include-book "kestrel/std/util/defmacro-plus" :dir :system)
(include-book "xdoc/defxdoc-plus" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ prime-field-abbreviations
  :parents (semaphore-specification)
  :short "Abbreviations for the prime field functions for Semaphore."
  :long
  (xdoc::topstring
   (xdoc::p
    "The Semaphore is defined on BabyJubjub,
     which means that the prime field of interest (e.g. in the R1CS)
     is the one defined by @(tsee baby-jubjub-prime).")
   (xdoc::p
    "The functions in the prime fields library
     are parameterized over the prime that defines the prime field.")
   (xdoc::p
    "Here we define macros that abbreviate
     calls of the prime field library functions with the BabyJubjub prime,
     where the prime is generated by the macro.
     So the macro only takes the other arguments,
     resulting in more readable terms.
     We also introduce macros for @($n$)-ary versions
     of the addition and multiplication operations,
     again for greater readability.")
   (xdoc::p
    "The increased readability applies to the ACL2 files where they are used.
     Proof output will show the expanded calls with the prime.
     We may consider defining optional output modifications
     (using ACL2's facilities for that) that print the calls
     as calls of these macro abbreviations."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfep (x)
  :short "Abbreviation to check if something is a BabyJubjub field element."
  `(pfield::fep ,x (baby-jubjub-prime)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfadd (x y)
  :short "Abbreviation for binary addition in the BabyJubjub prime field."
  `(pfield::add ,x ,y (baby-jubjub-prime)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfaddall (&rest rst)
  :short "Abbreviation for @($n$)-ary addition in the BabyJubjub prime field."
  (if rst
      (if (cdr rst)
          (xxxjoin 'pfadd rst)
        (cons 'pfadd
              (cons 0 (cons (car rst) nil))))
    0))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfneg (x)
  :short "Abbreviation for negation in the BabyJubjub prime field."
  `(pfield::neg ,x (baby-jubjub-prime)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfsub (x y)
  :short "Abbreviation for subtraction in the BabyJubjub prime field."
  `(pfield::sub ,x ,y (baby-jubjub-prime)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfmul (x y)
  :short "Abbreviation for binary multiplication in the BabyJubjub prime field."
  `(pfield::mul ,x ,y (baby-jubjub-prime)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfmulall (&rest rst)
  :short
  "Abbreviation for @($n$)-ary multiplication in the BabyJubjub prime field."
  (if rst
      (if (cdr rst)
          (xxxjoin 'pfmul rst)
        (cons 'pfmul
              (cons 1 (cons (car rst) nil))))
    1))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfinv (x)
  :short "Abbreviation for inverse in the BabyJubjub prime field."
  `(pfield::inv ,x (baby-jubjub-prime)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfdiv (x y)
  :short "Abbreviation for division in the BabyJubjub prime field."
  `(pfield::div ,x ,y (baby-jubjub-prime)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro+ pfminus1 ()
  :short "Abbreviation for -1 in the BabyJubjub prime field."
  '(pfield::minus1 (baby-jubjub-prime)))
