/* Copyright (C) 2016 Canonical Ltd.
 * Author: Luke Yelavich <luke.yelavich@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* Loads, sets, and resets profile settings, and provides access to information
 * stored in the profile manifest.
 */

namespace A11yProfileManager
{

/**
 * Lists the available accessibility profiles. Only valid profiles are
 * added to the list.
 *
 * @return a #GSList of {@link Profile} objects.
 *
 * @since 0.1
 */
public SList<Profile>?
list_profiles()
{
	SList<Profile> profiles = new SList<Profile>();
	Dir profile_dir = null;

	if (Posix.access(Config.PROFILEDIR, Posix.R_OK | Posix.X_OK) < 0)
	{
		warning(_("You do not have permission to examine the contents of the profile directory %s."), Config.PROFILEDIR);
		return null;
	}

	try
	{
		profile_dir = Dir.open(Config.PROFILEDIR, 0);
	}
	catch (Error e)
	{
		warning(e.message);
		return null;
	}

	string name;
	while ((name = profile_dir.read_name()) != null)
	{
		string profile_path = Config.PROFILEDIR + "/" + name;
		if (Posix.access(profile_path, Posix.R_OK | Posix.X_OK) < 0)
		{
			warning(_("You do not have permission to access %s."), profile_path);
			continue;
		}

		if (FileUtils.test(profile_path, FileTest.IS_DIR))
		{
			Profile profile = new Profile(name);

			try
			{
				profile.validate();
			}
			catch (Error e)
			{
				warning(e.message);
				continue;
			}

			if (profile.valid)
			{
				profiles.append(profile);
			}
		}
	}

	return profiles;
}

/**
 * Get the profile object with the given directory name, as per the output
 * of #list_profiles.
 * 
 * @param dir_name The directory name relative to the profiles directory
 *
 * @return A validated profile object, or %NULL if no profile found.
 *
 * @since 0.1
 */
public Profile?
get_profile_by_dir_name(string dir_name)
{
	SList<Profile> profile_list = list_profiles();

	foreach (var profile in profile_list)
	{
		if (profile.valid && profile.dir_name == dir_name)
		{
			return profile;
		}
	}
	return null;
}

/**
 * Get the profile object with the given name, as per the profile's manifest
 * file.
 * 
 * @param name The name of the profile as specified in its manifest file
 *
 * @return A validated profile object, or %NULL if no profile found.
 *
 * @since 0.1
 */
public Profile?
get_profile_by_name(string name)
{
	SList<Profile> profile_list = list_profiles();

	foreach (var profile in profile_list)
	{
		if (profile.valid && profile.name == name)
		{
			return profile;
		}
	}
	return null;
}

/**
 * Sets the active accessibility profile.
 *
 * @param profile_obj A validated #Profile object.
 *
 * @since 0.1
 */
public void
set_active_profile(Profile profile_obj) throws Error
{
	if (!profile_obj.valid)
	{
		return;
	}

	var profile_settings = new Settings("com.canonical.a11y-profile-manager");
	string active_profile_name = profile_settings.get_string("active-profile");

	if (profile_obj.dir_name == active_profile_name)
	{
		return;
	}

	if (active_profile_name != null && active_profile_name != "")
	{
		Profile active_profile = new Profile(active_profile_name);

		try
		{
			active_profile.validate();
		}
		catch (Error e)
		{
			warning(e.message);
		}

		if (active_profile.valid)
		{
			try
			{
				active_profile.reset_settings();
			}
			catch (Error e)
			{
				warning(e.message);
			}
		}
	}

	profile_settings.reset("active-profile");
	Settings.sync();

	try
	{
		profile_obj.set_settings();
	}
	catch (Error e)
	{
		throw e;
	}

	profile_settings.set_string("active-profile", profile_obj.dir_name);
	Settings.sync();

	return;
}

} // namespace A11yProfileManager
