#include "mfifo.h"
#include <linux/string.h>

static void mfifo_wrap_bin (mfifo_t *fifo, int *ending) {
	*ending &= fifo->mask;
}

static int mfifo_space_bin (mfifo_t *fifo) {
	return CIRC_SPACE (fifo->head, fifo->tail, fifo->size);
}

static void mfifo_wrap_gen (mfifo_t *fifo, int *ending) {
	if (*ending >= fifo->size)
		*ending %= fifo->size;
	else if (*ending < 0) 
		*ending = (*ending % fifo->size) + fifo->size;
}

static int mfifo_space_gen (mfifo_t *fifo) {
	int head = fifo->head + 1;
	int tail = (fifo->tail >= head) ? fifo->tail 
					      : fifo->tail + fifo->size;
	
	return tail - head;
}

/* set to the head */
static void mfifo_setu8 (mfifo_t *fifo, void *e) {
	((u8 *) fifo->buf) [fifo->head] = * (u8 *) e;
}

static void mfifo_setu16 (mfifo_t *fifo, void *e) {
	((u16 *) fifo->buf) [fifo->head] = * (u16 *) e;
}

static void mfifo_setu32 (mfifo_t *fifo, void *e) {
	((u32 *) fifo->buf) [fifo->head] = * (u32 *) e;
}

static void mfifo_setgen (mfifo_t *fifo, void *e) {
	memcpy (fifo->head * fifo->esize + (char *) fifo->buf, e, fifo->esize);
}

/* get from tail */
static void mfifo_getu8 (mfifo_t *fifo, void *e) {
	* (u8 *) e = ((u8 *) fifo->buf) [fifo->tail];
}

static void mfifo_getu16 (mfifo_t *fifo, void *e) {
	 * (u16 *) e = ((u16 *) fifo->buf) [fifo->tail];
}

static void mfifo_getu32 (mfifo_t *fifo, void *e) {
	* (u32 *) e = ((u32 *) fifo->buf) [fifo->tail];
}

static void mfifo_getgen (mfifo_t *fifo, void *e) {
	memcpy (e, fifo->tail * fifo->esize + (char *) fifo->buf, fifo->esize);
}

void mfifo_init (mfifo_t *fifo, void *_buf, int _size, int _esize) {
	if (_size <= 0 || ! _buf) {
		fifo->size = 1; fifo->mask = 0; 
		set_bit (MFIFO_BINARY, &fifo->flags);
		clear_bit (MFIFO_OK, &fifo->flags); 
		return;
	}

	fifo->flags = 0;

	set_bit (MFIFO_OK, &fifo->flags); 

	fifo->buf = _buf;
	fifo->size = _size;

	if ((1 << (ffs (_size) - 1)) == _size) {
		set_bit (MFIFO_BINARY, &fifo->flags); 
		fifo->mask = _size - 1;

		fifo->ops.wrap = mfifo_wrap_bin;
		fifo->ops.space = mfifo_space_bin;
	}
	else {
		fifo->ops.wrap = mfifo_wrap_gen;
		fifo->ops.space = mfifo_space_gen;
	}

	fifo->esize = _esize;

	switch (_esize) {
	case 1:
		set_bit (MFIFO_U8, &fifo->flags);
		fifo->ops.set = mfifo_setu8;
		fifo->ops.get = mfifo_getu8;
		break;

	case 2:
		set_bit (MFIFO_U16, &fifo->flags);
		fifo->ops.set = mfifo_setu16;
		fifo->ops.get = mfifo_getu16;
		break;

	case 4:
		set_bit (MFIFO_U32, &fifo->flags);
		fifo->ops.set = mfifo_setu32;
		fifo->ops.get = mfifo_getu32;
		break;

	default:
		fifo->ops.set = mfifo_setgen;
		fifo->ops.get = mfifo_getgen;
		break;
	}

	mfifo_flush (fifo);
}

