(* 
 * IO - Abstract input/output
 * Copyright (C) 2003 Nicolas Cannasse
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version,
 * with the special exception on linking described in file LICENSE.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(** High-order abstract I/O.

	IO module simply deals with abstract inputs/outputs. It provides a
	set of methods for working with these IO as well as several
	constructors that enable to write to an underlying channel, buffer,
	or enum.
*)

type input
(** The abstract input type. *)

exception No_more_input
(** This exception is raised when reading on an input with the [read] or
  [nread] functions while there is no available token to read. *)

exception Input_closed
(** This exception is raised when reading on a closed input. *)

(** {6 Standard API} *)
val nread : input -> int -> Istring.t
(** [nread i n] reads a string of size up to [n] from an input.
  The function will raise [No_more_input] if no input is available.
  It will raise [Invalid_argument] if [n] < 0. *)

val close_in : input -> unit
(** Close the input. It can no longer be read from. *)

(** {6 Creation of IO Inputs/Outputs} *)
val create_in : (unit -> char) -> (Istring.t -> int -> int -> int) -> (unit -> unit) -> input

(** Fully create an input by giving all the needed functions. *)

(** {6 Utilities} *)
val read_all : input -> Istring.t
(** read all the contents of the input until [No_more_input] is raised. *)

(** {6 Binary files API}

	Here is some API useful for working with binary files, in particular
	binary files generated by C applications. By default, encoding of
	multibyte integers is low-endian. The BigEndian module provide multibyte
	operations with other encoding.
*)

exception Overflow of string
(** Exception raised when a read or write operation cannot be completed. *)

val read_byte : input -> int
(** Read an unsigned 8-bit integer. *)

val read_ui16 : input -> int
(** Read an unsigned 16-bit word. *)

