/*
 * Copyright(C) 2010-2012 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

[DBus (name = "com.canonical.Friends.Dispatcher")]
private interface DispatcherInterface : GLib.Object {
        public abstract void Refresh () throws GLib.IOError;

        public abstract void ClearIndicators () throws GLib.IOError;

        public abstract async void Do
        (string action, string account_id, string message_id, out string result)
        throws GLib.IOError;

        public abstract async string SendMessage
        (string message, out string result_) throws GLib.IOError;

        public abstract async void SendReply
        (string account_id, string message_id, string message,
         out string result) throws GLib.IOError;

        public abstract async void Upload
        (string account_id, string uri, string description,
         out string result_url) throws GLib.IOError;

        public abstract string GetFeatures (string protocol) throws GLib.IOError;

        public abstract void Quit () throws GLib.IOError;

        public abstract string URLShorten (string url) throws GLib.IOError;
}

namespace Friends
{
    /**
     * An interface to the Friends Dispatcher.
     */
    public class Dispatcher : Object
    {
        private const string service_name  = "com.canonical.Friends.Dispatcher";
        private const string service_path  = "/com/canonical/friends/Dispatcher";

        private DispatcherInterface service;

        /**
         * Establish a connection to the local Friends Dispatcher daemon.
         * @return a new Dispatcher
         * @throws GLib.IOError if failed to connect to the
         * Friends Dispatcher daemon.
         */
        public Dispatcher () throws IOError
        {
            try {
                service = Bus.get_proxy_sync(
                    BusType.SESSION, service_name, service_path);
            } catch (GLib.IOError e) {
                warning ("Unable to get Friends dispatcher: " + e.message);
                throw (e);
            }
        }


        /**
         * Send some arbitrary command to the friends-service
         * dispatcher over DBus. This API was chosen in order to avoid
         * a large degree of code duplication between this file and
         * the python code implementing the dispatcher interface.
         *
         * @param action Can be `like`, `search`, `delete`, etc.
         * @param account_id Must be the {@link Ag.Account.id} to act upon.
         * @param argument Will often be a message_id, but can also be
         * a search query. Depends upon the value of `action`
         * parameter and is rather arbitrary.
         *
         * @since 0.1
         * @throws GLib.IOError if action failed.
         */
        private string do_sync (
            string action,
            uint? account_id,
            string argument
            ) throws IOError
        {
            var loop = new MainLoop ();
            string result = "";
            string account_id_str = "";
            if (account_id != null)
                account_id_str = account_id.to_string ();
            service.Do.begin (
                action, account_id_str, argument, (obj, res) => {
                    try {
                        service.Do.end (res, out result);
                    } catch (GLib.IOError e) {
                        warning (e.message);
                        throw (e);
                    }
                    loop.quit ();
                }
            );
            loop.run ();
            return result;
        }

        /**
         * Send some arbitrary command to the friends-service
         * dispatcher over DBus, asynchronously. This API was chosen
         * in order to avoid a large degree of code duplication
         * between this file and the python code implementing the
         * dispatcher interface.
         *
         * @param action Can be `like`, `search`, `delete`, etc.
         * @param account_id Must be the {@link Ag.Account.id} to act upon.
         * @param argument Will often be a message_id, but can also be
         * a search query. Depends upon the value of `action`
         * parameter and is rather arbitrary.
         *
         * @since 0.1
         * @throws GLib.IOError if action failed.
         */
        private async void do_async (
            string action,
            uint? account_id,
            string argument,
            out string result
            ) throws IOError
        {
            string account_id_str = "";
            if (account_id != null)
                account_id_str = account_id.to_string ();
            yield service.Do (action, account_id_str, argument, out result);
        }

        /**
         * Perform a refresh of all enabled accounts.
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Refresh fails.
         */
        public void refresh () throws IOError
        {
            service.Refresh ();
        }

        /**
         * Upload an image to the specified account_id.
         *
         * Currently this is only supported on Facebook, but we plan to
         * support other protocols soon.
         *
         * @param account_id The {@link Ag.Account.id} to post from.
         * @param uri Local or remote URI of a file to upload.
         * @param description Description or caption to include for the
         * uploaded photo.
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Upload fails.
         */
        public string upload (
            uint account_id,
            string uri,
            string description
            ) throws IOError
        {
            var loop = new MainLoop ();
            string result_url = "";
            service.Upload.begin (
                account_id.to_string (), uri, description, (obj, res) => {
                    try {
                        service.Upload.end (res, out result_url);
                    } catch (IOError e) {
                        warning (e.message);
                        throw (e);
                    }
                    loop.quit ();
                }
            );
            loop.run ();
            return result_url;
        }

        /**
         * Upload an image to the specified account_id, asynchronously.
         *
         * @param account_id The {@link Ag.Account.id} to post from.
         * @param uri Local or remote URI of a file to upload.
         * @param description Description or caption to include for the
         * uploaded photo.
         * @param result_url The destintion URL of the successfully
         * uploaded file.
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Upload fails.
         */
        public async void upload_async (
            uint account_id,
            string uri,
            string description,
            out string result_url
            ) throws IOError
        {
            yield service.Upload (
                account_id.to_string (), uri, description, out result_url);
        }


        /**
         * Reply to an existing message.
         *
         * Note that twitter will require you to @mention the name of
         * the person you are replying to in order for it to appear
         * correctly as a reply. Otherwise, the message will post
         * successfully but it will simply appear as a normal message,
         * without any conversation threading.
         *
         * @param account_id The {@link Ag.Account.id} to post from.
         * @param message_id The ID of the message that you are replying to.
         * @param message The text of the reply that you want to send.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.SendReply fails.
         */
        public string send_reply (
            uint account_id,
            string message_id,
            string message
            ) throws IOError
        {
            var loop = new MainLoop ();
            string result = "";
            service.SendReply.begin (
                account_id.to_string (), message_id, message,
                (obj, res) => {
                    try {
                        service.SendReply.end (res, out result);
                    } catch (GLib.IOError e) {
                        warning (e.message);
                        throw (e);
                    }
                    loop.quit ();
                }
            );
            loop.run ();
            return result;
        }

        /**
         * Reply to an existing message, asynchronously.
         *
         * Note that twitter will require you to @mention the name of
         * the person you are replying to in order for it to appear
         * correctly as a reply. Otherwise, the message will post
         * successfully but it will simply appear as a normal message,
         * without any conversation threading.
         *
         * @param account_id The {@link Ag.Account.id} to post from.
         * @param message_id The ID of the message that you are replying to.
         * @param message The text of the reply that you want to send.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.SendReply fails.
         */
        public async void send_reply_async (
            uint account_id,
            string message_id,
            string message,
            out string result
            ) throws IOError
        {
            yield service.SendReply (
                account_id.to_string (), message_id, message, out result);
        }

        /**
         * Write a public message to the specified account.
         *
         * @param account_id The {@link Ag.Account.id} to post from.
         * @param message The text of the message to post publicly.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.SendMessage fails.
         */
        public string send_message (
            uint? account_id,
            string message
            ) throws IOError
        {
            if (account_id != null)
                return do_sync ("send", account_id, message);
            else
            {
                var loop = new MainLoop ();
                string result = "";
                service.SendMessage.begin (message, (obj, res) => {
                        try {
                            service.SendMessage.end (res, out result);
                        } catch (IOError e) {
                            warning (e.message);
                            throw (e);
                        }
                        loop.quit ();
                    }
                );
                loop.run ();
                return result;
            }
        }

        /**
         * Write a public message to the specified account, asynchronously.
         *
         * @param account_id The {@link Ag.Account.id} to post from.
         * @param message The text of the message to post publicly.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.SendMessage fails.
         */
        public async void send_message_async (
            uint? account_id,
            string message,
            out string result
            ) throws IOError
        {
            if (account_id != null)
                yield do_async ("send", account_id, message, out result);
            else
                yield service.SendMessage (message, out result);
        }

        /**
         * Retweet an existing tweet.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you want to reshare.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Do fails.
         */
        public string retweet (
            uint account_id,
            string message_id
            ) throws IOError
        {
            return do_sync ("retweet", account_id, message_id);
        }

        /**
         * Retweet an existing tweet, asynchronously.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you want to reshare.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void retweet_async (
            uint account_id,
            string message_id,
            out string result
            ) throws IOError
        {
            yield do_async (
                "retweet", account_id, message_id, out result);
        }

        /**
         * Fetch the messages from the user's "home" screen. On
         * Facebook and Twitter this maps to the main feed you see on
         * the front page; public posts by people you follow but not
         * necessarily addressed directly to you.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Do fails.
         */
        public string home (uint? account_id) throws IOError
        {
            return do_sync ("home", account_id, "");
        }

        /**
         * Fetch the messages from the user's "home" screen,
         * asynchronously. On Facebook and Twitter this maps to the
         * main feed you see on the front page; public posts by people
         * you follow but not necessarily addressed directly to you.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void home_async (
            uint? account_id,
            out string result
            ) throws IOError
        {
            yield do_async ("home", account_id, "", out result);
        }

        /**
         * Fetch the messages from the user's "wall" screen. On
         * Facebook this fetches the messages directly posted on your
         * wall, on Twitter this maps to the public messages that have
         * directly @mentioned you. Not all protocols are expected to
         * support this feature.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Do fails.
         */
        public string wall (uint? account_id) throws IOError
        {
            return do_sync ("wall", account_id, "");
        }

        /**
         * Fetch the messages from the user's "wall" screen,
         * asynchronously. On Facebook this fetches the messages
         * directly posted on your wall, on Twitter this maps to the
         * public messages that have directly @mentioned you. Not all
         * protocols are expected to support this feature.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void wall_async (
            uint? account_id,
            out string result
            ) throws IOError
        {
            yield do_async ("wall", account_id, "", out result);
        }

        /**
         * Search public messages on a specific protocol. If
         * account_id is Null, then all accounts are searched.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param query The text you want to search for.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Do fails.
         */

        public string search (uint? account_id, string query) throws IOError
        {
            return do_sync ("search", account_id, query);
        }

        /**
         * Search public messages on a specific protocol,
         * asynchronously. If account_id is Null, then all accounts
         * are searched.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param query The text you want to search for.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void search_async (
            uint? account_id,
            string query,
            out string result
            ) throws IOError
        {
            yield do_async ("search", account_id, query, out result);
        }

        /**
         * Assert your approval of a specific message on a specific
         * protocol.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you appove of.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public string like (uint account_id, string message_id) throws IOError
        {
            return do_sync ("like", account_id, message_id);
        }

        /**
         * Assert your approval of a specific message on a specific
         * protocol, asynchronously.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you appove of.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void like_async (
            uint account_id,
            string message_id,
            out string result
            ) throws IOError
        {
            yield do_async ("like", account_id, message_id, out result);
        }

        /**
         * Revoke your approval of a specific message on a specific
         * protocol.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you appove of.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public string unlike (uint account_id, string message_id) throws IOError
        {
            return do_sync ("unlike", account_id, message_id);
        }

        /**
         * Revoke your approval of a specific message on a specific
         * protocol, asynchronously.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you appove of.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void unlike_async (
            uint account_id,
            string message_id,
            out string result
            ) throws IOError
        {
            yield do_async ("unlike", account_id, message_id, out result);
        }

        /**
         * Delete a specific message on a specific protocol.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you wish to
         * obliterate. Must be owned by you.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Do fails.
         */
        public string delete (uint account_id, string message_id) throws IOError
        {
            return do_sync ("delete", account_id, message_id);
        }

        /**
         * Delete a specific message on a specific protocol, asynchronously.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @param message_id The id of the message you wish to
         * obliterate. Must be owned by you.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void delete_async (
            uint account_id,
            string message_id,
            out string result
            ) throws IOError
        {
            yield do_async ("delete", account_id, message_id, out result);
        }

        /**
         * Fetch the contacts from the specified account and store
         * them in Evolution Data Server.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.Do fails.
         */
        public string contacts (uint? account_id) throws IOError
        {
            return do_sync ("contacts", account_id, "");
        }

        /**
         * Fetch the contacts from the specified account and store
         * them in Evolution Data Server, asynchronously.
         *
         * @param account_id The {@link Ag.Account.id} to fetch.
         * @return ``true`` for success, ``false`` otherwise
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Service.Do fails.
         */
        public async void contacts_async (
            uint? account_id,
            out string result
            ) throws IOError
        {
            yield do_async ("contacts", account_id, "", out result);
        }

        /**
         * Pass any URL to this method and it will shorten it for you.
         *
         * @param url The long URL you need shortened.
         * @return The shortened URL.
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.URLShorten fails.
         */
        public string shorten (string url) throws IOError
        {
            return service.URLShorten (url);
        }

        /**
         * This method returns a list of all the possible actions
         * supported by a given protocol. This can be used to
         * determine what actions are available for the requested
         * protocol.
         *
         * @param protocol One of ``facebook``, ``flickr``,
         * ``foursquare``, ``identica``, or ``twitter``.
         * @return an array of features for ``protocol`` or ``null`` (i.e. no
         * features) if there are no features found for ``protocol``.
         *
         * @since 0.1
         * @throws GLib.Error if the call to Friends.Dispatcher.GetFeatures fails.
         */
        public string[] features (string protocol) throws Error
        {
            string[] _features = null;
                var parser = new Json.Parser ();
                parser.load_from_data (service.GetFeatures (protocol), -1);
                var nodes = parser.get_root ().get_array ().get_elements ();
                foreach (var node in nodes)
                   _features += node.dup_string ();
            return _features;
        }

        /**
         * Clear all notifications from the messaging menu.
         *
         * @since 0.1
         * @throws GLib.IOError if the call to Friends.Dispatcher.ClearIndicators
         * fails.
         */
        public void messaging_menu_clear () throws IOError
        {
            service.ClearIndicators ();
        }
    }
}
