/*
  This file is part of TALER
  Copyright (C) 2024 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free Software
  Foundation; either version 2.1, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License along with
  TALER; see the file COPYING.LGPL.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * @file merchant_api_delete_donau_instance.c
 * @brief Implementation of the DELETE /donau/$charity_id request of the merchant's HTTP API
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 */

#include "platform.h"
#include <curl/curl.h>
#include <jansson.h>
#include <microhttpd.h>
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include "taler_merchant_service.h"
#include "merchant_api_curl_defaults.h"
#include <taler/taler_json_lib.h>
#include <taler/taler_signatures.h>
/* DONAU RELATED IMPORTS */
#include "taler_merchant_donau.h"
#include <donau/donau_service.h>


/**
 * Handle for a DELETE /donau/$charity_id operation.
 */
struct TALER_MERCHANT_DonauInstanceDeleteHandle
{
  /**
   * The URL for this request.
   */
  char *url;

  /**
   * Handle for the request.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Function to call with the result.
   */
  TALER_MERCHANT_DonauInstanceDeleteCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Reference to the execution context.
   */
  struct GNUNET_CURL_Context *ctx;
};

/**
 * Function called when we're done processing the
 * HTTP DELETE /donau/$charity_id request.
 *
 * @param cls the struct TALER_MERCHANT_DonauInstanceDeleteHandle
 * @param response_code HTTP response code, 0 on error
 * @param response response body, NULL if not in JSON
 */
static void
handle_delete_donau_instance_finished (void *cls,
                                       long response_code,
                                       const void *response)
{
  struct TALER_MERCHANT_DonauInstanceDeleteHandle *ddh = cls;
  const json_t *json = response;
  struct TALER_MERCHANT_HttpResponse hr = {
    .http_status = (unsigned int) response_code,
    .reply = json
  };

  ddh->job = NULL;

  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Got /donau/$charity_id response with status code %u\n",
              (unsigned int) response_code);

  switch (response_code)
  {
  case MHD_HTTP_NO_CONTENT:
    break;
  case MHD_HTTP_NOT_FOUND:
  case MHD_HTTP_UNAUTHORIZED:
    hr.ec = TALER_JSON_get_error_code (json);
    hr.hint = TALER_JSON_get_error_hint (json);
    break;
  default:
    /* Unexpected response */
    hr.ec = TALER_JSON_get_error_code (json);
    hr.hint = TALER_JSON_get_error_hint (json);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d for DELETE /donau/$charity_id\n",
                (unsigned int) response_code,
                (int) hr.ec);
    break;
  }
  ddh->cb (ddh->cb_cls,
           &hr);
  TALER_MERCHANT_donau_instance_delete_cancel (ddh);
}


/**
 * Initiates the DELETE /donau/$charity_id operation.
 *
 * @param ctx CURL context
 * @param backend_url Base URL for the backend
 * @param charity_id The ID of the charity to delete
 * @param cb Callback function to handle the response
 * @param cb_cls Closure for @a cb
 * @return the handle for the operation, or NULL on error
 */
struct TALER_MERCHANT_DonauInstanceDeleteHandle *
TALER_MERCHANT_donau_instance_delete (
  struct GNUNET_CURL_Context *ctx,
  const char *backend_url,
  uint64_t charity_id,
  TALER_MERCHANT_DonauInstanceDeleteCallback cb,
  void *cb_cls)
{
  struct TALER_MERCHANT_DonauInstanceDeleteHandle *ddh;
  char *charity_id_str;
  CURL *eh;

  GNUNET_asprintf (&charity_id_str,
                   "private/donau/%ld",
                   charity_id);
  if (NULL == charity_id_str)
    return NULL;

  ddh = GNUNET_new (struct TALER_MERCHANT_DonauInstanceDeleteHandle);
  ddh->ctx = ctx;
  ddh->cb = cb;
  ddh->cb_cls = cb_cls;

  ddh->url = TALER_url_join (backend_url,
                             charity_id_str,
                             NULL);
  GNUNET_free (charity_id_str);

  if (NULL == ddh->url)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Could not construct request URL.\n");
    GNUNET_free (ddh);
    return NULL;
  }

  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Requesting URL '%s'\n",
              ddh->url);

  eh = TALER_MERCHANT_curl_easy_get_ (ddh->url);
  GNUNET_assert (CURLE_OK == curl_easy_setopt (eh,
                                               CURLOPT_CUSTOMREQUEST,
                                               "DELETE"));
  ddh->job = GNUNET_CURL_job_add (ctx,
                                  eh,
                                  &handle_delete_donau_instance_finished,
                                  ddh);

  return ddh;
}


/**
 * Cancel the DELETE /donau/$charity_id operation.
 *
 * @param ddh Handle for the operation to cancel.
 */
void
TALER_MERCHANT_donau_instance_delete_cancel (
  struct TALER_MERCHANT_DonauInstanceDeleteHandle *ddh)
{
  if (NULL != ddh->job)
    GNUNET_CURL_job_cancel (ddh->job);

  GNUNET_free (ddh->url);
  GNUNET_free (ddh);
}