# pyOCD debugger
# Copyright (c) 2023 NXP
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from ...coresight.coresight_target import CoreSightTarget
from ...core.memory_map import (FlashRegion, RamRegion, MemoryMap)

# AP IDs:
# [1]   APB_AP
# [4]   CM7_0_AHB_AP
# [6]   MDM_AP
# [7]   SDA_AP
APB_AP_ID       = 1
CM7_0_AHB_AP_ID = 4
MDM_AP_ID       = 6
SDA_AP_ID       = 7
AP_ID_LIST      = [APB_AP_ID, CM7_0_AHB_AP_ID, MDM_AP_ID, SDA_AP_ID]

# SDA_AP registers:
# [0x80]    Debug Enable Control (DBGENCTRL)
SDA_AP_DBGENCTRL_ADDR   = 0x80

# SDA_AP DBGENCTRL bit fields:
# [31:30]   reserved
# [29]      Core Non-Invasive Debug Enable (CNIDEN)
# [28]      Core Debug Enable (CDBGEN)
# [27:8]    reserved
# [7]       Global Secure Privileged Non-Invasive Debug Enable (GSPNIDEN)
# [6]       Global Secure Privileged Debug Enable (GSPIDEN)
# [5]       Global Non-Invasive Debug Enable (GNIDEN)
# [4]       Global Debug Enable (GDBGEN)
# [3:0]     reserved
SDA_AP_CNIDEN_MASK      = 0x20000000
SDA_AP_CNIDEN_SHIFT     = 29
SDA_AP_CDBGEN_MASK      = 0x10000000
SDA_AP_CDBGEN_SHIFT     = 28
SDA_AP_GSPNIDEN_MASK    = 0x80
SDA_AP_GSPNIDEN_SHIFT   = 7
SDA_AP_GSPIDEN_MASK     = 0x40
SDA_AP_GSPIDEN_SHIFT    = 6
SDA_AP_GNIDEN_MASK      = 0x20
SDA_AP_GNIDEN_SHIFT     = 5
SDA_AP_GDBGEN_MASK      = 0x10
SDA_AP_GDBGEN_SHIFT     = 4
SDA_AP_EN_ALL           = (SDA_AP_CNIDEN_MASK | SDA_AP_CDBGEN_MASK | SDA_AP_GSPNIDEN_MASK |
                           SDA_AP_GSPIDEN_MASK | SDA_AP_GNIDEN_MASK | SDA_AP_GDBGEN_MASK)

FLASH_ALGO_CODE = {
    'load_address' : 0x20000000,

    # Flash algorithm as a hex string
    'instructions': [
    0xe7fdbe00,
    0x4a15b5f0, 0x21024648, 0x4d145481, 0x21035d44, 0x4b130409, 0x460f2c01, 0x461fd000, 0x4e124b11,
    0x461c2c01, 0x4634d000, 0x433e6826, 0x5c846026, 0x2c02480e, 0x464cd10e, 0x54a6260a, 0x2a015d62,
    0x4907d000, 0x4a082a01, 0x4613d000, 0x420a681a, 0x4807d100, 0x46c0bdf0, 0x000007f4, 0x000007e5,
    0xf3130000, 0x402f0024, 0x402ec004, 0x000027e4, 0x00005aa5, 0x4aa0b5f0, 0x2402464b, 0x281f549c,
    0x4c9ed819, 0x464b6d26, 0x549d250a, 0x461d2301, 0x422e4085, 0x4e9ad00b, 0x27031986, 0x683643be,
    0x0ec00780, 0xb2f040c6, 0xd0004288, 0x6920e115, 0x612043a8, 0x287fe110, 0x4c90d81c, 0x464b6de7,
    0x549d250a, 0x3b204603, 0x230108de, 0x40b5461d, 0xd00b422f, 0x40382718, 0x43be2703, 0x19f64f88,
    0x40c66c76, 0x4288b2f0, 0xe0f6d000, 0x43a869e0, 0xe0f161e0, 0xd81a289f, 0x6c264c80, 0x250a464b,
    0x4607549d, 0x23013f80, 0x40bd461d, 0xd00a422e, 0x27031906, 0x683643be, 0x0ec00780, 0xb2f040c6,
    0xd0004288, 0x6820e0d9, 0x602043a8, 0x28ffe0d4, 0x4c72d81c, 0x464b6e27, 0x549d250a, 0x3ba04603,
    0x230108de, 0x40b5461d, 0xd00b422f, 0x40382718, 0x43be2703, 0x19f64f6a, 0x40c66d76, 0x4288b2f0,
    0xe0bad000, 0x43a86a20, 0xe0b56220, 0x021c2301, 0x35204625, 0xd21d42a8, 0x6c664c60, 0x250a464b,
    0x23ff549d, 0x18c743db, 0x461d2301, 0x422e40bd, 0x4e5dd00c, 0x36c01986, 0x43be2703, 0x07806836,
    0x40c60ec0, 0x4288b2f0, 0xe096d000, 0x43a86860, 0xe0916060, 0x35804625, 0xd21c42a8, 0x6e674c4f,
    0x250a464b, 0x4b4f549d, 0x33e018c3, 0x230108de, 0x40b5461d, 0xd00a422f, 0x40382718, 0x43be2703,
    0x19f64f47, 0x40c66e76, 0x4288b2f0, 0x6a60d175, 0x626043a8, 0x34a0e070, 0xd21c42a0, 0x6ca64c3f,
    0x250a464b, 0x4b3f549d, 0x378018c7, 0x461d2301, 0x422e40bd, 0x4e3cd00b, 0x36601986, 0x43be2703,
    0x07806836, 0x40c60ec0, 0x4288b2f0, 0x68a0d155, 0x60a043a8, 0x2411e050, 0x46250164, 0x42a83d20,
    0x4c2ed21c, 0x464b6ea7, 0x549d250a, 0x18c34b2d, 0x08de3360, 0x461d2301, 0x422f40b5, 0x2718d00a,
    0x27034038, 0x4f2643be, 0x6f7619f6, 0xb2f040c6, 0xd1324288, 0x43a86aa0, 0xe02d62a0, 0xd21a42a0,
    0x6ce64c1e, 0x250a464b, 0x4b1e549d, 0x230118c7, 0x40bd461d, 0xd00a422e, 0x19864e1b, 0x43be2703,
    0x07806836, 0x40c60ec0, 0x4288b2f0, 0x68e0d115, 0x60e043a8, 0xd110e010, 0x6d844810, 0x260a464d,
    0x07e454ae, 0x4c0ed004, 0xb2e46c24, 0xd104428c, 0x23016981, 0x61814399, 0x46482300, 0x480b5c81,
    0x4601290a, 0x2b00d002, 0xbdf0d003, 0x2b004908, 0x4608d1fb, 0x46c0bdf0, 0x000007f4, 0x40268340,
    0x40268440, 0xfffffe00, 0x40268220, 0x000027e4, 0x00005aa5, 0x4807b580, 0x5c0b4649, 0x0441201f,
    0x46022000, 0xf8cef000, 0x1a404903, 0x41881e41, 0x46c0bd80, 0x00000018, 0x00005aa5, 0x4907b580,
    0x5c53464a, 0x18404906, 0x03492101, 0xf0002200, 0x4904f8b9, 0x1e411a40, 0xbd804188, 0x00000018,
    0xffc00000, 0x00005aa5, 0xb087b5f0, 0x4e4d460c, 0x22024649, 0x9006558a, 0x90052000, 0x5808484a,
    0x4d4a0340, 0x99063dc0, 0xd3004281, 0x1e61e086, 0xd3004281, 0x4946e082, 0x4b466808, 0x60084318,
    0x5d804648, 0x28024d44, 0x9202d178, 0x2703464d, 0x484255af, 0x98065829, 0x00800b40, 0x90039104,
    0xf0005840, 0x9005fd13, 0x03432001, 0x98061e59, 0x99054008, 0x91051809, 0x99055daa, 0xd15a2a0a,
    0x9d039a04, 0x920418aa, 0x1a189301, 0x9b021840, 0x558b4649, 0x99059a05, 0xd3092c04, 0xd107403a,
    0x1c496809, 0x9905d107, 0x91051d09, 0xe0081f24, 0x29ff7809, 0x4d25d001, 0x9905e004, 0x91051c49,
    0x4d271e64, 0x558f4649, 0x68094921, 0x42114a21, 0x4a1fd009, 0x4d1f6811, 0x60114329, 0x5d894649,
    0x4d1d2903, 0x464ad102, 0x55912104, 0x42829a05, 0x9403d114, 0x9f04463c, 0xf0006878, 0x9b02fcc7,
    0x46489005, 0x98055d81, 0x18809a01, 0x97041d3f, 0x9c034627, 0x4a10290a, 0xe003d101, 0x2904462a,
    0xd10a4615, 0x220a4649, 0x2c00558a, 0x490cd006, 0xd0ad428d, 0x4d08e002, 0x4d07e000, 0xb0074628,
    0x46c0bdf0, 0x000007f4, 0x000007ec, 0x00002e74, 0x402ec004, 0xf3100000, 0x000027e4, 0x000007e8,
    0x00005aa5, 0xb08fb5f0, 0x460b461e, 0x46494d95, 0x554f2702, 0x4894900e, 0x990e5808, 0x4c9304c9,
    0x4620d002, 0xbdf0b00f, 0x420b4991, 0x0340d1f9, 0x42811e59, 0x488fd2f5, 0x5c084649, 0xd1f02800,
    0x9202970a, 0x4649488c, 0x990e5808, 0x00890b49, 0x900c1808, 0x03812001, 0x900b9104, 0x900503c0,
    0x90062003, 0x90030400, 0x43c02000, 0x4c859007, 0x93014f86, 0x990a4648, 0x20115541, 0x68390100,
    0xd0004201, 0x4882e0e4, 0x28006800, 0xe0dfd500, 0x980c9309, 0x46316800, 0xfd8cf7ff, 0xd00042a0,
    0x4648e0d6, 0x280a5d40, 0xe0d1d000, 0x980a464c, 0x980c5560, 0xf0006800, 0x4976fc61, 0xd0004288,
    0x5d60e0c6, 0xd000280a, 0x464de0c2, 0x99064865, 0x980c5429, 0xf0006800, 0x900dfc29, 0x5c284861,
    0x280a4d60, 0xe0b3d000, 0x21044648, 0x496b5541, 0x29015c41, 0xd1174b64, 0x58404967, 0x1a09990b,
    0x980d4c60, 0x1c486018, 0x2900683a, 0x0c11d004, 0x42b1b2c9, 0xd1f44601, 0xd1002801, 0x4648e09b,
    0x28045d40, 0xe093d00d, 0x6018980d, 0x0c006838, 0x42b0b2c0, 0x4648d1f8, 0x28045d40, 0xd0004c51,
    0x4648e086, 0x55412105, 0x4a519907, 0x683a66d1, 0x91082110, 0x603a430a, 0x2220683b, 0x603b4393,
    0x990b683b, 0x603b430b, 0x28055d40, 0x4648d170, 0x55432306, 0x5c434949, 0x99052b01, 0x4b46d11e,
    0x9b0758c0, 0x46034358, 0x1c40687d, 0xd0012b00, 0xd0f8400d, 0x2107464d, 0x55294c32, 0x2b00687d,
    0x9904d022, 0xd01f400d, 0x99036878, 0xd1024008, 0x1d09990c, 0x4d2b910c, 0xe0104c32, 0x42086878,
    0x4648d0fc, 0x55412107, 0x99046878, 0xd0094208, 0x99036878, 0xd1024008, 0x1d09990c, 0x2800910c,
    0xd0074620, 0xe0054825, 0x48252801, 0x4823d000, 0x4c244d1c, 0x23084649, 0x6839554b, 0x43999b0b,
    0x68396039, 0x43999b08, 0x68396039, 0x60394391, 0xd11b42a0, 0x5d404648, 0xd1192808, 0x9b094820,
    0xd000181b, 0x4648e726, 0x5541210a, 0x28019802, 0xe6f6d000, 0x9901980e, 0xfe36f7ff, 0xd10d42a0,
    0x5d404648, 0xd000280a, 0x46204c0c, 0xbdf0b00f, 0xb00f480a, 0x480abdf0, 0xbdf0b00f, 0x30c04803,
    0xbdf0b00f, 0x000007f4, 0x000007ec, 0x00002db4, 0x00001fff, 0x000007e5, 0x000007e8, 0x000027e4,
    0x00002bd4, 0x00005aa5, 0x40268300, 0x402ec000, 0x402ec094, 0x0000d42b, 0x000007f0, 0x000007e4,
    0xffffe000, 0xb081b5f0, 0x4e674601, 0x22024648, 0x684d5582, 0x68ca688b, 0x51054c64, 0x51034c64,
    0x50c24b64, 0x4c64780b, 0x7c0b5503, 0x55c34f63, 0x55832303, 0x29004862, 0xe09bd100, 0x428a4961,
    0xe097d900, 0x95004633, 0x2605464c, 0x461d54e6, 0xfc26f7ff, 0x485c4601, 0xd1004281, 0x5d61e08a,
    0xd000290a, 0x4649e086, 0x5dc9554e, 0xd0612901, 0x9d002900, 0x2001d17e, 0x4b570202, 0x40116819,
    0x2610681c, 0x18744026, 0x2c0003c1, 0x681cd01a, 0x601c4384, 0x4e48464c, 0x2c015da4, 0x426cd10a,
    0x1c66685f, 0xd0022c00, 0x4634400f, 0x2e01d0f8, 0xe028d103, 0x420c685c, 0x2411d0fc, 0x681e0124,
    0x601e43a6, 0x2607464c, 0x55e64f37, 0x4e43681c, 0x601c4034, 0x438c681c, 0x4b41601c, 0x4214681c,
    0x681ad01d, 0x601a4382, 0x4c33464a, 0x2a015d12, 0x426cd10c, 0x1c62681d, 0xd0022c00, 0x4614400d,
    0x2a01d0f8, 0x4832d105, 0xbdf0b001, 0x420a681a, 0x4934d0fc, 0x681a3110, 0x601a400a, 0x46494f22,
    0x55ca220a, 0x681902c0, 0x60194381, 0x40306818, 0xe0356018, 0x2101462e, 0x4824020a, 0x40136803,
    0x24106802, 0x18e24014, 0x6802d01f, 0x6002438a, 0x4b19464a, 0x2a015cd2, 0x9a00d10e, 0x03ca4254,
    0x1c636845, 0xd0022c00, 0x461c4015, 0x2b01d0f8, 0x4817d106, 0xbdf0b001, 0x684303ca, 0xd0fc4213,
    0x01122211, 0x43936803, 0x464a6003, 0x5593230a, 0x6803030a, 0x60034393, 0x68020749, 0x6002430a,
    0xb001480c, 0x46c0bdf0, 0x000007f4, 0x000007f0, 0x000007e8, 0x000007ec, 0x000007e4, 0x000007e5,
    0x00002db4, 0x00000221, 0x000027e4, 0x402f0020, 0x00002bd4, 0x00005aa5, 0x402ec000, 0xffffefff,
    0x402ec0f0, 0xfffffeef, 0xb08db5f0, 0x460d920a, 0x2000900c, 0x9e0c900b, 0x464a489d, 0x499d5011,
    0x54502002, 0x5c11489c, 0x5814489c, 0x5810489c, 0x4f9c0340, 0x92099a0c, 0x42829a0c, 0x2900d204,
    0x990cd102, 0xd0020749, 0xb00d4638, 0x990abdf0, 0xd0f92900, 0x25079508, 0x9507460a, 0xd1f3402a,
    0x2a009a08, 0x4288d0f0, 0x4648d3ee, 0x22039006, 0x9d064888, 0x488c542a, 0x22116800, 0x42100112,
    0x4f8ad001, 0x488ae7e0, 0x28006800, 0x4648d459, 0x4a802504, 0x54859506, 0x2b014d86, 0x980cd10f,
    0xfcc2f7ff, 0xd00042a8, 0x4648e0cc, 0x5c404979, 0xd000280a, 0x462ae0c8, 0x9b064648, 0x0b705443,
    0x19040080, 0x68209913, 0xf7ff9105, 0x42a8fb33, 0x9406d139, 0xf0006820, 0x4977fa11, 0x4f734288,
    0x9501d1b2, 0x4e6b4648, 0x280a5d80, 0x464cd1ac, 0x55a02005, 0x0880980c, 0x4001211e, 0x5421486f,
    0x68009806, 0xf9d2f000, 0x2001900b, 0x1e690345, 0x4008980c, 0x1809990b, 0x5da2910b, 0x2a0a990b,
    0x9a12d192, 0x464a9200, 0x5593230b, 0x18401a28, 0x207f9004, 0x900343c0, 0x4e5e9b0a, 0x9502462c,
    0x4f5ae00d, 0x4f59e780, 0x4649e77e, 0x9c0255cc, 0x90041900, 0x4e579a0a, 0x2b00462b, 0x4648d078,
    0x5c414956, 0xd3112b80, 0xd10f2900, 0x980b461d, 0x9a052180, 0xf8a6f000, 0x4047270a, 0xd00042b7,
    0x980be762, 0x900b3080, 0xe0239803, 0x461d980b, 0xd3112b20, 0x40119a07, 0x2120d10e, 0xf0009a05,
    0x270af891, 0x42b74047, 0xe74dd000, 0x3020980b, 0x9803900b, 0xe00d3060, 0x9a052108, 0xf882f000,
    0x4047270a, 0xd00042b7, 0x980be73e, 0x900b3008, 0x30789803, 0x4f2f4649, 0x290a5dc9, 0xd1359a0a,
    0x182b462b, 0x24064648, 0x980b55c4, 0xd0b42b00, 0x42889904, 0x461dd1b1, 0x68409806, 0xf7ff9905,
    0x42b0fa99, 0x4648d13a, 0x280a5dc0, 0x9806d138, 0x464e1d00, 0x900655f4, 0xf0006800, 0x4926f96f,
    0xd12f4288, 0x280a5df0, 0x464ed12e, 0x980655f4, 0xf0006800, 0x900bf93b, 0x980b5df1, 0xd084290a,
    0xe7014f1a, 0xe6ff37c0, 0xe6fd4f18, 0xe6fb4f17, 0x210a4648, 0x980055c1, 0xd10d2801, 0x99089809,
    0xfa20f000, 0xd11142b0, 0x5dc04648, 0xd001280a, 0x9001480e, 0xe6e79f01, 0xe6e54637, 0xe6e34f0b,
    0xe6e14f0a, 0xe6df4f09, 0xe6dd4f08, 0xe6db4f0c, 0x0000001c, 0x000007f4, 0x000007e5, 0x000007e8,
    0x000007ec, 0x00002db4, 0x402ec000, 0x000027e4, 0x402ec094, 0x00005aa5, 0x0000d42b, 0x00000020,
    0x000033cc, 0x4d6eb5f0, 0x2402464b, 0x4c6d555c, 0x4c6d5d1b, 0x4d6e4f6d, 0xd1202b01, 0x4b6d4625,
    0x58e3464c, 0x1ae42401, 0x1c636038, 0x68364e68, 0xd0042c00, 0xb2e40c34, 0x461c4294, 0x4648d1f4,
    0x5c804a5f, 0xd1522802, 0x22034648, 0x55024c5c, 0x2b014a5b, 0x4d5e462c, 0x485fd112, 0x6038bdf0,
    0x0c1b682b, 0x4293b2db, 0x4648d1f9, 0x5c804a54, 0xd0002802, 0x4648e08a, 0x4b512203, 0x4a5054c2,
    0xd03c0888, 0x4a564955, 0x4d51e003, 0x1e40549c, 0x464bd02f, 0x585b585c, 0xd01807a4, 0x464b781c,
    0x1c76585e, 0x585e505e, 0x585f7836, 0x505f1c7f, 0x783f585f, 0x1c6d585d, 0x0235505d, 0x043d192c,
    0x585b192c, 0x061b781b, 0x2401191e, 0x681ee001, 0x464b2404, 0x192c585d, 0x5c9c505c, 0x4f4100a5,
    0x1c6451ee, 0x2d20b2e5, 0x2400d1cf, 0x4628e7cd, 0x4648bdf0, 0x5c804a32, 0x4c332803, 0x4648d146,
    0x54812104, 0x02112201, 0x430b682b, 0x682b602b, 0x602b4313, 0x5cc04b2b, 0x280103d6, 0x482dd120,
    0x5818464b, 0x46074240, 0x1c40686b, 0xd0012f00, 0xd0f84033, 0x4e22464b, 0x2b045d9b, 0x464bd126,
    0x559c240a, 0x2f00686b, 0x0394d022, 0xd01f4023, 0x6840481f, 0x0f800380, 0xd11c4c1b, 0xe0204822,
    0x42306868, 0x4648d0fc, 0x5d404d15, 0xd10d2804, 0x230a4648, 0x4d165543, 0x04406868, 0xd5114620,
    0x03806868, 0xd0080f80, 0xe00b4620, 0xbdf04620, 0xd0042801, 0x46204c0c, 0x4813e003, 0x480ee002,
    0x4d0b4c09, 0x4393682b, 0x682a602b, 0x602a438a, 0x4a034649, 0x290a5c89, 0x4620d000, 0x46c0bdf0,
    0x000007f4, 0x000007e4, 0x000027ee, 0x40268300, 0x402ec000, 0x000007f0, 0x00002bde, 0x0000001c,
    0x00000020, 0x402ec100, 0x00005aaf, 0x2100b081, 0x910043c9, 0xd805281f, 0x07092101, 0x18400340,
    0xe0119000, 0x2a100942, 0x210fd805, 0x03400489, 0x90001840, 0x2211e008, 0x42900152, 0x201bd103,
    0x90000600, 0x9100e000, 0x46494803, 0x540a220a, 0xb0019800, 0x46c04770, 0x000007f4, 0x4937b510,
    0x2302464a, 0x281f5453, 0x2201d804, 0x48354082, 0xe0546900, 0xd806287f, 0x08c03820, 0x40822201,
    0x69c04830, 0x289fe04b, 0x3880d805, 0x40822201, 0x6800482c, 0x2201e043, 0xd80528ff, 0x08c038a0,
    0x48284082, 0xe03a6a00, 0x461c0213, 0x42a03420, 0x23ffd206, 0x18c043db, 0x48224082, 0xe02e6840,
    0x3480461c, 0xd20742a0, 0x18c04b1f, 0x08c030e0, 0x481c4082, 0xe0226a40, 0x429833a0, 0x4b1ad206,
    0x308018c0, 0x48174082, 0xe0186880, 0x015b2311, 0x3c20461c, 0xd20742a0, 0x18c04b13, 0x08c03060,
    0x48104082, 0xe00a6a80, 0xd2054298, 0x18c04b0e, 0x480c4082, 0xe00268c0, 0x480ad10d, 0x40106980,
    0x230a464a, 0x28005453, 0x4809d002, 0xbd10b280, 0xb2804806, 0x4802bd10, 0xbd10b280, 0x000007f4,
    0x0000e187, 0x40268340, 0xfffffe00, 0xffffd42b, 0xffffd81b, 0x4c14b5b0, 0x19054648, 0xf91ef000,
    0x0101201f, 0x1c422000, 0xc5013020, 0x46104291, 0x480ed1f9, 0x2301464a, 0x18105413, 0x490c60c1,
    0x490c6041, 0x4b0c6809, 0x54d52500, 0x400b230f, 0x5453490a, 0x60811911, 0xf7ff7405, 0x4908fbf3,
    0x1e411a40, 0xbdb04188, 0x00000024, 0x00000004, 0x000f4254, 0x402780f4, 0x000007f4, 0x00000018,
    0x00005aa5, 0xb082b510, 0x490b460c, 0x5c59464b, 0x93002300, 0x07619101, 0x3408d002, 0x438c2107,
    0x18404906, 0x46224611, 0xfcbef7ff, 0x1a404904, 0x41881e41, 0xbd10b002, 0x00000018, 0xffc00000,
    0x00005aa5, 0xb087b5f0, 0x464d4c5a, 0x93012302, 0x4614552b, 0x20009006, 0x48579005, 0x4d575828,
    0x2c009a06, 0xe0a2d100, 0x42820340, 0xe09ed300, 0xd1002900, 0x42a0e09b, 0xe098d200, 0x68104a50,
    0x43184b50, 0x60104b4b, 0x5cc04648, 0x28024d4e, 0xe08cd000, 0x2603464d, 0x910454ee, 0x5829484b,
    0x0b409806, 0x91020080, 0x58409000, 0xfecef7ff, 0x90054b40, 0x03472001, 0x98061e79, 0x99054008,
    0x91051809, 0x99055cea, 0xd1662a0a, 0x9a029603, 0x18ae9d00, 0x18401a38, 0x46499000, 0x54c89801,
    0x9a059b04, 0x2c049904, 0xd30f9d03, 0x402a431a, 0x6809d10c, 0x68129a05, 0xd116428a, 0x1d099905,
    0x99049105, 0x91041d09, 0xe00b1f24, 0x9a057809, 0x428a7812, 0x9905d109, 0x91051c49, 0x1c499904,
    0x1e649104, 0x4d2b9402, 0x9402e001, 0x4a244d28, 0x46494b24, 0x4c1f9803, 0x68115508, 0xd0074219,
    0x43196811, 0x46496011, 0x29035d09, 0xd1084d1e, 0x22044649, 0x9905550a, 0x42819800, 0x4623d006,
    0x9a05e011, 0x42829800, 0xd11a4d17, 0xf7ff6870, 0x4b10fe6d, 0x46489005, 0x98055cc1, 0xd10e290a,
    0x19c01d36, 0x46499000, 0x54ca220a, 0x2c009c02, 0x4810d00d, 0xd0a04285, 0x4d0be009, 0x4d0ae007,
    0x2904e005, 0xd1024d08, 0x210a4648, 0x46285501, 0xbdf0b007, 0x000007f4, 0x000007ec, 0x00002db4,
    0x402ec004, 0xf3100000, 0x000027e4, 0x000007e8, 0x000033cc, 0x00005aa5, 0x47702000, 0x8010f3ef,
    0x0fc007c0, 0x4770b672, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000
    ],

    # Relative function addresses
    'pc_init': 0x20000f39,
    'pc_unInit': 0x2000117d,
    'pc_program_page': 0x20000fa9,
    'pc_erase_sector': 0x20000341,
    'pc_eraseAll': 0x20000319,

    'static_base' : 0x20000000 + 0x00000004 + 0x00001188,
    'begin_stack' : 0x20006990,
    'end_stack' : 0x20005990,
    'begin_data' : 0x20000000 + 0x1000,
    'page_size' : 0x2000,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    # Enable double buffering
    'page_buffers' : [
        0x20001990,
        0x20003990
    ],
    'min_program_length' : 0x2000,

    # Relative region addresses and sizes
    'ro_start': 0x4,
    'ro_size': 0x1188,
    'rw_start': 0x118c,
    'rw_size': 0x4,
    'zi_start': 0x1190,
    'zi_size': 0x7f4,

    # Flash information
    'flash_start': 0x400000,
    'flash_size': 0x3e0000,
    'sector_sizes': (
        (0x0, 0x2000),
    )
}

FLASH_ALGO_DATA = {
    'load_address' : 0x20000000,

    # Flash algorithm as a hex string
    'instructions': [
    0xe7fdbe00,
    0x4a15b5f0, 0x21024648, 0x4d145481, 0x21035d44, 0x4b130409, 0x460f2c01, 0x461fd000, 0x4e124b11,
    0x461c2c01, 0x4634d000, 0x433e6826, 0x5c846026, 0x2c02480e, 0x464cd10e, 0x54a6260a, 0x2a015d62,
    0x4907d000, 0x4a082a01, 0x4613d000, 0x420a681a, 0x4807d100, 0x46c0bdf0, 0x00000074, 0x00000065,
    0xf3130000, 0x402f0024, 0x402ec004, 0x000027e4, 0x00005aa5, 0x4aa0b5f0, 0x2402464b, 0x281f549c,
    0x4c9ed819, 0x464b6d26, 0x549d250a, 0x461d2301, 0x422e4085, 0x4e9ad00b, 0x27031986, 0x683643be,
    0x0ec00780, 0xb2f040c6, 0xd0004288, 0x6920e115, 0x612043a8, 0x287fe110, 0x4c90d81c, 0x464b6de7,
    0x549d250a, 0x3b204603, 0x230108de, 0x40b5461d, 0xd00b422f, 0x40382718, 0x43be2703, 0x19f64f88,
    0x40c66c76, 0x4288b2f0, 0xe0f6d000, 0x43a869e0, 0xe0f161e0, 0xd81a289f, 0x6c264c80, 0x250a464b,
    0x4607549d, 0x23013f80, 0x40bd461d, 0xd00a422e, 0x27031906, 0x683643be, 0x0ec00780, 0xb2f040c6,
    0xd0004288, 0x6820e0d9, 0x602043a8, 0x28ffe0d4, 0x4c72d81c, 0x464b6e27, 0x549d250a, 0x3ba04603,
    0x230108de, 0x40b5461d, 0xd00b422f, 0x40382718, 0x43be2703, 0x19f64f6a, 0x40c66d76, 0x4288b2f0,
    0xe0bad000, 0x43a86a20, 0xe0b56220, 0x021c2301, 0x35204625, 0xd21d42a8, 0x6c664c60, 0x250a464b,
    0x23ff549d, 0x18c743db, 0x461d2301, 0x422e40bd, 0x4e5dd00c, 0x36c01986, 0x43be2703, 0x07806836,
    0x40c60ec0, 0x4288b2f0, 0xe096d000, 0x43a86860, 0xe0916060, 0x35804625, 0xd21c42a8, 0x6e674c4f,
    0x250a464b, 0x4b4f549d, 0x33e018c3, 0x230108de, 0x40b5461d, 0xd00a422f, 0x40382718, 0x43be2703,
    0x19f64f47, 0x40c66e76, 0x4288b2f0, 0x6a60d175, 0x626043a8, 0x34a0e070, 0xd21c42a0, 0x6ca64c3f,
    0x250a464b, 0x4b3f549d, 0x378018c7, 0x461d2301, 0x422e40bd, 0x4e3cd00b, 0x36601986, 0x43be2703,
    0x07806836, 0x40c60ec0, 0x4288b2f0, 0x68a0d155, 0x60a043a8, 0x2411e050, 0x46250164, 0x42a83d20,
    0x4c2ed21c, 0x464b6ea7, 0x549d250a, 0x18c34b2d, 0x08de3360, 0x461d2301, 0x422f40b5, 0x2718d00a,
    0x27034038, 0x4f2643be, 0x6f7619f6, 0xb2f040c6, 0xd1324288, 0x43a86aa0, 0xe02d62a0, 0xd21a42a0,
    0x6ce64c1e, 0x250a464b, 0x4b1e549d, 0x230118c7, 0x40bd461d, 0xd00a422e, 0x19864e1b, 0x43be2703,
    0x07806836, 0x40c60ec0, 0x4288b2f0, 0x68e0d115, 0x60e043a8, 0xd110e010, 0x6d844810, 0x260a464d,
    0x07e454ae, 0x4c0ed004, 0xb2e46c24, 0xd104428c, 0x23016981, 0x61814399, 0x46482300, 0x480b5c81,
    0x4601290a, 0x2b00d002, 0xbdf0d003, 0x2b004908, 0x4608d1fb, 0x46c0bdf0, 0x00000074, 0x40268340,
    0x40268440, 0xfffffe00, 0x40268220, 0x000027e4, 0x00005aa5, 0x4807b580, 0x5c0b4649, 0x04412001,
    0x46022000, 0xf8cef000, 0x1a404903, 0x41881e41, 0x46c0bd80, 0x00000018, 0x00005aa5, 0x4908b580,
    0x5c53464a, 0x0709210f, 0x21011840, 0x22000349, 0xf8b8f000, 0x1a404903, 0x41881e41, 0x46c0bd80,
    0x00000018, 0x00005aa5, 0xb087b5f0, 0x4e4d460c, 0x22024649, 0x9006558a, 0x90052000, 0x5808484a,
    0x4d4a0340, 0x99063dc0, 0xd3004281, 0x1e61e086, 0xd3004281, 0x4946e082, 0x4b466808, 0x60084318,
    0x5d804648, 0x28024d44, 0x9202d178, 0x2703464d, 0x484255af, 0x98065829, 0x00800b40, 0x90039104,
    0xf0005840, 0x9005fd13, 0x03432001, 0x98061e59, 0x99054008, 0x91051809, 0x99055daa, 0xd15a2a0a,
    0x9d039a04, 0x920418aa, 0x1a189301, 0x9b021840, 0x558b4649, 0x99059a05, 0xd3092c04, 0xd107403a,
    0x1c496809, 0x9905d107, 0x91051d09, 0xe0081f24, 0x29ff7809, 0x4d25d001, 0x9905e004, 0x91051c49,
    0x4d271e64, 0x558f4649, 0x68094921, 0x42114a21, 0x4a1fd009, 0x4d1f6811, 0x60114329, 0x5d894649,
    0x4d1d2903, 0x464ad102, 0x55912104, 0x42829a05, 0x9403d114, 0x9f04463c, 0xf0006878, 0x9b02fcc7,
    0x46489005, 0x98055d81, 0x18809a01, 0x97041d3f, 0x9c034627, 0x4a10290a, 0xe003d101, 0x2904462a,
    0xd10a4615, 0x220a4649, 0x2c00558a, 0x490cd006, 0xd0ad428d, 0x4d08e002, 0x4d07e000, 0xb0074628,
    0x46c0bdf0, 0x00000074, 0x0000006c, 0x00002e74, 0x402ec004, 0xf3100000, 0x000027e4, 0x00000068,
    0x00005aa5, 0xb08fb5f0, 0x460b461e, 0x46494d95, 0x554f2702, 0x4894900e, 0x990e5808, 0x4c9304c9,
    0x4620d002, 0xbdf0b00f, 0x420b4991, 0x0340d1f9, 0x42811e59, 0x488fd2f5, 0x5c084649, 0xd1f02800,
    0x9202970a, 0x4649488c, 0x990e5808, 0x00890b49, 0x900c1808, 0x03812001, 0x900b9104, 0x900503c0,
    0x90062003, 0x90030400, 0x43c02000, 0x4c859007, 0x93014f86, 0x990a4648, 0x20115541, 0x68390100,
    0xd0004201, 0x4882e0e4, 0x28006800, 0xe0dfd500, 0x980c9309, 0x46316800, 0xfd8cf7ff, 0xd00042a0,
    0x4648e0d6, 0x280a5d40, 0xe0d1d000, 0x980a464c, 0x980c5560, 0xf0006800, 0x4976fc61, 0xd0004288,
    0x5d60e0c6, 0xd000280a, 0x464de0c2, 0x99064865, 0x980c5429, 0xf0006800, 0x900dfc29, 0x5c284861,
    0x280a4d60, 0xe0b3d000, 0x21044648, 0x496b5541, 0x29015c41, 0xd1174b64, 0x58404967, 0x1a09990b,
    0x980d4c60, 0x1c486018, 0x2900683a, 0x0c11d004, 0x42b1b2c9, 0xd1f44601, 0xd1002801, 0x4648e09b,
    0x28045d40, 0xe093d00d, 0x6018980d, 0x0c006838, 0x42b0b2c0, 0x4648d1f8, 0x28045d40, 0xd0004c51,
    0x4648e086, 0x55412105, 0x4a519907, 0x683a66d1, 0x91082110, 0x603a430a, 0x2220683b, 0x603b4393,
    0x990b683b, 0x603b430b, 0x28055d40, 0x4648d170, 0x55432306, 0x5c434949, 0x99052b01, 0x4b46d11e,
    0x9b0758c0, 0x46034358, 0x1c40687d, 0xd0012b00, 0xd0f8400d, 0x2107464d, 0x55294c32, 0x2b00687d,
    0x9904d022, 0xd01f400d, 0x99036878, 0xd1024008, 0x1d09990c, 0x4d2b910c, 0xe0104c32, 0x42086878,
    0x4648d0fc, 0x55412107, 0x99046878, 0xd0094208, 0x99036878, 0xd1024008, 0x1d09990c, 0x2800910c,
    0xd0074620, 0xe0054825, 0x48252801, 0x4823d000, 0x4c244d1c, 0x23084649, 0x6839554b, 0x43999b0b,
    0x68396039, 0x43999b08, 0x68396039, 0x60394391, 0xd11b42a0, 0x5d404648, 0xd1192808, 0x9b094820,
    0xd000181b, 0x4648e726, 0x5541210a, 0x28019802, 0xe6f6d000, 0x9901980e, 0xfe36f7ff, 0xd10d42a0,
    0x5d404648, 0xd000280a, 0x46204c0c, 0xbdf0b00f, 0xb00f480a, 0x480abdf0, 0xbdf0b00f, 0x30c04803,
    0xbdf0b00f, 0x00000074, 0x0000006c, 0x00002db4, 0x00001fff, 0x00000065, 0x00000068, 0x000027e4,
    0x00002bd4, 0x00005aa5, 0x40268300, 0x402ec000, 0x402ec094, 0x0000d42b, 0x00000070, 0x00000064,
    0xffffe000, 0xb081b5f0, 0x4e674601, 0x22024648, 0x684d5582, 0x68ca688b, 0x51054c64, 0x51034c64,
    0x50c24b64, 0x4c64780b, 0x7c0b5503, 0x55c34f63, 0x55832303, 0x29004862, 0xe09bd100, 0x428a4961,
    0xe097d900, 0x95004633, 0x2605464c, 0x461d54e6, 0xfc26f7ff, 0x485c4601, 0xd1004281, 0x5d61e08a,
    0xd000290a, 0x4649e086, 0x5dc9554e, 0xd0612901, 0x9d002900, 0x2001d17e, 0x4b570202, 0x40116819,
    0x2610681c, 0x18744026, 0x2c0003c1, 0x681cd01a, 0x601c4384, 0x4e48464c, 0x2c015da4, 0x426cd10a,
    0x1c66685f, 0xd0022c00, 0x4634400f, 0x2e01d0f8, 0xe028d103, 0x420c685c, 0x2411d0fc, 0x681e0124,
    0x601e43a6, 0x2607464c, 0x55e64f37, 0x4e43681c, 0x601c4034, 0x438c681c, 0x4b41601c, 0x4214681c,
    0x681ad01d, 0x601a4382, 0x4c33464a, 0x2a015d12, 0x426cd10c, 0x1c62681d, 0xd0022c00, 0x4614400d,
    0x2a01d0f8, 0x4832d105, 0xbdf0b001, 0x420a681a, 0x4934d0fc, 0x681a3110, 0x601a400a, 0x46494f22,
    0x55ca220a, 0x681902c0, 0x60194381, 0x40306818, 0xe0356018, 0x2101462e, 0x4824020a, 0x40136803,
    0x24106802, 0x18e24014, 0x6802d01f, 0x6002438a, 0x4b19464a, 0x2a015cd2, 0x9a00d10e, 0x03ca4254,
    0x1c636845, 0xd0022c00, 0x461c4015, 0x2b01d0f8, 0x4817d106, 0xbdf0b001, 0x684303ca, 0xd0fc4213,
    0x01122211, 0x43936803, 0x464a6003, 0x5593230a, 0x6803030a, 0x60034393, 0x68020749, 0x6002430a,
    0xb001480c, 0x46c0bdf0, 0x00000074, 0x00000070, 0x00000068, 0x0000006c, 0x00000064, 0x00000065,
    0x00002db4, 0x00000221, 0x000027e4, 0x402f0020, 0x00002bd4, 0x00005aa5, 0x402ec000, 0xffffefff,
    0x402ec0f0, 0xfffffeef, 0xb08db5f0, 0x460d920a, 0x2000900c, 0x9e0c900b, 0x464a489d, 0x499d5011,
    0x54502002, 0x5c11489c, 0x5814489c, 0x5810489c, 0x4f9c0340, 0x92099a0c, 0x42829a0c, 0x2900d204,
    0x990cd102, 0xd0020749, 0xb00d4638, 0x990abdf0, 0xd0f92900, 0x25079508, 0x9507460a, 0xd1f3402a,
    0x2a009a08, 0x4288d0f0, 0x4648d3ee, 0x22039006, 0x9d064888, 0x488c542a, 0x22116800, 0x42100112,
    0x4f8ad001, 0x488ae7e0, 0x28006800, 0x4648d459, 0x4a802504, 0x54859506, 0x2b014d86, 0x980cd10f,
    0xfcc2f7ff, 0xd00042a8, 0x4648e0cc, 0x5c404979, 0xd000280a, 0x462ae0c8, 0x9b064648, 0x0b705443,
    0x19040080, 0x68209913, 0xf7ff9105, 0x42a8fb33, 0x9406d139, 0xf0006820, 0x4977fa11, 0x4f734288,
    0x9501d1b2, 0x4e6b4648, 0x280a5d80, 0x464cd1ac, 0x55a02005, 0x0880980c, 0x4001211e, 0x5421486f,
    0x68009806, 0xf9d2f000, 0x2001900b, 0x1e690345, 0x4008980c, 0x1809990b, 0x5da2910b, 0x2a0a990b,
    0x9a12d192, 0x464a9200, 0x5593230b, 0x18401a28, 0x207f9004, 0x900343c0, 0x4e5e9b0a, 0x9502462c,
    0x4f5ae00d, 0x4f59e780, 0x4649e77e, 0x9c0255cc, 0x90041900, 0x4e579a0a, 0x2b00462b, 0x4648d078,
    0x5c414956, 0xd3112b80, 0xd10f2900, 0x980b461d, 0x9a052180, 0xf8a6f000, 0x4047270a, 0xd00042b7,
    0x980be762, 0x900b3080, 0xe0239803, 0x461d980b, 0xd3112b20, 0x40119a07, 0x2120d10e, 0xf0009a05,
    0x270af891, 0x42b74047, 0xe74dd000, 0x3020980b, 0x9803900b, 0xe00d3060, 0x9a052108, 0xf882f000,
    0x4047270a, 0xd00042b7, 0x980be73e, 0x900b3008, 0x30789803, 0x4f2f4649, 0x290a5dc9, 0xd1359a0a,
    0x182b462b, 0x24064648, 0x980b55c4, 0xd0b42b00, 0x42889904, 0x461dd1b1, 0x68409806, 0xf7ff9905,
    0x42b0fa99, 0x4648d13a, 0x280a5dc0, 0x9806d138, 0x464e1d00, 0x900655f4, 0xf0006800, 0x4926f96f,
    0xd12f4288, 0x280a5df0, 0x464ed12e, 0x980655f4, 0xf0006800, 0x900bf93b, 0x980b5df1, 0xd084290a,
    0xe7014f1a, 0xe6ff37c0, 0xe6fd4f18, 0xe6fb4f17, 0x210a4648, 0x980055c1, 0xd10d2801, 0x99089809,
    0xfa42f000, 0xd11142b0, 0x5dc04648, 0xd001280a, 0x9001480e, 0xe6e79f01, 0xe6e54637, 0xe6e34f0b,
    0xe6e14f0a, 0xe6df4f09, 0xe6dd4f08, 0xe6db4f0c, 0x0000001c, 0x00000074, 0x00000065, 0x00000068,
    0x0000006c, 0x00002db4, 0x402ec000, 0x000027e4, 0x402ec094, 0x00005aa5, 0x0000d42b, 0x00000020,
    0x000033cc, 0x4d6eb5f0, 0x2402464b, 0x4c6d555c, 0x4c6d5d1b, 0x4d6e4f6d, 0xd1202b01, 0x4b6d4625,
    0x58e3464c, 0x1ae42401, 0x1c636038, 0x68364e68, 0xd0042c00, 0xb2e40c34, 0x461c4294, 0x4648d1f4,
    0x5c804a5f, 0xd1522802, 0x22034648, 0x55024c5c, 0x2b014a5b, 0x4d5e462c, 0x485fd112, 0x6038bdf0,
    0x0c1b682b, 0x4293b2db, 0x4648d1f9, 0x5c804a54, 0xd0002802, 0x4648e08a, 0x4b512203, 0x4a5054c2,
    0xd03c0888, 0x4a564955, 0x4d51e003, 0x1e40549c, 0x464bd02f, 0x585b585c, 0xd01807a4, 0x464b781c,
    0x1c76585e, 0x585e505e, 0x585f7836, 0x505f1c7f, 0x783f585f, 0x1c6d585d, 0x0235505d, 0x043d192c,
    0x585b192c, 0x061b781b, 0x2401191e, 0x681ee001, 0x464b2404, 0x192c585d, 0x5c9c505c, 0x4f4100a5,
    0x1c6451ee, 0x2d20b2e5, 0x2400d1cf, 0x4628e7cd, 0x4648bdf0, 0x5c804a32, 0x4c332803, 0x4648d146,
    0x54812104, 0x02112201, 0x430b682b, 0x682b602b, 0x602b4313, 0x5cc04b2b, 0x280103d6, 0x482dd120,
    0x5818464b, 0x46074240, 0x1c40686b, 0xd0012f00, 0xd0f84033, 0x4e22464b, 0x2b045d9b, 0x464bd126,
    0x559c240a, 0x2f00686b, 0x0394d022, 0xd01f4023, 0x6840481f, 0x0f800380, 0xd11c4c1b, 0xe0204822,
    0x42306868, 0x4648d0fc, 0x5d404d15, 0xd10d2804, 0x230a4648, 0x4d165543, 0x04406868, 0xd5114620,
    0x03806868, 0xd0080f80, 0xe00b4620, 0xbdf04620, 0xd0042801, 0x46204c0c, 0x4813e003, 0x480ee002,
    0x4d0b4c09, 0x4393682b, 0x682a602b, 0x602a438a, 0x4a034649, 0x290a5c89, 0x4620d000, 0x46c0bdf0,
    0x00000074, 0x00000064, 0x000027ee, 0x40268300, 0x402ec000, 0x00000070, 0x00002bde, 0x0000001c,
    0x00000020, 0x402ec100, 0x00005aaf, 0x2100b081, 0x910043c9, 0xd805281f, 0x07092101, 0x18400340,
    0xe0119000, 0x2a100942, 0x210fd805, 0x03400489, 0x90001840, 0x2211e008, 0x42900152, 0x201bd103,
    0x90000600, 0x9100e000, 0x46494803, 0x540a220a, 0xb0019800, 0x46c04770, 0x00000074, 0x4937b510,
    0x2302464a, 0x281f5453, 0x2201d804, 0x48354082, 0xe0546900, 0xd806287f, 0x08c03820, 0x40822201,
    0x69c04830, 0x289fe04b, 0x3880d805, 0x40822201, 0x6800482c, 0x2201e043, 0xd80528ff, 0x08c038a0,
    0x48284082, 0xe03a6a00, 0x461c0213, 0x42a03420, 0x23ffd206, 0x18c043db, 0x48224082, 0xe02e6840,
    0x3480461c, 0xd20742a0, 0x18c04b1f, 0x08c030e0, 0x481c4082, 0xe0226a40, 0x429833a0, 0x4b1ad206,
    0x308018c0, 0x48174082, 0xe0186880, 0x015b2311, 0x3c20461c, 0xd20742a0, 0x18c04b13, 0x08c03060,
    0x48104082, 0xe00a6a80, 0xd2054298, 0x18c04b0e, 0x480c4082, 0xe00268c0, 0x480ad10d, 0x40106980,
    0x230a464a, 0x28005453, 0x4809d002, 0xbd10b280, 0xb2804806, 0x4802bd10, 0xbd10b280, 0x00000074,
    0x0000e187, 0x40268340, 0xfffffe00, 0xffffd42b, 0xffffd81b, 0xb085b5f0, 0xf942f000, 0x464a4b23,
    0x90042000, 0x482250d0, 0x91022101, 0x18d35411, 0x9003200f, 0x9101210e, 0x250c270d, 0x210a240b,
    0x625e2609, 0x3628461e, 0x9901c6b2, 0x63d86399, 0x90012008, 0x90002007, 0x26052406, 0x21032704,
    0x9d022002, 0x461d605d, 0xc5833508, 0x619c615e, 0x61d89800, 0x62189801, 0x1810480d, 0x60c12110,
    0x6041490c, 0x6809490c, 0x9d044c0c, 0x9c035515, 0x4c0b4021, 0x60835511, 0xf7ff7405, 0x4909fbd3,
    0x1e411a40, 0xb0054188, 0x46c0bdf0, 0x00000024, 0x00000004, 0x000f4254, 0x402780f4, 0x00000074,
    0x00000018, 0x00005aa5, 0xb082b510, 0x490c460c, 0x5c59464b, 0x93002300, 0x07619101, 0x3408d002,
    0x438c2107, 0x0709210f, 0x46111840, 0xf7ff4622, 0x4904fc9b, 0x1e411a40, 0xb0024188, 0x46c0bd10,
    0x00000018, 0x00005aa5, 0xb087b5f0, 0x464d4c5a, 0x93012302, 0x4614552b, 0x20009006, 0x48579005,
    0x4d575828, 0x2c009a06, 0xe0a2d100, 0x42820340, 0xe09ed300, 0xd1002900, 0x42a0e09b, 0xe098d200,
    0x68104a50, 0x43184b50, 0x60104b4b, 0x5cc04648, 0x28024d4e, 0xe08cd000, 0x2603464d, 0x910454ee,
    0x5829484b, 0x0b409806, 0x91020080, 0x58409000, 0xfeacf7ff, 0x90054b40, 0x03472001, 0x98061e79,
    0x99054008, 0x91051809, 0x99055cea, 0xd1662a0a, 0x9a029603, 0x18ae9d00, 0x18401a38, 0x46499000,
    0x54c89801, 0x9a059b04, 0x2c049904, 0xd30f9d03, 0x402a431a, 0x6809d10c, 0x68129a05, 0xd116428a,
    0x1d099905, 0x99049105, 0x91041d09, 0xe00b1f24, 0x9a057809, 0x428a7812, 0x9905d109, 0x91051c49,
    0x1c499904, 0x1e649104, 0x4d2b9402, 0x9402e001, 0x4a244d28, 0x46494b24, 0x4c1f9803, 0x68115508,
    0xd0074219, 0x43196811, 0x46496011, 0x29035d09, 0xd1084d1e, 0x22044649, 0x9905550a, 0x42819800,
    0x4623d006, 0x9a05e011, 0x42829800, 0xd11a4d17, 0xf7ff6870, 0x4b10fe4b, 0x46489005, 0x98055cc1,
    0xd10e290a, 0x19c01d36, 0x46499000, 0x54ca220a, 0x2c009c02, 0x4810d00d, 0xd0a04285, 0x4d0be009,
    0x4d0ae007, 0x2904e005, 0xd1024d08, 0x210a4648, 0x46285501, 0xbdf0b007, 0x00000074, 0x0000006c,
    0x00002db4, 0x402ec004, 0xf3100000, 0x000027e4, 0x00000068, 0x000033cc, 0x00005aa5, 0x47702000,
    0x8010f3ef, 0x0fc007c0, 0x4770b672, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000
    ],

    # Relative function addresses
    'pc_init': 0x20000f39,
    'pc_unInit': 0x200011c1,
    'pc_program_page': 0x20000fed,
    'pc_erase_sector': 0x20000341,
    'pc_eraseAll': 0x20000319,

    'static_base' : 0x20000000 + 0x00000004 + 0x000011cc,
    'begin_stack' : 0x20006250,
    'end_stack' : 0x20005250,
    'begin_data' : 0x20000000 + 0x1000,
    'page_size' : 0x2000,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    # Enable double buffering
    'page_buffers' : [
        0x20001250,
        0x20003250
    ],
    'min_program_length' : 0x2000,

    # Relative region addresses and sizes
    'ro_start': 0x4,
    'ro_size': 0x11cc,
    'rw_start': 0x11d0,
    'rw_size': 0x4,
    'zi_start': 0x11d4,
    'zi_size': 0x74,

    # Flash information
    'flash_start': 0x10000000,
    'flash_size': 0x20000,
    'sector_sizes': (
        (0x0, 0x2000),
    )
}

class S32K344(CoreSightTarget):

    VENDOR = "NXP"

    MEMORY_MAP = MemoryMap(
        FlashRegion(name="pflash",  start=0x00400000, end=0x7fffff, blocksize=0x2000, is_boot_memory=True, algo=FLASH_ALGO_CODE),
        FlashRegion(name="dflash",  start=0x10000000, end=0x1001ffff, blocksize=0x2000, algo=FLASH_ALGO_DATA),
        RamRegion(name="itcm",      start=0x00000000, length=0x10000), # 64 KB
        RamRegion(name="dtcm",      start=0x20000000, length=0x20000), # 128 KB
        RamRegion(name="sram",      start=0x20400000, length=0x50000), # 320 KB
        )

    def __init__(self, session):
        super(S32K344, self).__init__(session, self.MEMORY_MAP)

    def reset(self, reset_type=None):
        super(S32K344, self).reset(self.ResetType.SW_VECTRESET)

    def reset_and_halt(self, reset_type=None, map_to_user=True):
        super(S32K344, self).reset_and_halt(self.ResetType.SW_VECTRESET)

    def create_init_sequence(self):
        seq = super(S32K344, self).create_init_sequence()

        seq.wrap_task('discovery',
            lambda seq: seq
                # Normally the discovery sequence will scan for APs and then add those found
                # to a list. Unfortunately, the S32K344 freaks out when you scan for nonexistent
                # APs, so the list of APs are provided statically here.
                .replace_task('find_aps', self.create_s32k344_aps)

                # Debug needs to be enabled in the SDA_AP before pyOCD can probe for
                # components.
                .insert_before('find_components',
                    ('enable_debug', self.enable_s32k344_debug))
        )

        return seq

    def create_s32k344_aps(self):
        self.dp.valid_aps = AP_ID_LIST

    def enable_s32k344_debug(self):
        self.dp.aps[SDA_AP_ID].write_reg(SDA_AP_DBGENCTRL_ADDR, SDA_AP_EN_ALL)

